<?php
/**
 * Site Structure Generator
 * 
 * This script generates a comprehensive overview of the entire CodeIgniter 4 application structure
 * Place this file in your public folder and run via browser: yoursite.com/site_structure.php
 * 
 * SECURITY: Remove this file after use - it exposes your application structure
 */

// Prevent direct access in production
if ($_SERVER['HTTP_HOST'] !== 'localhost' && !isset($_GET['allow'])) {
    die('Access denied. Add ?allow=true for non-localhost access.');
}

// Check for developer mode
$devMode = isset($_GET['dev']) && $_GET['dev'] === 'true';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Site Structure Analysis</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        .section { margin-bottom: 30px; border: 1px solid #ddd; padding: 15px; border-radius: 5px; }
        .section h2 { margin-top: 0; color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }
        .file-tree { background: #f8f9fa; padding: 15px; border-radius: 4px; font-family: monospace; }
        .file { margin: 2px 0; }
        .folder { font-weight: bold; color: #007bff; }
        .php-file { color: #28a745; }
        .config-file { color: #ffc107; }
        .view-file { color: #6f42c1; }
        .sql-content { background: #e9ecef; padding: 10px; border-radius: 4px; margin: 5px 0; font-family: monospace; font-size: 12px; }
        .table-structure { border-collapse: collapse; width: 100%; margin: 10px 0; }
        .table-structure th, .table-structure td { border: 1px solid #ddd; padding: 8px; text-align: left; font-size: 12px; }
        .table-structure th { background: #f8f9fa; }
        .route-item { background: #e3f2fd; margin: 3px 0; padding: 5px; border-radius: 3px; font-family: monospace; }
        .warning { background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; border-radius: 4px; margin: 10px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🗗️ Complete Site Structure Analysis</h1>
        <div class="warning">
            <strong>⚠️ Security Notice:</strong> This file exposes your application structure. Remove it after analysis!
        </div>
        
        <?php if (!$devMode): ?>
        <div style="background: #e3f2fd; padding: 15px; border-radius: 4px; margin: 10px 0;">
            <strong>💡 Tip:</strong> Add <code>?dev=true</code> to URL for focused developer view (hides vendor files, fonts, etc.)
        </div>
        <?php else: ?>
        <div style="background: #f3e5f5; padding: 15px; border-radius: 4px; margin: 10px 0;">
            <strong>👨‍💻 Developer Mode:</strong> Showing only files relevant for core development work. 
            <a href="?">View full structure</a>
        </div>
        <?php endif; ?>

<?php

// Get the application root directory (go up one level from public)
$appRoot = dirname(__DIR__);
$publicRoot = __DIR__;

echo "<div class='section'>";
echo "<h2>📁 Application Directory Structure</h2>";
echo "<div class='file-tree'>";
scanDirectory($appRoot, 0, getDeveloperModeExclusions($devMode));
echo "</div>";
echo "</div>";

// Database Analysis
echo "<div class='section'>";
echo "<h2>🗄️ Database Structure</h2>";
analyzeDatabaseStructure();
echo "</div>";

// Controllers Analysis
echo "<div class='section'>";
echo "<h2>🎮 Controllers Analysis</h2>";
analyzeControllers($appRoot . '/app/Controllers');
echo "</div>";

// Models Analysis
echo "<div class='section'>";
echo "<h2>📊 Models Analysis</h2>";
analyzeModels($appRoot . '/app/Models');
echo "</div>";

// Views Analysis
echo "<div class='section'>";
echo "<h2>👁️ Views Structure</h2>";
analyzeViews($appRoot . '/app/Views');
echo "</div>";

// Routes Analysis
echo "<div class='section'>";
echo "<h2>🛣️ Routes Configuration</h2>";
analyzeRoutes($appRoot . '/app/Config/Routes.php');
echo "</div>";

// Config Analysis
echo "<div class='section'>";
echo "<h2>⚙️ Configuration Files</h2>";
analyzeConfig($appRoot . '/app/Config');
echo "</div>";

// Public Assets Analysis
echo "<div class='section'>";
echo "<h2>🎨 Public Assets</h2>";
analyzePublicAssets($publicRoot);
echo "</div>";

/**
 * Check if file has text after the extension (backup file)
 */
function isBackupFile($filename) {
    // List of valid file extensions that should be the final part of a filename
    $validExtensions = ['php', 'js', 'css', 'html', 'htm', 'json', 'xml', 'txt', 'md', 'sql', 'htaccess', 'scss'];
    
    // Check each valid extension
    foreach ($validExtensions as $ext) {
        // If filename ends with .[extension], it's a valid file
        if (preg_match('/\.' . preg_quote($ext, '/') . '$/', $filename)) {
            return false; // Not a backup file
        }
        
        // If filename contains .[extension] but has more text after it, it's a backup
        if (preg_match('/\.' . preg_quote($ext, '/') . './', $filename)) {
            return true; // It's a backup file
        }
    }
    
    // Also check for common backup patterns
    $backupPatterns = [
        '/\.php\d+$/',           // .php1, .php2, etc.
        '/\.css\d+$/',           // .css1, .css2, etc.
        '/\.js\d+$/',            // .js1, .js2, etc.
        '/-backup\d*\.php$/',    // -backup.php, -backup2.php
        '/-working\d*\.php$/',   // -working.php, -working9.php
        '/-bk\d*\.php$/',        // -bk.php, -bk1.php
        '/-old\.php$/',          // -old.php
        '/-orig\.php$/',         // -orig.php
        '/-copy\.php$/',         // -copy.php
        '/-gold/',               // contains -gold
        '/-before/',             // contains -before
        '/-last/',               // contains -last
        '/backup_/',             // contains backup_
    ];
    
    foreach ($backupPatterns as $pattern) {
        if (preg_match($pattern, $filename)) {
            return true;
        }
    }
    
    // If no valid extension found at all, keep the file (might be a valid file without extension)
    return false;
}

/**
 * Get exclusions based on developer mode
 */
function getDeveloperModeExclusions($devMode) {
    $baseExclusions = ['vendor', 'node_modules', '.git', 'writable/logs', 'writable/cache'];
    
    if (!$devMode) {
        return $baseExclusions;
    }
    
    // Developer mode - exclude much more
    return array_merge($baseExclusions, [
        // Third-party libraries and vendor files
        'public/assets/libs',
        'public/assets/fonts',
        'public/assets/scss/bootstrap',
        'app/ThirdParty',
        
        // System and framework files
        'tests',
        'writable/debugbar',
        'writable/session',
        
        // Build and config files
        'composer.lock',
        'builds',
        'LICENSE',
        'README.md',
        'phpunit.xml.dist',
        
        // Temp and backup directories
        'temp-codebase',
        
        // Log files and debugging
        '.log',
        'debug_log.txt',
        'project_log.txt',
        
        // Image directories (will be summarized anyway)
        'public/assets/images/categories',
        'public/assets/images/categories-orig',
        'public/assets/images/general-nature',
        'public/assets/images/lists',
        'public/assets/images/misc',
        'public/assets/images/splash',
        'public/assets/images/ui',
        'public/assets/images/colorbox',
        'public/assets/images/detector',
        'public/assets/images/gallery',
        
        // Font directories
        'public/assets/fonts/flaticon',
        'public/assets/fonts/icomoon',
        'public/assets/fonts/less',
        'public/assets/fonts/scss',
        'public/assets/css/fonts'
    ]);
}

/**
 * Check if directory should be summarized in dev mode
 */
function shouldSummarizeDirectory($dir, $devMode) {
    if (!$devMode) return false;
    
    $summarizePaths = [
        '/assets/css',
        '/assets/js', 
        '/public',
        '/import'
    ];
    
    foreach ($summarizePaths as $path) {
        if (strpos($dir, $path) !== false) {
            return true;
        }
    }
    return false;
}

/**
 * Check if file extension is an image type
 */
function isImageFile($extension) {
    $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'bmp', 'webp', 'ico'];
    return in_array(strtolower($extension), $imageExtensions);
}

/**
 * Recursively scan directory structure
 */
function scanDirectory($dir, $level = 0, $exclude = []) {
    if (!is_dir($dir)) return;
    
    $items = scandir($dir);
    $imageCounts = [];
    $nonImageFiles = [];
    $folders = [];
    
    // First pass: separate files and count images
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $fullPath = $dir . '/' . $item;
        $relativePath = str_replace(dirname(__DIR__), '', $fullPath);
        
        // Check if should be excluded
        $shouldExclude = false;
        foreach ($exclude as $excludePattern) {
            if (strpos($relativePath, $excludePattern) !== false) {
                $shouldExclude = true;
                break;
            }
        }
        if ($shouldExclude) continue;
        
        // Skip backup files (files with text after extension)
        if (!is_dir($fullPath) && isBackupFile($item)) {
            continue;
        }
        
        if (is_dir($fullPath)) {
            $folders[] = $item;
        } else {
            $extension = strtolower(pathinfo($item, PATHINFO_EXTENSION));
            
            if (isImageFile($extension)) {
                // Count image files by extension
                if (!isset($imageCounts[$extension])) {
                    $imageCounts[$extension] = 0;
                }
                $imageCounts[$extension]++;
            } else {
                // Store non-image files for individual listing
                $nonImageFiles[] = $item;
            }
        }
    }
    
    $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $level);
    
    // Display folders
    foreach ($folders as $folder) {
        echo "<div class='file folder'>{$indent}📁 {$folder}/</div>";
        if ($level < 4) { // Limit depth to prevent too much output
            scanDirectory($dir . '/' . $folder, $level + 1, $exclude);
        }
    }
    
    // Display non-image files individually
    foreach ($nonImageFiles as $file) {
        $extension = pathinfo($file, PATHINFO_EXTENSION);
        $icon = getFileIcon($extension);
        $class = getFileClass($extension);
        echo "<div class='file {$class}'>{$indent}{$icon} {$file}</div>";
    }
    
    // Display image file counts
    if (!empty($imageCounts)) {
        foreach ($imageCounts as $ext => $count) {
            $icon = getFileIcon($ext);
            echo "<div class='file'>{$indent}{$icon} {$count} {$ext} files</div>";
        }
    }
}

/**
 * Get file icon based on extension
 */
function getFileIcon($extension) {
    $icons = [
        'php' => '🐘',
        'js' => '📄',
        'css' => '🎨',
        'html' => '🌐',
        'sql' => '🗄️',
        'json' => '📋',
        'xml' => '📄',
        'txt' => '📝',
        'md' => '📖',
        'jpg' => '🖼️',
        'png' => '🖼️',
        'gif' => '🖼️',
        'svg' => '🖼️'
    ];
    return $icons[$extension] ?? '📄';
}

/**
 * Get CSS class for file type
 */
function getFileClass($extension) {
    $classes = [
        'php' => 'php-file',
        'html' => 'view-file',
        'css' => 'config-file',
        'js' => 'config-file',
        'json' => 'config-file',
        'xml' => 'config-file'
    ];
    return $classes[$extension] ?? '';
}

/**
 * Analyze database structure
 */
function analyzeDatabaseStructure() {
    try {
        // Try to connect to database using CodeIgniter config
        $configPath = dirname(__DIR__) . '/app/Config/Database.php';
        if (!file_exists($configPath)) {
            echo "<p>❌ Database config file not found</p>";
            return;
        }
        
        // Include the config file
        require_once($configPath);
        
        // Get database configuration
        $config = new Config\Database();
        $dbConfig = $config->default;
        
        $dsn = "mysql:host={$dbConfig['hostname']};dbname={$dbConfig['database']}";
        $pdo = new PDO($dsn, $dbConfig['username'], $dbConfig['password']);
        
        // Get all tables
        $stmt = $pdo->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        echo "<p><strong>📊 Database:</strong> {$dbConfig['database']} (" . count($tables) . " tables)</p>";
        
        foreach ($tables as $table) {
            echo "<h4>📋 Table: {$table}</h4>";
            
            // Get table structure
            $stmt = $pdo->query("DESCRIBE {$table}");
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo "<table class='table-structure'>";
            echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
            foreach ($columns as $column) {
                echo "<tr>";
                echo "<td>{$column['Field']}</td>";
                echo "<td>{$column['Type']}</td>";
                echo "<td>{$column['Null']}</td>";
                echo "<td>{$column['Key']}</td>";
                echo "<td>" . ($column['Default'] ?? 'NULL') . "</td>";
                echo "<td>{$column['Extra']}</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            // Get row count
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM {$table}");
            $count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
            echo "<p><em>📊 Rows: {$count}</em></p>";
        }
        
    } catch (Exception $e) {
        echo "<p>❌ Database connection failed: " . $e->getMessage() . "</p>";
        echo "<p>💡 To analyze database structure, ensure your database config is correct and accessible.</p>";
    }
}

/**
 * Analyze controllers
 */
function analyzeControllers($controllersDir) {
    if (!is_dir($controllersDir)) {
        echo "<p>Controllers directory not found</p>";
        return;
    }
    
    analyzePhpFiles($controllersDir, 'Controller');
}

/**
 * Analyze models
 */
function analyzeModels($modelsDir) {
    if (!is_dir($modelsDir)) {
        echo "<p>Models directory not found</p>";
        return;
    }
    
    analyzePhpFiles($modelsDir, 'Model');
}

/**
 * Analyze PHP files for classes and methods
 */
function analyzePhpFiles($dir, $type) {
    $files = glob($dir . '/*.php');
    $files = array_merge($files, glob($dir . '/*/*.php')); // Include subdirectories
    $files = array_merge($files, glob($dir . '/*/*/*.php')); // Include deeper subdirectories
    
    // Filter out backup files
    $files = array_filter($files, function($file) {
        return !isBackupFile(basename($file));
    });
    
    foreach ($files as $file) {
        $relativePath = str_replace(dirname(__DIR__) . '/', '', $file);
        echo "<h4>📄 {$relativePath}</h4>";
        
        $content = file_get_contents($file);
        
        // Extract class name
        if (preg_match('/class\s+(\w+)/', $content, $matches)) {
            $className = $matches[1];
            echo "<p><strong>Class:</strong> {$className}</p>";
        }
        
        // Extract methods
        preg_match_all('/(?:public|private|protected)\s+function\s+(\w+)\s*\(([^)]*)\)/', $content, $matches);
        if (!empty($matches[1])) {
            echo "<p><strong>Methods:</strong></p>";
            echo "<ul>";
            for ($i = 0; $i < count($matches[1]); $i++) {
                $method = $matches[1][$i];
                $params = trim($matches[2][$i]);
                echo "<li><code>{$method}(" . ($params ?: '') . ")</code></li>";
            }
            echo "</ul>";
        }
        
        // Show file size and last modified
        $size = filesize($file);
        $modified = date('Y-m-d H:i:s', filemtime($file));
        echo "<p><em>📏 Size: {$size} bytes | 📅 Modified: {$modified}</em></p>";
    }
}

/**
 * Analyze views structure
 */
function analyzeViews($viewsDir) {
    if (!is_dir($viewsDir)) {
        echo "<p>Views directory not found</p>";
        return;
    }
    
    echo "<div class='file-tree'>";
    scanDirectory($viewsDir, 0, []);
    echo "</div>";
}

/**
 * Analyze routes configuration
 */
function analyzeRoutes($routesFile) {
    if (!file_exists($routesFile)) {
        echo "<p>Routes file not found</p>";
        return;
    }
    
    $content = file_get_contents($routesFile);
    
    // Extract route definitions
    preg_match_all('/\$routes->(get|post|put|delete|patch|options|head)\s*\(\s*[\'"]([^\'"]+)[\'"]\s*,\s*[\'"]([^\'"]+)[\'"]/', $content, $matches);
    
    if (!empty($matches[0])) {
        echo "<p><strong>🔍 Defined Routes:</strong></p>";
        for ($i = 0; $i < count($matches[0]); $i++) {
            $method = strtoupper($matches[1][$i]);
            $route = $matches[2][$i];
            $handler = $matches[3][$i];
            
            echo "<div class='route-item'>";
            echo "<strong>{$method}</strong> /{$route} → {$handler}";
            echo "</div>";
        }
    }
    
    // Show file content (truncated)
    echo "<h4>📄 Routes File Content (first 2000 chars):</h4>";
    echo "<div class='sql-content'>" . htmlspecialchars(substr($content, 0, 2000)) . "...</div>";
}

/**
 * Analyze configuration files
 */
function analyzeConfig($configDir) {
    if (!is_dir($configDir)) {
        echo "<p>Config directory not found</p>";
        return;
    }
    
    $files = glob($configDir . '/*.php');
    
    // Filter out backup files
    $files = array_filter($files, function($file) {
        return !isBackupFile(basename($file));
    });
    
    foreach ($files as $file) {
        $fileName = basename($file);
        echo "<h4>⚙️ {$fileName}</h4>";
        
        $size = filesize($file);
        $modified = date('Y-m-d H:i:s', filemtime($file));
        echo "<p><em>📏 Size: {$size} bytes | 📅 Modified: {$modified}</em></p>";
    }
}

/**
 * Analyze public assets
 */
function analyzePublicAssets($publicDir) {
    if (!is_dir($publicDir)) {
        echo "<p>Public directory not found</p>";
        return;
    }
    
    echo "<div class='file-tree'>";
    scanDirectory($publicDir, 0, [basename(__FILE__)]);
    echo "</div>";
}

?>

        <div class="section">
            <h2>🔧 Analysis Complete</h2>
            <p><strong>Generated:</strong> <?= date('Y-m-d H:i:s') ?></p>
            <p><strong>Server:</strong> <?= $_SERVER['HTTP_HOST'] ?></p>
            <p><strong>PHP Version:</strong> <?= PHP_VERSION ?></p>
            <div class="warning">
                <strong>🔒 Security Reminder:</strong> Delete this file (site_structure.php) after analysis to prevent unauthorized access to your application structure.
            </div>
        </div>
    </div>
</body>
</html>