<?php
/**
 * Database Connection Handler
 * /public/import/classes/DatabaseConnection.php
 */

if (!defined('IMPORT_ACCESS')) {
    die('Direct access not allowed');
}

class DatabaseConnection {
    private $pdo;
    private $timezoneCache = [];
    private $logger;
    
    public function __construct($logger = null) {
        $this->logger = $logger;
        $this->connect();
        $this->loadTimezoneCache();
    }
    
    private function connect() {
        try {
            $dsn = sprintf(
                "mysql:host=%s;dbname=%s;charset=%s",
                DB_CONFIG['host'],
                DB_CONFIG['database'],
                DB_CONFIG['charset']
            );
            
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ];
            
            $this->pdo = new PDO($dsn, DB_CONFIG['username'], DB_CONFIG['password'], $options);
            
            if ($this->logger) {
                $this->logger->log("Database connection established to " . DB_CONFIG['database']);
            }
            
        } catch (PDOException $e) {
            $error = "Database connection failed: " . $e->getMessage();
            if ($this->logger) {
                $this->logger->log($error);
            }
            die($error);
        }
    }
    
    private function loadTimezoneCache() {
        try {
            $stmt = $this->pdo->query("SELECT id, name FROM timezone ORDER BY id");
            
            while ($row = $stmt->fetch()) {
                $this->timezoneCache[$row['name']] = $row['id'];
            }
            
            if ($this->logger) {
                $this->logger->log("Loaded " . count($this->timezoneCache) . " timezones from database");
            }
            
        } catch (PDOException $e) {
            if ($this->logger) {
                $this->logger->log("Failed to load timezone cache: " . $e->getMessage());
            }
        }
    }
    
    public function getPdo() {
        return $this->pdo;
    }
    
    public function getTimezoneId($timezoneName) {
        if (empty($timezoneName)) {
            return IMPORT_SETTINGS['default_timezone_id'];
        }
        
        if (isset($this->timezoneCache[$timezoneName])) {
            return $this->timezoneCache[$timezoneName];
        }
        
        $lowerName = strtolower($timezoneName);
        foreach ($this->timezoneCache as $dbTimezone => $id) {
            if (strtolower($dbTimezone) === $lowerName) {
                return $id;
            }
        }
        
        if ($this->logger) {
            $this->logger->logTimezoneFailure($timezoneName);
        }
        
        return IMPORT_SETTINGS['default_timezone_id'];
    }
    
    public function storeExists($storeId) {
    try {
        $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM store WHERE store_id = ?");
        $stmt->execute([$storeId]);
        return $stmt->fetchColumn() > 0;
    } catch (PDOException $e) {
        if ($this->logger) {
            $this->logger->log("Error checking store existence: " . $e->getMessage());
        }
        return false;
    }
}
    
    public function insertStore($storeData) {
        try {
            $storeData['create_date'] = date('Y-m-d H:i:s');
            $storeData['update_date'] = date('Y-m-d H:i:s');
            
            $columns = implode(', ', array_keys($storeData));
            $placeholders = ':' . implode(', :', array_keys($storeData));
            
            $sql = "INSERT INTO store ({$columns}) VALUES ({$placeholders})";
            $stmt = $this->pdo->prepare($sql);
            
            return $stmt->execute($storeData);
            
        } catch (PDOException $e) {
            if ($this->logger) {
                $this->logger->log("Database insert error: " . $e->getMessage());
            }
            throw $e;
        }
    }
    
    public function assignCategoryToStore($storeId, $categoryId) {
        try {
            // Check if assignment already exists
            $stmt = $this->pdo->prepare("SELECT id FROM store_category WHERE store_id = ? AND category_id = ?");
            $stmt->execute([$storeId, $categoryId]);
            
            if (!$stmt->fetchColumn()) {
                // Insert new assignment
                $stmt = $this->pdo->prepare("INSERT INTO store_category (store_id, category_id) VALUES (?, ?)");
                $result = $stmt->execute([$storeId, $categoryId]);
                
                if ($this->logger) {
                    $this->logger->log("Assigned category {$categoryId} to store {$storeId}");
                }
                
                return $result;
            }
            
            return true; // Already exists
            
        } catch (PDOException $e) {
            if ($this->logger) {
                $this->logger->log("Failed to assign category to store {$storeId}: " . $e->getMessage());
            }
            throw $e;
        }
    }
    
    public function createHourType($storeId) {
        try {
            $sql = "INSERT INTO store_hour_types 
                    (store_id, global_type_id, custom_name, display_order, affects_main_status, is_active, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([
                $storeId, 1, 'Regular Hours', 1, 1, 1
            ]);
            
            return $this->pdo->lastInsertId();
            
        } catch (PDOException $e) {
            if ($this->logger) {
                $this->logger->log("Failed to create hour type for store {$storeId}: " . $e->getMessage());
            }
            throw $e;
        }
    }
    
    public function insertSchedule($storeId, $hourTypeId, $dayOfWeek, $schedule) {
        try {
            $sql = "INSERT INTO store_schedules 
                    (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, slot_2_open, slot_2_close, is_closed, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
            
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute([
                $storeId, $hourTypeId, $dayOfWeek,
                $schedule['slot_1_open'], $schedule['slot_1_close'],
                $schedule['slot_2_open'], $schedule['slot_2_close'],
                $schedule['is_closed']
            ]);
            
        } catch (PDOException $e) {
            if ($this->logger) {
                $this->logger->log("Failed to insert schedule for store {$storeId}, day {$dayOfWeek}: " . $e->getMessage());
            }
            throw $e;
        }
    }
    
    public function beginTransaction() { return $this->pdo->beginTransaction(); }
    public function commit() { return $this->pdo->commit(); }
    public function rollback() { return $this->pdo->rollback(); }
    
    public function validateDatabase() {
        $requiredTables = ['store', 'store_category', 'store_hour_types', 'store_schedules', 'timezone'];
        $errors = [];
        
        foreach ($requiredTables as $table) {
            try {
                $stmt = $this->pdo->query("SELECT 1 FROM {$table} LIMIT 1");
            } catch (PDOException $e) {
                $errors[] = "Table '{$table}' not accessible: " . $e->getMessage();
            }
        }
        
        if (!empty($errors)) {
            if ($this->logger) {
                foreach ($errors as $error) {
                    $this->logger->log($error);
                }
            }
            return false;
        }
        
        return true;
    }
    
    public function getTimezoneList() {
        return $this->timezoneCache;
    }
}
?>