// UNIFIED MAIN SCRIPT - COORDINATES ALL COMPONENTS
// This file ensures all variables and functions are globally available

// ===========================================
// GLOBAL VARIABLES - SHARED ACROSS ALL FILES
// ===========================================
let isSearching = false;
let availableCategories = [];
let selectedCategoryId = '';
let currentSearchData = {};
let isInitialLoad = true;
let currentPage = 1;
let isLoading = false;
let hasPerformedSearch = false;
let hasMoreResults = true;
let scrollHandler = null;
let totalCounts = { total_all: 0, total_open: 0 };
let isCategoryBrowserVisible = false;

// URL constants from PHP - use the passed configuration
const ajaxLocatorUrl = window.siteConfig?.ajaxLocatorUrl || '/ajax/locator';
const storeDetailsUrl = window.siteConfig?.storeDetailsUrl || '/ajax/store-details';
const categoriesApiUrl = window.siteConfig?.categoriesApiUrl || '/api/categories';

// ===========================================
// INITIALIZATION - ENTRY POINT
// ===========================================
document.addEventListener('DOMContentLoaded', function() {
    try {
        console.log('DOM loaded - initializing enhanced category system...');
        
        // Force toggle to show immediately
        const filterToggle = document.getElementById('filterToggleContainer');
        if (filterToggle) {
            filterToggle.style.display = 'block';
            filterToggle.style.visibility = 'visible';
            filterToggle.style.opacity = '1';
            console.log('Toggle forced visible on DOM load');
        }
        
        // Initialize all components
        loadCategories();
        setupToggleListeners();
        setupSearchListeners();
        setupZipCodeHandlers();
        
        // Make search functions globally available for hero section integration
        window.geocodeZipAndSearch = geocodeZipAndSearch;
        window.geocodeAddressAndSearch = geocodeAddressAndSearch;
        window.getLocation = getLocation;
        window.updateLocationStatus = updateLocationStatus;
        window.loadStoreModal = loadStoreModal;
        
        // Initialize page load
        if (isInitialLoad) {
            initializeWithLocation();
        }
        
    } catch (error) {
        console.error('Error in DOM ready:', error);
        setTimeout(() => loadAllBusinesses(false), 1000);
    }
});

// ===========================================
// ZIP CODE HANDLERS
// ===========================================
function setupZipCodeHandlers() {
    const zipDigits = document.querySelectorAll('.zip-digit');
    const clearZipBtn = document.getElementById('clear-zip-btn');
    const zipSearchBtn = document.getElementById('zip-search-btn');
    
    if (!zipDigits.length) return;
    
    zipDigits.forEach((digit, index) => {
        digit.addEventListener('input', function(e) {
            const value = e.target.value;
            
            if (!/^\d$/.test(value) && value !== '') {
                e.target.value = '';
                e.target.classList.add('error');
                setTimeout(() => e.target.classList.remove('error'), 300);
                return;
            }
            
            if (value) {
                e.target.classList.add('filled');
                if (index < zipDigits.length - 1) {
                    zipDigits[index + 1].focus();
                }
            } else {
                e.target.classList.remove('filled');
            }
            
            updateZipState();
        });
        
        digit.addEventListener('keydown', function(e) {
            if (e.key === 'Backspace' && !e.target.value && index > 0) {
                zipDigits[index - 1].focus();
                zipDigits[index - 1].value = '';
                zipDigits[index - 1].classList.remove('filled');
                updateZipState();
            }
            // REMOVED: Auto-submit on Enter - let form handler do it
        });
        
        digit.addEventListener('paste', function(e) {
            e.preventDefault();
            const pastedData = e.clipboardData.getData('text').replace(/\D/g, '');
            
            if (pastedData.length === 5) {
                zipDigits.forEach((input, i) => {
                    input.value = pastedData[i] || '';
                    if (pastedData[i]) {
                        input.classList.add('filled');
                    } else {
                        input.classList.remove('filled');
                    }
                });
                updateZipState();
                // REMOVED: Auto-submit on paste - let form handler do it
            }
        });
    });
    
    if (clearZipBtn) {
        clearZipBtn.addEventListener('click', function() {
            zipDigits.forEach(digit => {
                digit.value = '';
                digit.classList.remove('filled');
            });
            zipDigits[0].focus();
            updateZipState();
        });
    }
    
    function updateZipState() {
        const zipValue = Array.from(zipDigits).map(d => d.value).join('');
        const zipField = document.getElementById('form_zipcode');
        if (zipField) zipField.value = zipValue;
        
        if (clearZipBtn) {
            if (zipValue.length > 0) {
                clearZipBtn.style.display = 'inline-flex';
            } else {
                clearZipBtn.style.display = 'none';
            }
        }
        
        if (zipSearchBtn) {
            if (zipValue.length === 5) {
                zipSearchBtn.classList.add('ready');
            } else {
                zipSearchBtn.classList.remove('ready');
            }
        }
        
        // REMOVED: Auto-submit when 5 digits entered - causes duplicate
    }
    
    if (zipDigits[0]) {
        zipDigits[0].focus();
    }
}

// ===========================================
// LOCATION & INITIALIZATION FUNCTIONS
// ===========================================
function initializeWithLocation() {
    try {
        console.log('Initializing with location...');
        updateLocationStatus('Enter your ZIP code or click crosshairs for nearby businesses', 'default');
        
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    try {
                        console.log('Got location:', position.coords.latitude, position.coords.longitude);
                        document.querySelector('#form_latitude').value = position.coords.latitude;
                        document.querySelector('#form_longitude').value = position.coords.longitude;
                        fetchHumanLocation(position.coords.latitude, position.coords.longitude);
                        loadAllBusinesses(false); // Only load here if geolocation succeeds
                    } catch (error) {
                        console.error('Error processing location:', error);
                        loadAllBusinesses(false);
                    }
                },
                function(error) {
                    try {
                        console.log('Geolocation error:', error);
                        updateLocationStatus('Enter your ZIP code to find businesses in your area', 'warning');
                        loadAllBusinesses(false); // Only load here if geolocation fails
                    } catch (e) {
                        console.error('Error handling location error:', e);
                        loadAllBusinesses(false);
                    }
                },
                { timeout: 5000, enableHighAccuracy: false, maximumAge: 300000 }
            );
        } else {
            console.log('Geolocation not supported');
            updateLocationStatus('Enter your ZIP code to find businesses', 'default');
            loadAllBusinesses(false); // Only load here if geolocation not supported
        }
        // REMOVED: loadAllBusinesses(false) was here causing double load
    } catch (error) {
        console.error('Error in initializeWithLocation:', error);
        loadAllBusinesses(false);
    }
}

function fetchHumanLocation(lat, lon) {
    fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lon}`)
        .then(res => res.json())
        .then(data => {
            if (data && data.address) {
                const city = data.address.road || data.address.neighbourhood || data.address.suburb || 
                            data.address.city || data.address.town || data.address.village || '';
                const state = data.address.state || '';
                const locationName = `${city}${state ? ', ' + state : ''}`;
                
                if (locationName.trim()) {
                    updateLocationStatus(`Your device locates at ${locationName}`, 'success');
                } else {
                    updateLocationStatus('Showing businesses near you', 'success');
                }
            } else {
                updateLocationStatus('Showing businesses near you', 'success');
            }
        })
        .catch(error => {
            console.log('Location name lookup failed, using generic message');
            updateLocationStatus('Showing businesses near you', 'success');
        });
}

function updateLocationStatus(message, type = 'default') {
    const statusEl = document.getElementById('location-status');
    const textEl = document.getElementById('location-status-text');
    
    if (textEl) textEl.textContent = message;
    if (statusEl) {
        statusEl.classList.remove('success', 'warning', 'error');
        if (type !== 'default') statusEl.classList.add(type);
    }
}

// ===========================================
// SEARCH HANDLING FUNCTIONS
// ===========================================
function setupSearchListeners() {
    const heroForm = document.getElementById('hero-search-form');
    if (heroForm) {
        heroForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const searchMode = document.getElementById('search_mode')?.value;
            if (searchMode === 'zip') {
                handleZipSearch();
            } else if (searchMode === 'address') {
                handleAddressSearch();
            }
        });
    }

    const stickyForm = document.getElementById('sticky-search-form');
    if (stickyForm) {
        stickyForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleKeywordSearch();
        });
    }

    const crosshairsBtn = document.getElementById('crosshairs-btn');
    if (crosshairsBtn) {
        crosshairsBtn.addEventListener('click', function(e) {
            e.preventDefault();
            getLocation();
        });
    }
    
    const stickyCrosshairs = document.querySelector('.sticky-crosshairs');
    if (stickyCrosshairs) {
        stickyCrosshairs.addEventListener('click', function(e) {
            e.preventDefault();
            getLocation();
        });
    }

    const stickyInput = document.getElementById('sticky_search_input');
    const stickyClearBtn = document.getElementById('sticky-clear-btn');
    
    if (stickyInput && stickyClearBtn) {
        stickyClearBtn.addEventListener('click', function() {
            stickyInput.value = '';
            this.style.display = 'none';
            resetToHome();
        });
        
        stickyInput.addEventListener('input', function() {
            if (this.value.trim()) {
                stickyClearBtn.style.display = 'flex';
            } else {
                stickyClearBtn.style.display = 'none';
            }
        });
    }
}

function handleZipSearch() {
    const zipCode = document.getElementById('form_zipcode')?.value || '';
    console.log('=== ZIP CODE SEARCH SUBMITTED ===', zipCode);
    
    if (zipCode.length !== 5) {
        alert('Please enter a complete 5-digit ZIP code.');
        const firstEmptyDigit = document.querySelector('.zip-digit:not(.filled)');
        if (firstEmptyDigit) firstEmptyDigit.focus();
        return;
    }
    
    showClearSearchBtn();
    geocodeZipAndSearch(zipCode);
}

function handleAddressSearch() {
    const address = document.getElementById('address_search_input')?.value.trim() || '';
    console.log('=== ADDRESS SEARCH SUBMITTED ===', address);
    
    if (!address) {
        alert('Please enter an address to search.');
        document.getElementById('address_search_input')?.focus();
        return;
    }
    
    showClearSearchBtn();
    geocodeAddressAndSearch(address);
}

function handleKeywordSearch() {
    const keyword = document.getElementById('sticky_search_input')?.value.trim() || '';
    console.log('=== KEYWORD SEARCH SUBMITTED ===', keyword);
    
    // TEMPORARILY DISABLED FOR DEBUGGING - VALIDATION REMOVED
    // if (!keyword) {
    //     alert('Please enter a business name or keyword to search.');
    //     return;
    // }
    
    showClearSearchBtn();
    performKeywordSearch(keyword);
}

function geocodeZipAndSearch(zipCode) {
    updateLocationStatus('Finding businesses near ZIP ' + zipCode + '...', 'default');
    
    const query = `${zipCode}, United States`;
    
    fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(query)}&limit=1&countrycodes=us`)
        .then(response => response.json())
        .then(data => {
            if (data && data.length > 0) {
                const location = data[0];
                document.getElementById('form_latitude').value = location.lat;
                document.getElementById('form_longitude').value = location.lon;
                
                const parts = location.display_name.split(',');
                const cityState = parts.length >= 3 ? `${parts[0]}, ${parts[1]}` : `ZIP ${zipCode}`;
                updateLocationStatus(`Searching near ${cityState}`, 'success');
                
                hasPerformedSearch = true;
                performLocationBasedSearch();
            } else {
                updateLocationStatus('ZIP code not found. Please check and try again.', 'error');
                clearZipCode();
            }
        })
        .catch(error => {
            console.error('ZIP geocoding error:', error);
            updateLocationStatus('Unable to find ZIP code. Please try again.', 'error');
        });
}

function geocodeAddressAndSearch(address) {
    updateLocationStatus('Finding businesses near ' + address + '...', 'default');
    
    const query = encodeURIComponent(address);
    
    fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${query}&limit=1&countrycodes=us`)
        .then(response => response.json())
        .then(data => {
            if (data && data.length > 0) {
                const location = data[0];
                document.getElementById('form_latitude').value = location.lat;
                document.getElementById('form_longitude').value = location.lon;
                
                const displayName = location.display_name;
                const parts = displayName.split(',');
                const locationName = parts.length >= 2 ? `${parts[0]}, ${parts[1]}` : address;
                updateLocationStatus(`Searching near ${locationName}`, 'success');
                
                hasPerformedSearch = true;
                performLocationBasedSearch();
            } else {
                updateLocationStatus('Address not found. Please check and try again.', 'error');
                const addressInput = document.getElementById('address_search_input');
                if (addressInput) {
                    addressInput.value = '';
                    addressInput.focus();
                }
            }
        })
        .catch(error => {
            console.error('Address geocoding error:', error);
            updateLocationStatus('Unable to find address. Please try again.', 'error');
        });
}

function performKeywordSearch(keyword) {
    updateLocationStatus(`Searching for "${keyword}"...`, 'default');
    
    document.getElementById('form_keyword').value = keyword;
    
    hasPerformedSearch = true;
    
    // Use the same path as hero searches
    if (document.getElementById('form_latitude').value && document.getElementById('form_longitude').value) {
        performLocationBasedSearch(); // Use location-based path like hero
    } else {
        performSearch(); // Fallback to regular search
    }
}

function performLocationBasedSearch() {
    if (isSearching) return;
    isSearching = true;
    
    document.getElementById('form_keyword').value = '';
    performSearch();
    
    setTimeout(() => { isSearching = false; }, 2000);
}

function getLocation() {
    updateLocationStatus('Getting your location...', 'default');
    
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(function(position) {
            document.querySelector('#form_latitude').value = position.coords.latitude;
            document.querySelector('#form_longitude').value = position.coords.longitude;
            fetchHumanLocation(position.coords.latitude, position.coords.longitude);
            loadCategories();
            
            const stickyInput = document.getElementById('sticky_search_input');
            if (stickyInput) stickyInput.value = '';
            const keywordField = document.getElementById('form_keyword');
            if (keywordField) keywordField.value = '';
            
            clearZipCode();
            
            const addressInput = document.getElementById('address_search_input');
            if (addressInput) addressInput.value = '';
            
            const isOpenOnly = document.getElementById('openOnlyToggle')?.checked || false;
            hasPerformedSearch = true;
            showClearSearchBtn();
            
            if (hasPerformedSearch || selectedCategoryId) {
                performLocationBasedSearch();
            } else {
                loadAllBusinesses(isOpenOnly);
            }
        }, function(error) {
            console.error('Geolocation error:', error);
            updateLocationStatus('Unable to get location', 'error');
            alert("Unable to retrieve your location. Please enter a ZIP code or address manually.");
        }, { timeout: 8000, enableHighAccuracy: false });
    } else {
        alert("Geolocation is not supported by this browser.");
    }
}

function clearZipCode() {
    const zipDigits = document.querySelectorAll('.zip-digit');
    zipDigits.forEach(digit => {
        digit.value = '';
        digit.classList.remove('filled');
    });
    const clearZipBtn = document.getElementById('clear-zip-btn');
    if (clearZipBtn) clearZipBtn.style.display = 'none';
    if (zipDigits[0]) zipDigits[0].focus();
}

function showClearSearchBtn() {
    const clearBtn = document.getElementById('clearSearchBtn');
    if (clearBtn) clearBtn.style.display = 'inline-flex';
}

// ===========================================
// TOGGLE & CATEGORY FUNCTIONS
// ===========================================
function setupToggleListeners() {
    try {
        const openToggle = document.getElementById('openOnlyToggle');
        if (openToggle) {
            openToggle.addEventListener('change', function() {
                const isOpenOnly = this.checked;
                updateFilterInfo(isOpenOnly);
                console.log('=== OPEN TOGGLE CHANGED ===', isOpenOnly);
                
                loadCategories();
                currentPage = 1;
                hasMoreResults = true;
                
                if (hasPerformedSearch) {
                    performSearch(true);
                } else {
                    loadAllBusinesses(isOpenOnly);
                }
            });
        }
    } catch (error) {
        console.error('Error setting up open toggle:', error);
    }
    
    try {
        const categoryToggle = document.getElementById('categoryToggle');
        if (categoryToggle) {
            categoryToggle.addEventListener('change', function() {
                const showCategories = this.checked;
                console.log('=== CATEGORY TOGGLE CHANGED ===', showCategories);
                
                if (showCategories) {
                    showCategoryBrowser();
                } else {
                    hideCategoryBrowser();
                }
            });
        }
    } catch (error) {
        console.error('Error setting up category toggle:', error);
    }
}

function loadCategories() {
    const isOpenOnly = document.getElementById('openOnlyToggle')?.checked || false;
    const hasCoords = document.getElementById('form_latitude')?.value && document.getElementById('form_longitude')?.value;
    
    let apiUrl = categoriesApiUrl;
    const params = [];
    
    if (isOpenOnly) params.push('open_only=true');
    if (hasCoords) {
        const lat = document.getElementById('form_latitude').value;
        const lng = document.getElementById('form_longitude').value;
        params.push(`coords=${lat},${lng}`);
    }
    
    if (params.length > 0) apiUrl += '?' + params.join('&');
    
    console.log('Loading categories:', apiUrl);
    
    fetch(apiUrl)
        .then(response => response.json())
        .then(data => {
            if (data.categories) {
                availableCategories = data.categories;
                populateCategoryGrid();
                console.log(`Loaded ${availableCategories.length} categories (Open Only: ${isOpenOnly})`);
            }
        })
        .catch(error => {
            console.error('Error loading categories:', error);
            const categoryGrid = document.getElementById('categoryGrid');
            if (categoryGrid) {
                categoryGrid.innerHTML = `
                    <div class="category-loading">
                        <p class="text-muted">Categories temporarily unavailable</p>
                    </div>
                `;
            }
        });
}

function populateCategoryGrid() {
    const gridContainer = document.getElementById('categoryGrid');
    const isOpenOnly = document.getElementById('openOnlyToggle')?.checked || false;
    
    if (!gridContainer) return;
    
    if (availableCategories.length === 0) {
        const message = isOpenOnly ? 'No categories have open businesses right now' : 'No categories available';
        gridContainer.innerHTML = `<div class="category-loading"><p class="text-muted">${message}</p></div>`;
        return;
    }
    
    let gridHTML = '';
    availableCategories.forEach(category => {
        const isAvailable = category.is_available;
        const featuredClass = category.top_category == 1 ? 'featured' : '';
        const availabilityClass = isAvailable ? '' : 'unavailable';
        
        gridHTML += `
            <div class="category-item ${featuredClass} ${availabilityClass}" 
                 onclick="selectCategory('${category.category_id}', '${category.title}')" 
                 data-category-name="${category.title.toLowerCase()}"
                 data-available="${isAvailable}">
                <i class="${category.icon}"></i>
                <h5>${category.title}</h5>
                <div class="store-count ${isAvailable ? '' : 'no-results'}">
                    ${category.status_text}
                </div>
            </div>
        `;
    });
    
    gridContainer.innerHTML = gridHTML;
    console.log(`Category grid updated: ${availableCategories.length} categories`);
}

function selectCategory(categoryId, categoryTitle) {
    console.log('=== CATEGORY SELECTED ===', categoryId, categoryTitle);
    
    if (categoryId) {
        const categoryData = availableCategories.find(cat => cat.category_id === categoryId);
        if (categoryData && !categoryData.is_available) {
            const isOpenOnly = document.getElementById('openOnlyToggle')?.checked || false;
            const message = isOpenOnly ? 
                `No businesses in "${categoryTitle}" are currently open. Try turning off "Show Open Only" or selecting a different category.` :
                `No businesses found in "${categoryTitle}".`;
            alert(message);
            return;
        }
    }
    
    selectedCategoryId = categoryId;
    const selectedCategoryField = document.getElementById('selected_category');
    if (selectedCategoryField) selectedCategoryField.value = categoryId;
    
    const selectedCategoryText = document.getElementById('selectedCategoryText');
    if (selectedCategoryText) selectedCategoryText.textContent = categoryTitle;
    
    const categoryToggle = document.getElementById('categoryToggle');
    if (categoryToggle) categoryToggle.checked = false;
    
    hideCategoryBrowser();
    
    const stickyInput = document.getElementById('sticky_search_input');
    if (stickyInput) stickyInput.value = '';
    
    const keywordField = document.getElementById('form_keyword');
    if (keywordField) keywordField.value = '';
    
    hasPerformedSearch = true;
    performSearch();
    
    if (categoryId) {
        addClearCategoryButton(categoryTitle);
    } else {
        removeClearCategoryButton();
        selectedCategoryId = '';
        if (selectedCategoryText) selectedCategoryText.textContent = 'All categories';
    }
}

function showCategoryBrowser() {
    const browser = document.getElementById('categoryBrowser');
    const resultsContainer = document.getElementById('store_results');
    
    if (browser) {
        browser.style.display = 'block';
        setTimeout(() => {
            browser.classList.add('show');
            if (resultsContainer) resultsContainer.style.opacity = '0.3';
        }, 10);
    }
    
    isCategoryBrowserVisible = true;
}

function hideCategoryBrowser() {
    const browser = document.getElementById('categoryBrowser');
    const resultsContainer = document.getElementById('store_results');
    
    if (browser) {
        browser.classList.remove('show');
        if (resultsContainer) resultsContainer.style.opacity = '1';
        
        setTimeout(() => browser.style.display = 'none', 400);
    }
    isCategoryBrowserVisible = false;
}

function addClearCategoryButton(categoryTitle) {
    removeClearCategoryButton();
    const filterContainer = document.getElementById('filterToggleContainer');
    if (filterContainer) {
        const clearBtn = document.createElement('button');
        clearBtn.className = 'clear-category-btn';
        clearBtn.innerHTML = `<i class="fas fa-times"></i> Clear "${categoryTitle}"`;
        clearBtn.onclick = () => selectCategory('', 'Clear Categories');
        filterContainer.style.position = 'relative';
        filterContainer.appendChild(clearBtn);
    }
}

function removeClearCategoryButton() {
    const existingBtn = document.querySelector('.clear-category-btn');
    if (existingBtn) existingBtn.remove();
}

function updateFilterInfo(isOpenOnly) {
    const filterInfo = document.getElementById('filterInfo');
    if (filterInfo) {
        if (isOpenOnly) {
            filterInfo.textContent = 'Showing open businesses only';
            filterInfo.style.color = '#28a745';
        } else {
            filterInfo.textContent = 'Showing all businesses';
            filterInfo.style.color = '#666';
        }
    }
}

// ===========================================
// BUSINESS LOADING & SEARCH FUNCTIONS
// ===========================================
function loadAllBusinesses(openOnly = false) {
    if (isLoading) return;
    isLoading = true;
    
    console.log('=== LOAD ALL BUSINESSES ===', 'Open only:', openOnly, 'Selected category:', selectedCategoryId);
    
    showLoader();
    clearResults();
    currentPage = 1;
    hasMoreResults = true;
    removeScrollHandler();
    
    const latitude = document.getElementById('form_latitude')?.value || null;
    const longitude = document.getElementById('form_longitude')?.value || null;
    
    currentSearchData = {
        keyword: '',
        location: '',
        category: selectedCategoryId,
        coords: { lat: latitude, lng: longitude },
        page: 1,
        open_only: openOnly
    };
    
    makeRequest(currentSearchData);
}

function performSearch(isToggleChange = false) {
    if (isLoading) return;
    isLoading = true;
    
    if (!isToggleChange) hasPerformedSearch = true;
    
    const isOpenOnly = document.getElementById('openOnlyToggle')?.checked || false;
    const keyword = document.getElementById('form_keyword')?.value || 
                    document.getElementById('sticky_search_input')?.value || '';
    
    console.log('=== PERFORM SEARCH ===', 'Toggle change:', isToggleChange, 'Open only:', isOpenOnly, 'Keyword:', keyword);
    
    showLoader();
    clearResults();
    currentPage = 1;
    hasMoreResults = true;
    removeScrollHandler();
    
    const latitude = document.getElementById('form_latitude')?.value || null;
    const longitude = document.getElementById('form_longitude')?.value || null;
    
    currentSearchData = {
        keyword: keyword.trim(),
        location: '',
        category: selectedCategoryId,
        coords: { lat: latitude, lng: longitude },
        page: 1,
        open_only: isOpenOnly
    };
    
    makeRequest(currentSearchData);
}

function makeRequest(requestData) {
    console.log('=== MAKING REQUEST ===', requestData);

    fetch(ajaxLocatorUrl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(requestData)
    })
    .then(response => response.text())
    .then(html => {
        hideLoader();
        isLoading = false;

        if (html && html.trim()) {
            const metadata = parseMetadata(html) || {};
            hasMoreResults = !!metadata.has_more_results;

            if (metadata.total_open_businesses !== undefined && metadata.total_all_businesses !== undefined) {
                totalCounts.total_open = metadata.total_open_businesses;
                totalCounts.total_all = metadata.total_all_businesses;
            }

            console.log('=== RESPONSE RECEIVED ===', 'Results:', metadata.results_count, 'Has more:', hasMoreResults);

            const resultsContainer = document.getElementById('store_results');
            if (requestData.page === 1 && resultsContainer) {
                resultsContainer.innerHTML = html;
                // After this line: resultsContainer.innerHTML = html;
// Add this mobile constraint enforcement:
if (window.innerWidth <= 768) {
    const newCards = resultsContainer.querySelectorAll('.appstore-card, .stores_list_modal');
    newCards.forEach(card => {
        card.style.maxWidth = '100%';
        card.style.overflow = 'hidden';
    });
}
                const filterToggle = document.getElementById('filterToggleContainer');
                if (filterToggle) filterToggle.style.display = 'block';
            } else {
    appendNewResults(html);
    // Add mobile constraints for infinite scroll results too
    if (window.innerWidth <= 768) {
        const newCards = document.getElementById('store_results').querySelectorAll('.appstore-card, .stores_list_modal');
        newCards.forEach(card => {
            card.style.maxWidth = '100%';
            card.style.overflow = 'hidden';
        });
    }
}

            updateResultsCount(metadata.displayed_count || 0, requestData.open_only);
            setupModalHandlers();

            if (hasMoreResults) setupInfiniteScroll();
            isInitialLoad = false;
        } else {
            showNoResults();
            updateNoResultsMessage(requestData.open_only, hasPerformedSearch);
            hasMoreResults = false;
        }
    })
    .catch(error => {
        console.error('Request error:', error);
        hideLoader();
        showNoResults();
        isLoading = false;
        hasMoreResults = false;
    });
}

// Helper functions
function showLoader() {
    const loader = document.getElementById('loader');
    if (loader) loader.style.display = 'block';
}

function hideLoader() {
    const loader = document.getElementById('loader');
    if (loader) loader.style.display = 'none';
}

function clearResults() {
    const resultsContainer = document.getElementById('store_results');
    const noResults = document.getElementById('no_results');
    if (resultsContainer) resultsContainer.innerHTML = '';
    if (noResults) noResults.style.display = 'none';
}

function showNoResults() {
    const noResults = document.getElementById('no_results');
    if (noResults) noResults.style.display = 'block';
}

function appendNewResults(html) {
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = html;
    const newCards = tempDiv.querySelectorAll('.stores_list_modal');
    const resultsContainer = document.getElementById('store_results');

    newCards.forEach(card => {
        if (resultsContainer) {
            if (card.id && !resultsContainer.querySelector(`#${card.id}`)) {
                resultsContainer.appendChild(card.cloneNode(true));
            } else if (!card.id) {
                resultsContainer.appendChild(card.cloneNode(true));
            }
        }
    });
}

function parseMetadata(html) {
    const metadataMatch = html.match(/<!-- METADATA: (.+?) -->/);
    if (metadataMatch) {
        try {
            return JSON.parse(metadataMatch[1]);
        } catch (e) {
            console.error('Failed to parse metadata:', e);
        }
    }
    
    return {
        has_more_results: false,
        current_page: currentPage,
        results_count: 0,
        displayed_count: 0,
        total_open_businesses: 0,
        total_all_businesses: 0
    };
}

function setupModalHandlers() {
    const storeCards = document.querySelectorAll('.stores_list_modal');
    
    storeCards.forEach(card => {
        const newCard = card.cloneNode(true);
        card.parentNode.replaceChild(newCard, card);
        
        newCard.style.cursor = 'pointer';
        newCard.addEventListener('click', function() {
            const storeId = this.getAttribute('data-val') || this.dataset.val;
            if (storeId) {
                loadStoreModal(storeId);
            } else {
                console.error('No store ID found');
                alert('Unable to load business details.');
            }
        });
    });
}

// ===========================================
// MODAL HANDLING FUNCTIONS
// ===========================================
function loadStoreModal(storeId) {
    const modalContent = document.getElementById('modal-store-content');
    const modalElement = document.getElementById('store-info-dialog');
    if (!modalContent || !modalElement) return;
    
    modalContent.innerHTML = '<div class="text-center p-4"><img src="/assets/images/loader.gif" style="width:50px;"></div>';
    
    // Clean up any existing modal states first
    document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    document.body.classList.remove('modal-open');
    document.body.style.paddingRight = '';
    document.body.style.overflow = '';
    
    if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
        const modal = new bootstrap.Modal(modalElement);
        modal.show();
        
        // Enhanced cleanup when modal is hidden
        modalElement.addEventListener('hidden.bs.modal', function() {
            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
            document.body.classList.remove('modal-open');
            document.body.style.paddingRight = '';
            document.body.style.overflow = '';
            modalElement.style.display = 'none';
            modalElement.classList.remove('show');
            modalElement.setAttribute('aria-hidden', 'true');
            modalElement.removeAttribute('aria-modal');
            modalElement.removeAttribute('role');
        }, { once: true });
    } else {
        modalElement.style.display = 'block';
        modalElement.classList.add('show');
    }
    
    // Create form data for POST request
    const formData = new FormData();
    formData.append('id', storeId);
    
    fetch(storeDetailsUrl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.text())
    .then(html => {
        modalContent.innerHTML = html;
    })
    .catch(error => {
        console.error('Modal error:', error);
        modalContent.innerHTML = '<div class="text-center p-4 text-danger">Error loading details.</div>';
    });
}

// Infinite scroll setup
function setupInfiniteScroll() {
    removeScrollHandler();
    scrollHandler = function() {
        if ((window.innerHeight + window.scrollY) >= document.body.offsetHeight - 1000) {
            loadMoreResults();
        }
    };
    window.addEventListener('scroll', scrollHandler, { passive: true });
}

function removeScrollHandler() {
    if (scrollHandler) {
        window.removeEventListener('scroll', scrollHandler);
        scrollHandler = null;
    }
}

function loadMoreResults() {
    if (isLoading || !hasMoreResults) return;
    
    const currentToggleState = document.getElementById('openOnlyToggle')?.checked || false;
    isLoading = true;
    currentPage++;
    
    const loadMoreData = {
        ...currentSearchData,
        page: currentPage,
        open_only: currentToggleState,
        category: selectedCategoryId || ''
    };
    
    makeRequest(loadMoreData);
}

// Results display functions
function updateResultsCount(displayedCount, isOpenOnly) {
    const resultsCount = document.getElementById('resultsCount');
    if (!resultsCount) return;
    
    if (isOpenOnly) {
        const totalOpen = totalCounts.total_open || displayedCount;
        resultsCount.textContent = displayedCount >= totalOpen ? 
            `${totalOpen} open businesses` : 
            `${displayedCount} of ${totalOpen} open businesses`;
    } else {
        const totalAll = totalCounts.total_all || displayedCount;
        resultsCount.textContent = displayedCount >= totalAll ? 
            `${totalAll} businesses` : 
            `${displayedCount} of ${totalAll} businesses`;
    }
}

function updateNoResultsMessage(isOpenOnly, wasSearch) {
    const noResults = document.getElementById('no_results');
    if (!noResults) return;
    
    const messageElement = noResults.querySelector('.list-group-item-text strong');
    const detailElement = noResults.querySelector('div');
    
    if (messageElement && detailElement) {
        if (isOpenOnly) {
            messageElement.innerHTML = '<i class="fa fa-bed"></i> No businesses are currently open.';
            detailElement.innerHTML = '<br>Try turning off "Show Open Only" or searching in a different area.<br><br>Thanks for visiting!';
        } else {
            messageElement.innerHTML = '<i class="fa fa-info-circle"></i> No results for this search.';
            detailElement.innerHTML = '<br>Please try a different location or keyword.<br><br>Thanks for visiting!';
        }
    }
}

// ===========================================
// RESET FUNCTION
// ===========================================
function resetToHome() {
    console.log('Resetting to home state...');
    
    clearZipCode();
    
    const addressInput = document.getElementById('address_search_input');
    if (addressInput) addressInput.value = '';
    
    const stickyInput = document.getElementById('sticky_search_input');
    if (stickyInput) stickyInput.value = '';
    
    const keywordField = document.getElementById('form_keyword');
    if (keywordField) keywordField.value = '';
    
    const zipTab = document.getElementById('zip-tab');
    const addressTab = document.getElementById('address-tab');
    const zipSearchMode = document.getElementById('zip-search-mode');
    const addressSearchMode = document.getElementById('address-search-mode');
    const searchModeField = document.getElementById('search_mode');
    
    if (zipTab && addressTab && zipSearchMode && addressSearchMode) {
        zipTab.classList.add('active');
        addressTab.classList.remove('active');
        zipSearchMode.style.display = 'block';
        addressSearchMode.style.display = 'none';
        if (searchModeField) searchModeField.value = 'zip';
    }
    
    const openToggle = document.getElementById('openOnlyToggle');
    const categoryToggle = document.getElementById('categoryToggle');
    if (openToggle) openToggle.checked = false;
    if (categoryToggle) categoryToggle.checked = false;
    
    selectedCategoryId = '';
    const selectedCategoryField = document.getElementById('selected_category');
    if (selectedCategoryField) selectedCategoryField.value = '';
    
    const selectedCategoryText = document.getElementById('selectedCategoryText');
    if (selectedCategoryText) selectedCategoryText.textContent = 'All categories';
    
    hideCategoryBrowser();
    removeClearCategoryButton();
    
    const stickyClearBtn = document.getElementById('sticky-clear-btn');
    const clearSearchBtn = document.getElementById('clearSearchBtn');
    if (stickyClearBtn) stickyClearBtn.style.display = 'none';
    if (clearSearchBtn) clearSearchBtn.style.display = 'none';
    
    updateLocationStatus('Enter your ZIP code or click the crosshairs for nearby businesses', 'default');
    
    hasPerformedSearch = false;
    updateFilterInfo(false);
    loadAllBusinesses(false);
    
    console.log('Reset to home complete');
}

// Cleanup
window.addEventListener('beforeunload', function() {
    removeScrollHandler();
});