(function() {
    'use strict';
    
    // Configuration
    const CONFIG = {
        apiUrl: 'https://bestcallhvac.com/public/api/badge-verify.php',
        frontImage: 'https://www.bestcallhvac.com/HVAC_F_VIEW.png',
        rearImage: 'https://www.bestcallhvac.com/HVAC_R_VIEW.png',
        verifyUrl: 'https://bestcallhvac.com/verify/', // Placeholder URL
        ocrFont: '"OCR A Extended", "Courier New", monospace'
    };

    // Get the current script tag to extract data attributes
    const currentScript = document.currentScript || document.querySelector('script[data-form-id]');
    
    if (!currentScript) {
        console.error('BestCall Badge: Script tag not found or missing data-form-id');
        return;
    }

    const formId = currentScript.getAttribute('data-form-id');
    
    if (!formId) {
        console.error('BestCall Badge: data-form-id attribute is required');
        return;
    }

    // Create the badge widget
    function createBadgeWidget() {
        const today = new Date().toLocaleDateString('en-US', {
            month: '2-digit',
            day: '2-digit',  
            year: 'numeric'
        });

        // Create widget container
        const widget = document.createElement('div');
        widget.className = 'bestcall-badge-widget';
        
        // Widget HTML structure
        widget.innerHTML = `
            <div class="bestcall-badge-container">
                <div class="bestcall-badge">
                    <div class="bestcall-badge-inner">
                        <div class="bestcall-badge-front">
                            <img src="${CONFIG.frontImage}" alt="BestCall Certified Badge">
                        </div>
                        <div class="bestcall-badge-back">
                            <img src="${CONFIG.rearImage}" alt="BestCall Verified Badge">
                        </div>
                    </div>
                </div>
                <div class="bestcall-badge-info">
                    <div class="bestcall-badge-date">${today}</div>
                    <div class="bestcall-badge-code">${formId}</div>
                    <div class="bestcall-badge-verify">Verify Live</div>
                </div>
            </div>
        `;

        // Add click handler to badge
        const badgeContainer = widget.querySelector('.bestcall-badge');
        badgeContainer.addEventListener('click', function() {
            window.open(CONFIG.verifyUrl + formId, '_blank');
        });

        // Add hover effect
        badgeContainer.addEventListener('mouseenter', function() {
            badgeContainer.style.cursor = 'pointer';
        });

        return widget;
    }

    // Inject CSS styles
    function injectStyles() {
        const styleId = 'bestcall-badge-styles';
        
        // Check if styles already exist
        if (document.getElementById(styleId)) {
            return;
        }

        const styles = document.createElement('style');
        styles.id = styleId;
        styles.textContent = `
            .bestcall-badge-widget {
                font-family: ${CONFIG.ocrFont};
                text-align: center;
                display: inline-block;
                margin: 10px;
            }
            
            .bestcall-badge-container {
                display: flex;
                flex-direction: column;
                align-items: center;
            }
            
            .bestcall-badge {
                width: 120px;
                height: 120px;
                perspective: 1000px;
                cursor: pointer;
                transition: transform 0.3s ease;
            }
            
            .bestcall-badge:hover {
                transform: scale(1.05);
            }
            
            .bestcall-badge-inner {
                width: 100%;
                height: 100%;
                position: relative;
                transform-style: preserve-3d;
                animation: bestcall-spin 15s linear infinite;
            }
            
            .bestcall-badge-front,
            .bestcall-badge-back {
                position: absolute;
                width: 100%;
                height: 100%;
                backface-visibility: hidden;
                border-radius: 50%;
                overflow: hidden;
                box-shadow: 0 4px 8px rgba(0,0,0,0.2);
            }
            
            .bestcall-badge-front img,
            .bestcall-badge-back img {
                width: 100%;
                height: 100%;
                object-fit: cover;
                display: block;
            }
            
            .bestcall-badge-back {
                transform: rotateY(180deg);
            }
            
            .bestcall-badge-info {
                margin-top: 10px;
                line-height: 1.3;
            }
            
            .bestcall-badge-date {
                font-size: 14px;
                font-weight: bold;
                color: #333;
                margin-bottom: 3px;
            }
            
            .bestcall-badge-code {
                font-size: 16px;
                font-weight: bold;
                color: #0066cc;
                margin-bottom: 3px;
                letter-spacing: 1px;
            }
            
            .bestcall-badge-verify {
                font-size: 12px;
                color: #666;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            
            @keyframes bestcall-spin {
                from { transform: rotateY(0deg); }
                to { transform: rotateY(360deg); }
            }
            
            /* Responsive adjustments */
            @media (max-width: 768px) {
                .bestcall-badge {
                    width: 100px;
                    height: 100px;
                }
                
                .bestcall-badge-date {
                    font-size: 12px;
                }
                
                .bestcall-badge-code {
                    font-size: 14px;
                }
                
                .bestcall-badge-verify {
                    font-size: 11px;
                }
            }
        `;
        
        document.head.appendChild(styles);
    }

    // Verify business with API
    function verifyBusiness(formId, callback) {
        const xhr = new XMLHttpRequest();
        xhr.open('POST', CONFIG.apiUrl, true);
        xhr.setRequestHeader('Content-Type', 'application/json');
        
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        callback(response.valid === true, response.message);
                    } catch (e) {
                        callback(false, 'Invalid API response');
                    }
                } else {
                    callback(false, 'API request failed');
                }
            }
        };
        
        xhr.send(JSON.stringify({
            formId: formId,
            domain: window.location.hostname
        }));
    }

    // Initialize the widget
    function init() {
        // Verify business first
        verifyBusiness(formId, function(isValid, message) {
            if (isValid) {
                // Inject styles
                injectStyles();
                
                // Create and insert widget
                const widget = createBadgeWidget();
                
                // Insert widget after the script tag
                currentScript.parentNode.insertBefore(widget, currentScript.nextSibling);
                
            } else {
                console.error('BestCall Badge: ' + message);
                
                // Optionally show error message in development
                if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
                    const errorDiv = document.createElement('div');
                    errorDiv.style.cssText = 'color: red; font-size: 12px; padding: 5px; border: 1px solid red;';
                    errorDiv.textContent = 'BestCall Badge Error: ' + message;
                    currentScript.parentNode.insertBefore(errorDiv, currentScript.nextSibling);
                }
            }
        });
    }

    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();