<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Digital Business Verification Tool</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background: #f5f5f5;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #007bff;
            padding-bottom: 20px;
        }
        
        .header h1 {
            color: #333;
            margin: 0;
        }
        
        .header p {
            color: #666;
            margin: 10px 0 0 0;
        }
        
        .controls {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            font-weight: bold;
            transition: all 0.3s ease;
        }
        
        .btn-primary { background: #007bff; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-info { background: #17a2b8; color: white; }
        
        .btn:hover { opacity: 0.8; transform: translateY(-1px); }
        
        .address-input-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border: 2px solid #007bff;
        }
        
        .input-group {
            display: flex;
            gap: 10px;
            align-items: center;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .input-group input {
            flex: 1;
            min-width: 300px;
            max-width: 500px;
            padding: 12px;
            font-size: 16px;
            border: 2px solid #ddd;
            border-radius: 5px;
        }
        
        .address-status {
            text-align: center;
            margin-top: 10px;
            padding: 8px;
            border-radius: 4px;
            font-weight: bold;
        }
        
        .status-success { background: #d4edda; color: #155724; }
        .status-error { background: #f8d7da; color: #721c24; }
        .status-warning { background: #fff3cd; color: #856404; }
        
        .info-bar {
            background: #e9ecef;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            text-align: center;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .verification-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        
        .verification-table th {
            background: #343a40;
            color: white;
            padding: 12px 8px;
            text-align: left;
            font-weight: bold;
            border: 1px solid #dee2e6;
            cursor: pointer;
            user-select: none;
            position: relative;
        }
        
        .verification-table th:hover {
            background: #495057;
        }
        
        .verification-table th.sortable::after {
            content: ' ⇅';
            opacity: 0.5;
        }
        
        .verification-table th.sort-asc::after {
            content: ' ↑';
            opacity: 1;
            color: #007bff;
        }
        
        .verification-table th.sort-desc::after {
            content: ' ↓';
            opacity: 1;
            color: #007bff;
        }
        
        .verification-table td {
            padding: 10px 8px;
            border: 1px solid #dee2e6;
            vertical-align: top;
        }
        
        .verification-table tbody tr:nth-child(even) {
            background: #f8f9fa;
        }
        
        .verification-table tbody tr:hover {
            background: #e3f2fd;
        }
        
        .verification-table tbody tr.verified {
            background: #d4f6d4;
        }
        
        .verification-table tbody tr.verified:hover {
            background: #c3e6c3;
        }
        
        .verify-btn {
            padding: 6px 12px;
            font-size: 12px;
            border-radius: 4px;
            border: none;
            cursor: pointer;
            margin-right: 5px;
        }
        
        .verify-verified { background: #28a745; color: white; }
        .verify-not-found { background: #dc3545; color: white; }
        .verify-relocated { background: #ffc107; color: black; }
        .verify-closed { background: #6c757d; color: white; }
        
        .notes-input {
            width: 100%;
            min-height: 60px;
            padding: 5px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 12px;
            resize: vertical;
        }
        
        .last-verified {
            font-size: 11px;
            color: #666;
        }
        
        .never-verified {
            color: #dc3545;
            font-weight: bold;
        }
        
        .recently-verified {
            color: #28a745;
        }
        
        .distance-col { width: 80px; text-align: center; }
        .name-col { width: 200px; font-weight: bold; }
        .address-col { width: 250px; }
        .status-col { width: 120px; text-align: center; }
        .verified-col { width: 140px; text-align: center; }
        .actions-col { width: 200px; text-align: center; }
        .notes-col { width: 200px; }
        
        .status-open {
            background: #d4edda;
            color: #155724;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .status-closed {
            background: #f8d7da;
            color: #721c24;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        
        .error {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .new-business-section {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 20px;
            border-radius: 8px;
            margin-top: 30px;
        }
        
        .new-business-form {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-top: 15px;
        }
        
        .new-business-form input,
        .new-business-form select {
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
        }
        
        .new-business-form .full-width {
            grid-column: 1 / -1;
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }
        
        .modal-content {
            background-color: white;
            margin: 15% auto;
            padding: 20px;
            border-radius: 8px;
            width: 80%;
            max-width: 500px;
        }
        
        .close {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }
        
        .close:hover {
            color: black;
        }
        
        /* Mobile Card Layout */
        .mobile-business-card {
            display: none;
            background: white;
            border: 1px solid #ddd;
            border-radius: 8px;
            margin-bottom: 15px;
            padding: 15px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .mobile-business-card.verified {
            background: #f0fff0;
            border-color: #90EE90;
        }
        
        .mobile-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 10px;
        }
        
        .mobile-business-name {
            font-weight: bold;
            font-size: 16px;
            color: #333;
            flex: 1;
            margin-right: 10px;
        }
        
        .mobile-distance {
            background: #007bff;
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: bold;
            white-space: nowrap;
        }
        
        .mobile-address {
            color: #666;
            font-size: 14px;
            margin-bottom: 8px;
            line-height: 1.3;
        }
        
        .mobile-status-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        
        .mobile-status {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .mobile-verified {
            font-size: 11px;
            color: #666;
        }
        
        .mobile-verified.never-verified {
            color: #dc3545;
            font-weight: bold;
        }
        
        .mobile-verified.recently-verified {
            color: #28a745;
        }
        
        .mobile-actions {
            display: flex;
            gap: 8px;
            margin-bottom: 10px;
            flex-wrap: wrap;
        }
        
        .mobile-verify-btn {
            padding: 6px 12px;
            font-size: 11px;
            border-radius: 4px;
            border: none;
            cursor: pointer;
            flex: 1;
            min-width: 80px;
        }
        
        .mobile-notes {
            margin-top: 10px;
        }
        
        .mobile-notes textarea {
            width: 100%;
            min-height: 50px;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 12px;
            resize: vertical;
        }

        @media (max-width: 768px) {
            .container {
                margin: 5px;
                padding: 10px;
            }
            
            .header h1 {
                font-size: 20px;
            }
            
            .header p {
                font-size: 14px;
            }
            
            .address-input-section {
                padding: 15px;
            }
            
            .input-group {
                flex-direction: column;
                gap: 10px;
            }
            
            .input-group input {
                min-width: 100%;
                max-width: none;
            }
            
            .controls {
                flex-direction: column;
                gap: 8px;
            }
            
            .controls .btn {
                width: 100%;
                max-width: 300px;
            }
            
            .info-bar {
                flex-direction: column;
                gap: 8px;
                text-align: center;
                font-size: 14px;
            }
            
            /* Hide table on mobile, show cards */
            .verification-table {
                display: none !important;
            }
            
            .mobile-business-card {
                display: block;
            }
            
            .new-business-form {
                grid-template-columns: 1fr;
            }
            
            .modal-content {
                width: 95%;
                margin: 10% auto;
                padding: 15px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📱 Digital Business Verification Tool</h1>
            <p>Real-time verification tracking with digital notes</p>
            <div id="location-info"></div>
        </div>
        
        <div class="address-input-section">
            <div class="address-input-container">
                <h3 style="text-align: center; color: #007bff; margin-bottom: 15px;">📍 Choose Starting Point</h3>
                <div class="input-group">
                    <input type="text" 
                           id="custom-address" 
                           class="form-control" 
                           placeholder="Enter starting address (e.g., 123 Main St, City, State)">
                    <button class="btn btn-success btn-lg" onclick="sortFromAddress()">
                        🎯 Sort from This Address
                    </button>
                    <button class="btn btn-info btn-lg" onclick="useCurrentLocation()">
                        📱 Use My Location
                    </button>
                </div>
                <div id="address-status" class="address-status"></div>
            </div>
        </div>
        
        <div class="controls">
            <button class="btn btn-primary" onclick="loadBusinesses()">🔄 Refresh List</button>
            <button class="btn btn-secondary" onclick="sortAlphabetically()">🔤 Sort Alphabetically</button>
            <button class="btn btn-success" onclick="exportToExcel()">📊 Export to Excel</button>
            <button class="btn btn-warning" onclick="toggleNewBusinessForm()">➕ Add New Business</button>
            <button class="btn btn-info" onclick="saveAllNotes()">💾 Save All Notes</button>
        </div>
        
        <div class="info-bar">
            <div>
                <span id="business-count">Loading businesses...</span> | 
                <span id="sort-method">Distance from your location</span>
            </div>
            <div>
                <span id="generation-time"></span> | 
                <span id="verified-count">Verified: 0</span>
            </div>
        </div>
        
        <div id="loading" class="loading">
            <p>📍 Getting your location and loading businesses...</p>
        </div>
        
        <div id="error-message" class="error" style="display: none;"></div>
        
        <!-- Desktop Table View -->
        <table class="verification-table" id="verification-table" style="display: none;">
            <thead>
                <tr>
                    <th class="distance-col sortable" data-sort="distance">Distance</th>
                    <th class="name-col sortable" data-sort="name">Business Name</th>
                    <th class="address-col sortable" data-sort="address">Address</th>
                    <th class="status-col sortable" data-sort="status">Current Status</th>
                    <th class="verified-col sortable" data-sort="verified">Last Verified</th>
                    <th class="actions-col">Verification Actions</th>
                    <th class="notes-col">Admin Notes</th>
                </tr>
            </thead>
            <tbody id="business-tbody">
            </tbody>
        </table>
        
        <!-- Mobile Card View -->
        <div id="mobile-business-list" style="display: none;">
        </div>
        
        <div class="new-business-section" id="new-business-section" style="display: none;">
            <h3 style="margin-top: 0; color: #856404;">🏪 Add New Business Discovery</h3>
            <p>Found a business that's not in our system? Add it here and we'll process it later.</p>
            
            <div class="new-business-form">
                <input type="text" id="new-business-name" placeholder="Business Name *" required>
                <input type="text" id="new-business-type" placeholder="Business Type (e.g., Restaurant, Auto Repair)">
                <input type="text" id="new-street-address" placeholder="Street Address *" required class="full-width">
                <input type="text" id="new-city" placeholder="City *" required>
                <input type="text" id="new-state" placeholder="State *" required maxlength="2">
                <input type="text" id="new-zip" placeholder="Zip Code *" required>
                <input type="tel" id="new-phone" placeholder="Phone Number">
                <textarea id="new-admin-notes" placeholder="Additional notes about this business..." class="full-width" style="min-height: 80px;"></textarea>
            </div>
            
            <div style="text-align: center; margin-top: 20px;">
                <button class="btn btn-success" onclick="saveNewBusiness()">💾 Save New Business</button>
                <button class="btn btn-secondary" onclick="cancelNewBusiness()">❌ Cancel</button>
            </div>
        </div>
    </div>
    
    <!-- Verification Modal -->
    <div id="verificationModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeVerificationModal()">&times;</span>
            <h3 id="modal-business-name">Verify Business</h3>
            <p>Choose verification status:</p>
            <div style="margin: 20px 0;">
                <button class="verify-btn verify-verified" onclick="verifyBusiness('verified')">✅ Verified</button>
                <button class="verify-btn verify-not-found" onclick="verifyBusiness('not_found')">❌ Not Found</button>
                <button class="verify-btn verify-relocated" onclick="verifyBusiness('relocated')">📍 Relocated</button>
                <button class="verify-btn verify-closed" onclick="verifyBusiness('closed_permanently')">🏪 Closed</button>
            </div>
            <textarea id="modal-notes" placeholder="Add verification notes..." style="width: 100%; height: 80px; padding: 10px; border: 1px solid #ddd; border-radius: 4px;"></textarea>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script>
        let businessData = [];
        let currentSort = 'distance';
        let sortDirection = 'asc';
        let userLocation = null;
        let customLocation = null;
        let currentSortingAddress = '';
        let currentVerificationBusinessId = null;

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            checkPasswordProtection();
        });

        // Password protection check
        function checkPasswordProtection() {
            const password = prompt('Enter verification tool password:');
            
            fetch('<?= base_url('verification/check-password') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ password: password })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    initializeApp();
                } else {
                    alert('❌ Incorrect password. Access denied.');
                    window.location.href = '/';
                }
            })
            .catch(error => {
                console.error('Password check error:', error);
                alert('❌ Error checking password. Please try again.');
                window.location.href = '/';
            });
        }

        // Initialize app after password verification
        function initializeApp() {
            getCurrentLocation();
            setupTableSorting();
        }

        // Setup table sorting
        function setupTableSorting() {
            const sortableHeaders = document.querySelectorAll('.sortable');
            sortableHeaders.forEach(header => {
                header.addEventListener('click', function() {
                    const sortBy = this.dataset.sort;
                    sortTable(sortBy);
                });
            });
        }

        // Sort table by column
        function sortTable(sortBy) {
            if (currentSort === sortBy) {
                sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
            } else {
                sortDirection = 'asc';
                currentSort = sortBy;
            }
            
            // Update header classes
            document.querySelectorAll('.sortable').forEach(th => {
                th.classList.remove('sort-asc', 'sort-desc');
            });
            
            const currentHeader = document.querySelector(`[data-sort="${sortBy}"]`);
            currentHeader.classList.add(sortDirection === 'asc' ? 'sort-asc' : 'sort-desc');
            
            // Sort the data
            businessData.sort((a, b) => {
                let aVal, bVal;
                
                switch(sortBy) {
                    case 'distance':
                        aVal = a.distance || 999;
                        bVal = b.distance || 999;
                        break;
                    case 'name':
                        aVal = a.name.toLowerCase();
                        bVal = b.name.toLowerCase();
                        break;
                    case 'address':
                        aVal = a.fullAddress.toLowerCase();
                        bVal = b.fullAddress.toLowerCase();
                        break;
                    case 'status':
                        aVal = a.status.toLowerCase();
                        bVal = b.status.toLowerCase();
                        break;
                    case 'verified':
                        aVal = a.lastVerified || '0';
                        bVal = b.lastVerified || '0';
                        break;
                    default:
                        return 0;
                }
                
                if (typeof aVal === 'string') {
                    return sortDirection === 'asc' ? aVal.localeCompare(bVal) : bVal.localeCompare(aVal);
                } else {
                    return sortDirection === 'asc' ? aVal - bVal : bVal - aVal;
                }
            });
            
            displayBusinessTable();
            updateInfoBar();
        }

        // Get user's current location
        function getCurrentLocation() {
            const locationInfo = document.getElementById('location-info');
            
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(
                    function(position) {
                        userLocation = {
                            lat: position.coords.latitude,
                            lng: position.coords.longitude
                        };
                        
                        locationInfo.innerHTML = `<small>📱 Your Current Location: ${userLocation.lat.toFixed(4)}, ${userLocation.lng.toFixed(4)}</small>`;
                        currentSortingAddress = 'Your current location';
                        loadBusinesses();
                    },
                    function(error) {
                        console.error('Geolocation error:', error);
                        locationInfo.innerHTML = '<small>⚠️ Location not available - use address input below</small>';
                        showAddressPrompt();
                    },
                    {
                        timeout: 10000,
                        enableHighAccuracy: true
                    }
                );
            } else {
                locationInfo.innerHTML = '<small>⚠️ Geolocation not supported - use address input below</small>';
                showAddressPrompt();
            }
        }

        // Show prompt for address input
        function showAddressPrompt() {
            const addressStatus = document.getElementById('address-status');
            addressStatus.innerHTML = 'Enter a starting address above to sort businesses by distance';
            addressStatus.className = 'address-status status-warning';
        }

        // Use current location button
        function useCurrentLocation() {
            if (userLocation) {
                customLocation = null;
                currentSortingAddress = 'Your current location';
                updateAddressStatus('Using your current location', 'success');
                loadBusinesses();
            } else {
                getCurrentLocation();
            }
        }

        // Sort from custom address
        async function sortFromAddress() {
            const addressInput = document.getElementById('custom-address');
            const address = addressInput.value.trim();
            
            if (!address) {
                updateAddressStatus('Please enter an address', 'error');
                return;
            }

            updateAddressStatus('Looking up address...', 'warning');

            try {
                const coords = await geocodeAddress(address);
                
                if (coords) {
                    customLocation = coords;
                    currentSortingAddress = address;
                    updateAddressStatus(`✅ Found: ${address}`, 'success');
                    loadBusinesses();
                } else {
                    updateAddressStatus('Address not found. Try a different format.', 'error');
                }
            } catch (error) {
                console.error('Geocoding error:', error);
                updateAddressStatus('Error looking up address. Please try again.', 'error');
            }
        }

        // Geocode address using a free service
        async function geocodeAddress(address) {
            try {
                const encodedAddress = encodeURIComponent(address);
                const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodedAddress}&limit=1`);
                const data = await response.json();
                
                if (data && data.length > 0) {
                    return {
                        lat: parseFloat(data[0].lat),
                        lng: parseFloat(data[0].lon)
                    };
                }
                
                return null;
            } catch (error) {
                console.error('Geocoding API error:', error);
                return null;
            }
        }

        // Update address status display
        function updateAddressStatus(message, type) {
            const addressStatus = document.getElementById('address-status');
            addressStatus.innerHTML = message;
            addressStatus.className = `address-status status-${type}`;
        }

        // Load all businesses from the server
        function loadBusinesses() {
            const loading = document.getElementById('loading');
            const table = document.getElementById('verification-table');
            const errorMessage = document.getElementById('error-message');
            
            const activeLocation = customLocation || userLocation;
            
            if (!activeLocation) {
                showError('Please provide a location (current location or address) to sort businesses by distance.');
                return;
            }
            
            loading.style.display = 'block';
            table.style.display = 'none';
            errorMessage.style.display = 'none';

            const requestData = {
                coords: activeLocation,
                verification_mode: true
            };

            fetch('<?= base_url('verification/data') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(requestData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    businessData = data.businesses;
                    
                    // Sort by distance initially
                    sortTable('distance');
                    
                    loading.style.display = 'none';
                    
                    // Show appropriate view based on screen size
                    const table = document.getElementById('verification-table');
                    const mobileList = document.getElementById('mobile-business-list');
                    
                    if (window.innerWidth <= 768) {
                        table.style.display = 'none';
                        mobileList.style.display = 'block';
                    } else {
                        table.style.display = 'table';
                        mobileList.style.display = 'none';
                    }
                    
                    updateInfoBar();
                } else {
                    showError(data.message || 'Failed to load business data');
                    loading.style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Error loading businesses:', error);
                showError('Failed to load business data. Please try again.');
                loading.style.display = 'none';
            });
        }

        // Display business table and mobile cards
        function displayBusinessTable() {
            const tbody = document.getElementById('business-tbody');
            const mobileList = document.getElementById('mobile-business-list');
            tbody.innerHTML = '';
            mobileList.innerHTML = '';
            
            businessData.forEach((business, index) => {
                // Check if verified recently
                const isVerified = business.lastVerified && business.lastVerified !== 'Never';
                
                const statusClass = business.status.toLowerCase().includes('open') ? 'status-open' : 'status-closed';
                const distanceDisplay = business.distance === 999 ? 'N/A' : `${business.distance.toFixed(1)} MI`;
                
                let verifiedDisplay = 'Never';
                let verifiedClass = 'never-verified';
                
                if (business.lastVerified && business.lastVerified !== 'Never') {
                    const verifiedDate = new Date(business.lastVerified);
                    const now = new Date();
                    const daysDiff = Math.floor((now - verifiedDate) / (1000 * 60 * 60 * 24));
                    
                    if (daysDiff === 0) {
                        verifiedDisplay = 'Today';
                        verifiedClass = 'recently-verified';
                    } else if (daysDiff === 1) {
                        verifiedDisplay = 'Yesterday';
                        verifiedClass = 'recently-verified';
                    } else if (daysDiff <= 7) {
                        verifiedDisplay = `${daysDiff} days ago`;
                        verifiedClass = 'recently-verified';
                    } else {
                        verifiedDisplay = verifiedDate.toLocaleDateString();
                        verifiedClass = '';
                    }
                }
                
                // Desktop table row
                const row = document.createElement('tr');
                if (isVerified) {
                    row.classList.add('verified');
                }
                
                row.innerHTML = `
                    <td class="distance-col">${distanceDisplay}</td>
                    <td class="name-col">${business.name}</td>
                    <td class="address-col">${business.fullAddress}</td>
                    <td class="status-col">
                        <span class="${statusClass}">${business.status}</span>
                    </td>
                    <td class="verified-col">
                        <span class="last-verified ${verifiedClass}">${verifiedDisplay}</span>
                    </td>
                    <td class="actions-col">
                        <button class="verify-btn verify-verified" onclick="openVerificationModal(${business.id}, '${business.name}')">
                            🔍 Verify
                        </button>
                        <button class="verify-btn verify-closed" onclick="markInactive(${business.id}, '${business.name}')" 
                                style="background: #dc3545; color: white; margin-left: 5px;">
                            🚫 Mark Inactive
                        </button>
                    </td>
                    <td class="notes-col">
                        <textarea class="notes-input" 
                                  data-business-id="${business.id}" 
                                  placeholder="Admin notes...">${business.adminNotes || ''}</textarea>
                    </td>
                `;
                
                tbody.appendChild(row);
                
                // Mobile card
                const card = document.createElement('div');
                card.className = `mobile-business-card ${isVerified ? 'verified' : ''}`;
                
                card.innerHTML = `
                    <div class="mobile-card-header">
                        <div class="mobile-business-name">${business.name}</div>
                        <div class="mobile-distance">${distanceDisplay}</div>
                    </div>
                    
                    <div class="mobile-address">${business.fullAddress}</div>
                    
                    <div class="mobile-status-row">
                        <span class="mobile-status ${statusClass}">${business.status}</span>
                        <span class="mobile-verified ${verifiedClass}">
                            Last verified: ${verifiedDisplay}
                        </span>
                    </div>
                    
                    <div class="mobile-actions">
                        <button class="mobile-verify-btn verify-verified" onclick="openVerificationModal(${business.id}, '${business.name}')">
                            🔍 Verify
                        </button>
                        <button class="mobile-verify-btn verify-closed" onclick="markInactive(${business.id}, '${business.name}')" 
                                style="background: #dc3545; color: white;">
                            🚫 Inactive
                        </button>
                    </div>
                    
                    <div class="mobile-notes">
                        <textarea class="notes-input" 
                                  data-business-id="${business.id}" 
                                  placeholder="Admin notes...">${business.adminNotes || ''}</textarea>
                    </div>
                `;
                
                mobileList.appendChild(card);
            });
            
            // Show/hide appropriate view
            const table = document.getElementById('verification-table');
            const mobileListContainer = document.getElementById('mobile-business-list');
            
            if (window.innerWidth <= 768) {
                table.style.display = 'none';
                mobileListContainer.style.display = 'block';
            } else {
                table.style.display = 'table';
                mobileListContainer.style.display = 'none';
            }
        }

        // Open verification modal
        function openVerificationModal(businessId, businessName) {
            currentVerificationBusinessId = businessId;
            document.getElementById('modal-business-name').textContent = `Verify: ${businessName}`;
            document.getElementById('modal-notes').value = '';
            document.getElementById('verificationModal').style.display = 'block';
        }

        // Close verification modal
        function closeVerificationModal() {
            document.getElementById('verificationModal').style.display = 'none';
            currentVerificationBusinessId = null;
        }

        // Verify business
        function verifyBusiness(status) {
            if (!currentVerificationBusinessId) return;
            
            const notes = document.getElementById('modal-notes').value;
            
            const verificationData = {
                store_id: currentVerificationBusinessId,
                verification_status: status,
                admin_notes: notes,
                verified_by: 'Admin' // This should come from actual login system
            };
            
            fetch('<?= base_url('verification/verify') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(verificationData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update the business data locally
                    const business = businessData.find(b => b.id == currentVerificationBusinessId);
                    if (business) {
                        business.lastVerified = new Date().toISOString();
                        business.verificationStatus = status;
                    }
                    
                    // Refresh the display
                    displayBusinessTable();
                    updateInfoBar();
                    
                    closeVerificationModal();
                    
                    // Show success message
                    alert(`✅ Business verified as: ${status.replace('_', ' ')}`);
                } else {
                    alert('❌ Error saving verification: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Verification error:', error);
                alert('❌ Error saving verification. Please try again.');
            });
        }

        // Save all notes
        function saveAllNotes() {
            const noteInputs = document.querySelectorAll('.notes-input');
            const notesToSave = [];
            
            noteInputs.forEach(input => {
                if (input.value.trim()) {
                    notesToSave.push({
                        business_id: input.dataset.businessId,
                        notes: input.value.trim()
                    });
                }
            });
            
            if (notesToSave.length === 0) {
                alert('No notes to save');
                return;
            }
            
            fetch('<?= base_url('verification/save-notes') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ notes: notesToSave })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(`✅ Saved notes for ${data.saved_count} businesses`);
                } else {
                    alert('❌ Error saving notes: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Save notes error:', error);
                alert('❌ Error saving notes. Please try again.');
            });
        }

        // Toggle new business form
        function toggleNewBusinessForm() {
            const section = document.getElementById('new-business-section');
            const isHidden = section.style.display === 'none' || section.style.display === '';
            
            if (isHidden) {
                section.style.display = 'block';
                // Scroll to the form
                section.scrollIntoView({ behavior: 'smooth', block: 'start' });
            } else {
                section.style.display = 'none';
            }
        }

        // Cancel new business
        function cancelNewBusiness() {
            document.getElementById('new-business-section').style.display = 'none';
            clearNewBusinessForm();
        }

        // Clear new business form
        function clearNewBusinessForm() {
            document.getElementById('new-business-name').value = '';
            document.getElementById('new-business-type').value = '';
            document.getElementById('new-street-address').value = '';
            document.getElementById('new-city').value = '';
            document.getElementById('new-state').value = '';
            document.getElementById('new-zip').value = '';
            document.getElementById('new-phone').value = '';
            document.getElementById('new-admin-notes').value = '';
        }

        // Save new business
        async function saveNewBusiness() {
            const name = document.getElementById('new-business-name').value.trim();
            const streetAddress = document.getElementById('new-street-address').value.trim();
            const city = document.getElementById('new-city').value.trim();
            const state = document.getElementById('new-state').value.trim();
            const zipCode = document.getElementById('new-zip').value.trim();
            
            if (!name || !streetAddress || !city || !state || !zipCode) {
                alert('Please fill in all required fields (marked with *)');
                return;
            }
            
            const newBusinessData = {
                business_name: name,
                street_address: streetAddress,
                city: city,
                state: state.toUpperCase(),
                zip_code: zipCode,
                phone: document.getElementById('new-phone').value.trim(),
                business_type: document.getElementById('new-business-type').value.trim(),
                admin_notes: document.getElementById('new-admin-notes').value.trim(),
                discovered_by: 'Admin' // This should come from actual login system
            };
            
            // Try to geocode the address
            const fullAddress = `${streetAddress}, ${city}, ${state} ${zipCode}`;
            try {
                const coords = await geocodeAddress(fullAddress);
                if (coords) {
                    newBusinessData.latitude = coords.lat;
                    newBusinessData.longitude = coords.lng;
                }
            } catch (error) {
                console.warn('Could not geocode new business address:', error);
            }
            
            fetch('<?= base_url('verification/add-business') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(newBusinessData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(`✅ New business "${name}" added successfully!`);
                    clearNewBusinessForm();
                    cancelNewBusiness();
                    
                    // Optionally reload the list to include the new business
                    if (confirm('Reload the business list to include the new business?')) {
                        loadBusinesses();
                    }
                } else {
                    alert('❌ Error adding business: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Add business error:', error);
                alert('❌ Error adding business. Please try again.');
            });
        }

        // Sort alphabetically
        function sortAlphabetically() {
            sortTable('name');
        }

        // Update info bar
        function updateInfoBar() {
            const businessCount = document.getElementById('business-count');
            const sortMethod = document.getElementById('sort-method');
            const generationTime = document.getElementById('generation-time');
            const verifiedCount = document.getElementById('verified-count');
            
            businessCount.textContent = `${businessData.length} businesses loaded`;
            
            const verifiedBusinesses = businessData.filter(b => b.lastVerified && b.lastVerified !== 'Never').length;
            verifiedCount.textContent = `Verified: ${verifiedBusinesses}`;
            
            if (currentSort === 'distance') {
                sortMethod.textContent = `Sorted by distance from: ${currentSortingAddress}`;
            } else {
                const sortLabel = currentSort.charAt(0).toUpperCase() + currentSort.slice(1);
                sortMethod.textContent = `Sorted by ${sortLabel} (${sortDirection})`;
            }
            
            generationTime.textContent = `Updated: ${new Date().toLocaleString()}`;
        }

        // Mark business as inactive
        function markInactive(businessId, businessName) {
            if (!confirm(`⚠️ IMPORTANT: This will mark "${businessName}" as INACTIVE and hide it from public search.\n\nThis action should only be used for businesses that are permanently closed or no longer operational.\n\nContinue?`)) {
                return;
            }

            const reason = prompt('Please provide a reason for marking this business inactive:');
            if (reason === null) return; // User cancelled

            const inactiveData = {
                store_id: businessId,
                reason: reason.trim() || 'No reason provided',
                marked_by: 'Admin' // This should come from actual login system
            };

            fetch('<?= base_url('verification/mark-inactive') ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(inactiveData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(`✅ "${businessName}" has been marked as inactive and removed from public search.`);
                    
                    // Remove from current list immediately
                    businessData = businessData.filter(b => b.id != businessId);
                    displayBusinessTable();
                    updateInfoBar();
                } else {
                    alert('❌ Error marking business inactive: ' + (data.message || 'Unknown error'));
                }
            })
            .catch(error => {
                console.error('Mark inactive error:', error);
                alert('❌ Error marking business inactive. Please try again.');
            });
        }
        function exportToExcel() {
            if (businessData.length === 0) {
                alert('No data to export. Please load businesses first.');
                return;
            }

            // Prepare verification data
            const verificationData = businessData.map((business) => ({
                'Distance (MI)': business.distance === 999 ? 'N/A' : business.distance.toFixed(1),
                'Business Name': business.name,
                'Street Address': business.streetAddress,
                'City': business.city,
                'State': business.state,
                'Zip Code': business.zipCode,
                'Current Status': business.status,
                'Last Verified': business.lastVerified || 'Never',
                'Verification Status': business.verificationStatus || 'Not Verified',
                'Admin Notes': business.adminNotes || ''
            }));

            const worksheet = XLSX.utils.json_to_sheet(verificationData);
            worksheet['!cols'] = [
                { width: 10 }, // Distance
                { width: 25 }, // Business Name
                { width: 30 }, // Street Address
                { width: 15 }, // City
                { width: 8 },  // State
                { width: 10 }, // Zip Code
                { width: 15 }, // Status
                { width: 15 }, // Last Verified
                { width: 15 }, // Verification Status
                { width: 30 }  // Admin Notes
            ];

            const workbook = XLSX.utils.book_new();
            XLSX.utils.book_append_sheet(workbook, worksheet, 'Digital Verification');

            // Add summary sheet
            const verifiedCount = businessData.filter(b => b.lastVerified && b.lastVerified !== 'Never').length;
            const summaryData = [
                ['Digital Verification Report'],
                [''],
                ['Generated On:', new Date().toLocaleString()],
                ['Total Businesses:', businessData.length],
                ['Verified Businesses:', verifiedCount],
                ['Unverified Businesses:', businessData.length - verifiedCount],
                ['Sorted By:', currentSort === 'distance' ? `Distance from: ${currentSortingAddress}` : `${currentSort} (${sortDirection})`],
                [''],
                ['Digital Features:'],
                ['• Real-time verification tracking'],
                ['• Timestamp recording'],
                ['• Admin notes system'],
                ['• New business discovery'],
                ['• Sortable columns']
            ];
            
            const summaryWorksheet = XLSX.utils.aoa_to_sheet(summaryData);
            summaryWorksheet['!cols'] = [{ width: 20 }, { width: 40 }];
            XLSX.utils.book_append_sheet(workbook, summaryWorksheet, 'Summary');

            const timestamp = new Date().toISOString().split('T')[0];
            const filename = `digital-verification-${timestamp}.xlsx`;
            XLSX.writeFile(workbook, filename);
        }

        // Show error message
        function showError(message) {
            const errorDiv = document.getElementById('error-message');
            errorDiv.textContent = message;
            errorDiv.style.display = 'block';
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('verificationModal');
            if (event.target === modal) {
                closeVerificationModal();
            }
        }

        // Handle window resize to switch between table and mobile view
        window.addEventListener('resize', function() {
            const table = document.getElementById('verification-table');
            const mobileList = document.getElementById('mobile-business-list');
            
            if (window.innerWidth <= 768) {
                table.style.display = 'none';
                mobileList.style.display = 'block';
            } else {
                table.style.display = 'table';
                mobileList.style.display = 'none';
            }
        });
    </script>
</body>
</html>