<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hours Import Tool - Business Directory Admin</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }

        .admin-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
            padding: 1rem 0;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.1);
        }

        .nav-container {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 2rem;
        }

        .nav-brand {
            font-size: 1.5rem;
            font-weight: 700;
            color: #4a5568;
            text-decoration: none;
        }

        .nav-links {
            display: flex;
            gap: 0;
        }

        .nav-links a {
            color: #4a5568;
            text-decoration: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            transition: all 0.3s ease;
            font-weight: 500;
        }

        .nav-links a:hover {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
        }

        .nav-links a.active {
            background: #667eea;
            color: white;
            box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
        }

        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .page-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-left: 4px solid;
        }

        .stat-card.danger { border-left-color: #dc3545; }
        .stat-card.warning { border-left-color: #ffc107; }
        .stat-card.success { border-left-color: #28a745; }
        .stat-card.info { border-left-color: #17a2b8; }
        .stat-card.test-data { border-left-color: #fd7e14; }

        .businesses-section {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }

        .place-result {
            border: 2px solid #dee2e6;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .place-result:hover {
            border-color: #667eea;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.2);
        }

        .place-result.selected {
            border-color: #667eea;
            background-color: #f8f9ff;
        }

        .confidence-badge {
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 12px;
            font-weight: 600;
        }

        .confidence-high { background: #d4edda; color: #155724; }
        .confidence-medium { background: #fff3cd; color: #856404; }
        .confidence-low { background: #f8d7da; color: #721c24; }

        .test-data-badge {
            background: #fd7e14;
            color: white;
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .table-hover tbody tr:hover {
            background-color: rgba(102, 126, 234, 0.05);
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 10px;
            margin-bottom: 1.5rem;
            border-left: 4px solid;
            font-weight: 500;
        }

        .alert-success {
            background: #d4edda;
            border-color: #28a745;
            color: #155724;
        }

        .alert-danger {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }

        .alert-info {
            background: #d1ecf1;
            border-color: #17a2b8;
            color: #0c5460;
        }

        .merge-strategy-card {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .merge-strategy-card:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }

        .merge-strategy-card.selected {
            border-color: #667eea;
            background: #f8f9ff;
        }

        .service-type-tag {
            display: inline-block;
            background: #e9ecef;
            color: #495057;
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            font-size: 0.8rem;
            margin: 0.125rem;
        }

        .hours-preview {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
        }

        .day-hours {
            display: flex;
            justify-content: space-between;
            padding: 0.5rem 0;
            border-bottom: 1px solid #dee2e6;
        }

        .day-hours:last-child {
            border-bottom: none;
        }

        .progress-section {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 1.5rem;
            margin: 1rem 0;
        }
    </style>
</head>
<body>
    <header class="admin-header">
        <div class="nav-container">
            <a href="/admin/dashboard" class="nav-brand">
                <i class="fas fa-store"></i> Business Directory Admin
            </a>
            
            <nav class="nav-links">
                <a href="/admin/dashboard">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="/admin/stores">
                    <i class="fas fa-building"></i> Stores
                </a>
                <a href="/admin/hours">
                    <i class="fas fa-clock"></i> Hours
                </a>
                <a href="/admin/hours-import" class="active">
                    <i class="fas fa-download"></i> Import Hours
                </a>
                <a href="/admin/categories">
                    <i class="fas fa-tags"></i> Categories
                </a>
            </nav>
            
            <div>
                <a href="/admin/logout" class="btn btn-outline-danger btn-sm">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </header>

    <main class="main-container">
        <!-- Page Header -->
        <div class="page-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="h2 mb-2">
                        <i class="fas fa-download"></i> Business Hours Import Tool
                    </h1>
                    <p class="text-muted mb-0">Import accurate business hours from Google Places API with smart merge capabilities</p>
                </div>
                <div>
                    <button class="btn btn-success me-2" id="bulkImportBtn">
                        <i class="fas fa-download"></i> Bulk Import
                    </button>
                    <a href="/admin/hours" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left"></i> Back to Hours
                    </a>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="stats-grid">
            <div class="stat-card danger">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-uppercase text-muted mb-2">Need Hours Setup</h6>
                        <h2 class="mb-0" id="needs-hours-count">127</h2>
                        <small class="text-muted">No hours configured</small>
                    </div>
                    <div>
                        <i class="fas fa-clock fa-2x text-muted opacity-50"></i>
                    </div>
                </div>
            </div>

            <div class="stat-card test-data">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-uppercase text-muted mb-2">Test Data Detected</h6>
                        <h2 class="mb-0" id="test-data-count">298</h2>
                        <small class="text-muted">Sample/test patterns found</small>
                    </div>
                    <div>
                        <i class="fas fa-bug fa-2x text-muted opacity-50"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card warning">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-uppercase text-muted mb-2">Partial Hours</h6>
                        <h2 class="mb-0" id="partial-hours-count">35</h2>
                        <small class="text-muted">Missing main hours</small>
                    </div>
                    <div>
                        <i class="fas fa-exclamation-triangle fa-2x text-muted opacity-50"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card success">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-uppercase text-muted mb-2">Complete Hours</h6>
                        <h2 class="mb-0" id="complete-hours-count">11</h2>
                        <small class="text-muted">Real hours configured</small>
                    </div>
                    <div>
                        <i class="fas fa-check-circle fa-2x text-muted opacity-50"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card info">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="text-uppercase text-muted mb-2">Total Businesses</h6>
                        <h2 class="mb-0" id="total-count">471</h2>
                        <small class="text-muted">In directory</small>
                    </div>
                    <div>
                        <i class="fas fa-building fa-2x text-muted opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Flash Messages -->
        <div id="flash-messages"></div>

        <!-- Filter and Actions Bar -->
        <div class="businesses-section">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h4>Business Hours Status</h4>
                    <p class="text-muted mb-0">Review and import hours for businesses</p>
                </div>
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-outline-primary" data-filter="needs-hours">
                        <i class="fas fa-clock"></i> Need Setup (<span id="filter-needs-count">127</span>)
                    </button>
                    <button type="button" class="btn btn-outline-warning" data-filter="test-data">
                        <i class="fas fa-bug"></i> Test Data (<span id="filter-test-count">298</span>)
                    </button>
                    <button type="button" class="btn btn-outline-secondary" data-filter="partial">
                        <i class="fas fa-exclamation-triangle"></i> Partial (<span id="filter-partial-count">35</span>)
                    </button>
                    <button type="button" class="btn btn-outline-success" data-filter="complete">
                        <i class="fas fa-check"></i> Complete (<span id="filter-complete-count">11</span>)
                    </button>
                    <button type="button" class="btn btn-outline-dark" data-filter="all">
                        <i class="fas fa-list"></i> All
                    </button>
                </div>
            </div>

            <!-- Bulk Actions -->
            <div class="alert alert-info" id="bulk-controls" style="display: none;">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <strong><span id="selected-count">0</span></strong> businesses selected
                        <span class="text-muted ms-2">Use bulk actions to process multiple businesses at once</span>
                    </div>
                    <div>
                        <button class="btn btn-primary btn-sm me-2" id="bulkImportSelected">
                            <i class="fas fa-download"></i> Import Selected
                        </button>
                        <button class="btn btn-outline-secondary btn-sm" id="clearSelection">
                            <i class="fas fa-times"></i> Clear
                        </button>
                    </div>
                </div>
            </div>

            <!-- Business Table -->
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead class="table-dark">
                        <tr>
                            <th width="50">
                                <input type="checkbox" id="selectAllBusinesses" class="form-check-input">
                            </th>
                            <th>Business Name</th>
                            <th>Location</th>
                            <th>Current Hours Status</th>
                            <th>Services</th>
                            <th width="200">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="businessTableBody">
                        <!-- Businesses will be loaded here via JavaScript -->
                    </tbody>
                </table>
            </div>

            <!-- Loading State -->
            <div id="loadingState" class="text-center py-5">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading businesses...</span>
                </div>
                <p class="mt-2">Loading business data...</p>
            </div>

            <!-- No Results State -->
            <div id="noResultsState" class="text-center py-5" style="display: none;">
                <i class="fas fa-search fa-3x text-muted opacity-50 mb-3"></i>
                <h5>No businesses found</h5>
                <p class="text-muted">Try adjusting your filter or check back later.</p>
            </div>
        </div>
    </main>

    <!-- Bulk Import Modal -->
    <div class="modal fade" id="bulkImportModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-download"></i> Bulk Import Hours
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <h6 class="alert-heading">Bulk Import Options</h6>
                        <ul class="mb-0 mt-2">
                            <li><strong>Auto-match:</strong> Automatically import hours from the first Google Places match</li>
                            <li><strong>Replace existing:</strong> Overwrite existing hours (otherwise skip businesses with hours)</li>
                            <li><strong>Confidence threshold:</strong> Only import matches above the specified confidence level</li>
                        </ul>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-check mb-3">
                                <input type="checkbox" class="form-check-input" id="autoMatchCheckbox" checked>
                                <label class="form-check-label" for="autoMatchCheckbox">
                                    <strong>Auto-match and import</strong><br>
                                    <small class="text-muted">Recommended for well-known businesses</small>
                                </label>
                            </div>
                            
                            <div class="form-check mb-3">
                                <input type="checkbox" class="form-check-input" id="replaceExistingBulk">
                                <label class="form-check-label" for="replaceExistingBulk">
                                    <strong>Replace existing hours</strong><br>
                                    <small class="text-muted">Overwrite any existing data</small>
                                </label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="confidenceThreshold" class="form-label">Confidence Threshold</label>
                                <select class="form-select" id="confidenceThreshold">
                                    <option value="50">50% - Import most matches</option>
                                    <option value="70" selected>70% - Good matches only</option>
                                    <option value="85">85% - High confidence only</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="bulkStrategy" class="form-label">Import Strategy</label>
                                <select class="form-select" id="bulkStrategy">
                                    <option value="smart_merge" selected>Smart Merge</option>
                                    <option value="replace_all">Replace All</option>
                                    <option value="merge_keep_additional">Update Main Hours Only</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Bulk Progress -->
                    <div id="bulkProgress" class="progress-section" style="display: none;">
                        <div class="progress mb-3">
                            <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                        </div>
                        <div id="bulkStatus">Preparing bulk import...</div>
                        <div id="bulkDetails" class="small text-muted mt-2"></div>
                    </div>
                    
                    <!-- Bulk Results -->
                    <div id="bulkResults" style="display: none;">
                        <h6>Import Results:</h6>
                        <div id="bulkSummary" class="alert"></div>
                        <div id="bulkResultsList" class="border rounded p-3" style="max-height: 300px; overflow-y: auto;"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="startBulkImport">
                        <i class="fas fa-play"></i> Start Bulk Import (<span id="bulkImportCount">0</span> businesses)
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    // Global variables
    let currentStoreId = null;
    let selectedPlaceId = null;
    let selectedPlaceData = null;
    let currentFilter = 'needs-hours';
    let businessData = {};
    let selectedBusinesses = [];

    document.addEventListener('DOMContentLoaded', function() {
        initializeInterface();
        loadBusinessData();
        setupEventHandlers();
    });

    function initializeInterface() {
        // Set initial filter
        document.querySelector('[data-filter="needs-hours"]').classList.add('active');
    }

    function setupEventHandlers() {
        // Filter buttons
        document.querySelectorAll('[data-filter]').forEach(btn => {
            btn.addEventListener('click', function() {
                document.querySelectorAll('[data-filter]').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                currentFilter = this.dataset.filter;
                renderBusinessTable();
            });
        });

        // Select all checkbox
        document.getElementById('selectAllBusinesses').addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.business-checkbox');
            checkboxes.forEach(cb => {
                cb.checked = this.checked;
                updateBusinessSelection(cb.value, this.checked);
            });
            updateBulkControls();
        });

        // Bulk actions
        document.getElementById('bulkImportBtn').addEventListener('click', showBulkImportModal);
        document.getElementById('bulkImportSelected').addEventListener('click', showBulkImportModal);
        document.getElementById('startBulkImport').addEventListener('click', startBulkImport);
        document.getElementById('clearSelection').addEventListener('click', clearSelection);
    }

    async function loadBusinessData() {
        try {
            const response = await fetch('/admin/hours-import/get-business-data');
            const data = await response.json();
            
            if (data.success) {
                businessData = data.data;
                updateStatistics();
                renderBusinessTable();
            } else {
                showError('Failed to load business data: ' + data.message);
            }
        } catch (error) {
            console.error('Error loading business data:', error);
            showError('Failed to load business data');
        } finally {
            document.getElementById('loadingState').style.display = 'none';
        }
    }

    function updateStatistics() {
        document.getElementById('needs-hours-count').textContent = businessData.needsHours?.length || 0;
        document.getElementById('test-data-count').textContent = businessData.hasTestData?.length || 0;
        document.getElementById('partial-hours-count').textContent = businessData.hasPartialHours?.length || 0;
        document.getElementById('complete-hours-count').textContent = businessData.hasCompleteHours?.length || 0;
        document.getElementById('total-count').textContent = businessData.totalBusinesses || 0;
        
        // Update filter counts
        document.getElementById('filter-needs-count').textContent = businessData.needsHours?.length || 0;
        document.getElementById('filter-test-count').textContent = businessData.hasTestData?.length || 0;
        document.getElementById('filter-partial-count').textContent = businessData.hasPartialHours?.length || 0;
        document.getElementById('filter-complete-count').textContent = businessData.hasCompleteHours?.length || 0;
    }

    function renderBusinessTable() {
        const tbody = document.getElementById('businessTableBody');
        const noResults = document.getElementById('noResultsState');
        
        let businesses = [];
        
        switch (currentFilter) {
            case 'needs-hours':
                businesses = businessData.needsHours || [];
                break;
            case 'test-data':
                businesses = businessData.hasTestData || [];
                break;
            case 'partial':
                businesses = businessData.hasPartialHours || [];
                break;
            case 'complete':
                businesses = businessData.hasCompleteHours || [];
                break;
            case 'all':
                businesses = [
                    ...(businessData.needsHours || []),
                    ...(businessData.hasTestData || []),
                    ...(businessData.hasPartialHours || []),
                    ...(businessData.hasCompleteHours || [])
                ];
                break;
        }

        if (businesses.length === 0) {
            tbody.innerHTML = '';
            noResults.style.display = 'block';
            return;
        }

        noResults.style.display = 'none';
        
        tbody.innerHTML = businesses.map(business => `
            <tr>
                <td>
                    <input type="checkbox" class="business-checkbox form-check-input" 
                           value="${business.store_id}" 
                           ${selectedBusinesses.includes(business.store_id) ? 'checked' : ''}
                           onchange="updateBusinessSelection('${business.store_id}', this.checked)">
                </td>
                <td>
                    <div class="fw-bold">${escapeHtml(business.name)}</div>
                    <small class="text-muted">ID: ${business.store_id}</small>
                    ${business.test_data_detected ? '<span class="test-data-badge ms-2">Test Data</span>' : ''}
                </td>
                <td>
                    <div>${escapeHtml(business.address)}</div>
                    <small class="text-muted">${escapeHtml(business.city)}, ${escapeHtml(business.state)} ${business.postal_code || ''}</small>
                </td>
                <td>
                    ${getStatusBadge(business)}
                </td>
                <td>
                    ${getServiceTypes(business)}
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-primary search-hours-btn"
                                onclick="searchBusinessHours('${business.store_id}', '${escapeHtml(business.name)}', '${escapeHtml(business.address)}', '${escapeHtml(business.city)}', '${escapeHtml(business.state)}')">
                            <i class="fas fa-search"></i> Search
                        </button>
                        <a href="/admin/hours/store/${business.store_id}" class="btn btn-outline-secondary">
                            <i class="fas fa-edit"></i>
                        </a>
                    </div>
                </td>
            </tr>
        `).join('');
    }

    function getStatusBadge(business) {
        if (business.hour_types_count == 0) {
            return '<span class="badge bg-danger">No Hours Set</span>';
        } else if (business.test_data_detected) {
            return '<span class="badge bg-warning text-dark">Test Data Detected</span>';
        } else if (business.main_hour_types_count == 0) {
            return '<span class="badge bg-warning text-dark">Missing Main Hours</span>';
        } else {
            return '<span class="badge bg-success">Complete</span>';
        }
    }

    function getServiceTypes(business) {
        if (!business.hour_types_list) {
            return '<span class="text-muted">None</span>';
        }
        
        const types = business.hour_types_list.split(', ');
        return types.map(type => `<span class="service-type-tag">${escapeHtml(type)}</span>`).join('');
    }

    function updateBusinessSelection(storeId, isSelected) {
        if (isSelected) {
            if (!selectedBusinesses.includes(storeId)) {
                selectedBusinesses.push(storeId);
            }
        } else {
            selectedBusinesses = selectedBusinesses.filter(id => id !== storeId);
        }
        updateBulkControls();
    }

    function updateBulkControls() {
        const bulkControls = document.getElementById('bulk-controls');
        const selectedCount = selectedBusinesses.length;
        
        document.getElementById('selected-count').textContent = selectedCount;
        document.getElementById('bulkImportCount').textContent = selectedCount;
        
        bulkControls.style.display = selectedCount > 0 ? 'block' : 'none';
        document.getElementById('bulkImportSelected').disabled = selectedCount === 0;
    }

    function clearSelection() {
        selectedBusinesses = [];
        document.querySelectorAll('.business-checkbox').forEach(cb => cb.checked = false);
        document.getElementById('selectAllBusinesses').checked = false;
        updateBulkControls();
    }

    function showBulkImportModal() {
        const count = selectedBusinesses.length;
        if (count === 0) {
            showError('Please select businesses to import hours for.');
            return;
        }
        
        document.getElementById('bulkImportCount').textContent = count;
        new bootstrap.Modal(document.getElementById('bulkImportModal')).show();
    }

    async function startBulkImport() {
        const storeIds = selectedBusinesses.slice(); // Copy array
        const autoMatch = document.getElementById('autoMatchCheckbox').checked;
        const replaceExisting = document.getElementById('replaceExistingBulk').checked;
        const confidenceThreshold = document.getElementById('confidenceThreshold').value;
        const strategy = document.getElementById('bulkStrategy').value;
        
        console.log('Starting bulk import with:', {
            storeIds: storeIds,
            autoMatch: autoMatch,
            replaceExisting: replaceExisting,
            confidenceThreshold: confidenceThreshold,
            strategy: strategy
        });
        
        if (storeIds.length === 0) {
            showError('No businesses selected for import');
            return;
        }
        
        // Show progress and disable button
        document.getElementById('bulkProgress').style.display = 'block';
        document.getElementById('bulkResults').style.display = 'none';
        document.getElementById('startBulkImport').disabled = true;
        
        // Update status to show we're starting
        updateBulkProgress(0, storeIds.length, 'Starting bulk import...');
        
        try {
            // Create FormData object for proper POST submission
            const formData = new FormData();
            
            // Add all parameters
            formData.append('auto_match', autoMatch ? '1' : '0');
            formData.append('replace_existing', replaceExisting ? '1' : '0');
            formData.append('confidence_threshold', confidenceThreshold);
            formData.append('merge_strategy', strategy);
            
            // Add store IDs as array
            storeIds.forEach(id => {
                formData.append('store_ids[]', id);
            });
            
            console.log('Sending bulk import request...');
            
            // Make the POST request
            const response = await fetch('/admin/hours-import/bulk-import', {
                method: 'POST',
                body: formData // Use FormData instead of URLSearchParams
            });
            
            console.log('Response status:', response.status);
            
            // Check if response is ok
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const responseText = await response.text();
            console.log('Raw response:', responseText);
            
            let data;
            try {
                data = JSON.parse(responseText);
            } catch (parseError) {
                console.error('JSON parse error:', parseError);
                throw new Error('Invalid JSON response: ' + responseText.substring(0, 200));
            }
            
            console.log('Parsed response data:', data);
            
            if (data.success) {
                displayBulkResults(data.results, data.summary);
                clearSelection();
                loadBusinessData(); // Refresh data
                showSuccess('Bulk import completed successfully!');
            } else {
                if (data.debug_info) {
                    console.error('Server debug info:', data.debug_info);
                }
                showError('Bulk import failed: ' + (data.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Bulk import request failed:', error);
            showError('Bulk import request failed: ' + error.message);
        } finally {
            document.getElementById('bulkProgress').style.display = 'none';
            document.getElementById('startBulkImport').disabled = false;
        }
    }

    function updateBulkProgress(current, total, message) {
        const progress = Math.min((current / total) * 100, 100);
        document.querySelector('#bulkProgress .progress-bar').style.width = progress + '%';
        document.getElementById('bulkStatus').textContent = message;
        document.getElementById('bulkDetails').textContent = `${current} of ${total} businesses processed`;
    }

    function displayBulkResults(results, summary) {
        const summaryClass = summary.successful > summary.failed ? 'alert-success' : 'alert-warning';
        
        document.getElementById('bulkSummary').className = `alert ${summaryClass}`;
        document.getElementById('bulkSummary').innerHTML = `
            <strong>Import Complete:</strong> 
            ${summary.successful} successful, ${summary.failed} failed out of ${summary.total_processed} processed
        `;
        
        const resultsHtml = results.map(result => {
            const isSuccess = result.includes('successfully');
            const icon = isSuccess ? 'check text-success' : 'times text-danger';
            return `<div class="mb-1"><i class="fas fa-${icon} me-2"></i>${escapeHtml(result)}</div>`;
        }).join('');
        
        document.getElementById('bulkResultsList').innerHTML = resultsHtml;
        document.getElementById('bulkResults').style.display = 'block';
    }

    function showSuccess(message) {
        showFlashMessage(message, 'success');
    }

    function showError(message) {
        showFlashMessage(message, 'danger');
    }

    function showFlashMessage(message, type) {
        const flash = document.getElementById('flash-messages');
        flash.innerHTML = `
            <div class="alert alert-${type} alert-dismissible fade show">
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i> ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;
    }

    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Stub functions for search functionality (to prevent errors)
    function searchBusinessHours(storeId, name, address, city, state) {
        showError('Individual search feature temporarily disabled. Please use bulk import.');
    }
    </script>
</body>
</html>