<?php

namespace App\Controllers;

use App\Controllers\BaseController;

class VerificationTool extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    /**
     * Display the verification tool interface
     */
    public function index()
    {
        return view('verification_tool');
    }

    /**
     * Get business data for verification (updated for digital verification)
     */
    public function getData()
    {
        try {
            $json = $this->request->getJSON(true);
            
            if (!isset($json['coords']) || !isset($json['coords']['lat']) || !isset($json['coords']['lng'])) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Invalid coordinates provided'
                ]);
            }

            $lat = (float) $json['coords']['lat'];
            $lng = (float) $json['coords']['lng'];

            // Get all stores with verification data using your existing schema
            $query = $this->db->query("
                SELECT 
                    s.id,
                    s.store_id,
                    s.name,
                    s.address,
                    s.city,
                    s.state,
                    s.postal_code,
                    s.latitude,
                    s.longitude,
                    s.last_verified_at,
                    s.verification_count,
                    COALESCE(
                        (SELECT admin_notes 
                         FROM business_verifications 
                         WHERE store_id = s.id 
                         ORDER BY verified_at DESC 
                         LIMIT 1), 
                        ''
                    ) as admin_notes,
                    COALESCE(
                        (SELECT verification_status 
                         FROM business_verifications 
                         WHERE store_id = s.id 
                         ORDER BY verified_at DESC 
                         LIMIT 1), 
                        'not_verified'
                    ) as verification_status,
                    -- Calculate distance using Haversine formula
                    (
                        3959 * acos(
                            cos(radians(?)) * 
                            cos(radians(s.latitude)) * 
                            cos(radians(s.longitude) - radians(?)) + 
                            sin(radians(?)) * 
                            sin(radians(s.latitude))
                        )
                    ) AS distance_miles
                FROM store s
                WHERE s.latitude IS NOT NULL 
                  AND s.longitude IS NOT NULL
                  AND s.status = 1
                ORDER BY distance_miles ASC
            ", [$lat, $lng, $lat]);

            $stores = $query->getResultArray();
            
            // Format the data for the frontend
            $businesses = [];
            foreach ($stores as $store) {
                // Get current open/closed status using your existing method
                $statusData = $this->getBusinessStatusFromCRON($store['id']);
                $currentStatus = $this->formatStatusForVerification($statusData);
                
                // Format verification date
                $lastVerified = 'Never';
                if ($store['last_verified_at']) {
                    $lastVerified = $store['last_verified_at'];
                }
                
                $businesses[] = [
                    'id' => (int) $store['id'],
                    'name' => $store['name'],
                    'streetAddress' => $store['address'],
                    'city' => $store['city'],
                    'state' => $store['state'],
                    'zipCode' => $store['postal_code'],
                    'fullAddress' => trim($store['address'] . ', ' . $store['city'] . ' ' . $store['state'] . ' ' . $store['postal_code']),
                    'distance' => round((float) $store['distance_miles'], 1),
                    'status' => $currentStatus,
                    'lastVerified' => $lastVerified,
                    'adminNotes' => $store['admin_notes'] ?? '',
                    'verificationStatus' => $store['verification_status']
                ];
            }

            return $this->response->setJSON([
                'success' => true,
                'businesses' => $businesses,
                'total_count' => count($businesses)
            ]);

        } catch (\Exception $e) {
            log_message('error', 'VerificationTool getData error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error loading business data: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Verify a business with timestamp and notes
     */
    public function verify()
    {
        try {
            $json = $this->request->getJSON(true);
            
            if (!isset($json['store_id']) || !isset($json['verification_status'])) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Missing required fields: store_id and verification_status'
                ]);
            }

            $storeId = (int) $json['store_id'];
            $verificationStatus = $json['verification_status'];
            $adminNotes = $json['admin_notes'] ?? '';
            $verifiedBy = $json['verified_by'] ?? 'Admin';

            // Validate verification status
            $validStatuses = ['verified', 'not_found', 'relocated', 'closed_permanently'];
            if (!in_array($verificationStatus, $validStatuses)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Invalid verification status'
                ]);
            }

            // Check if store exists
            $store = $this->db->query("SELECT id FROM store WHERE id = ?", [$storeId])->getRow();
            if (!$store) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Store not found'
                ]);
            }

            $db = $this->db;
            
            // Insert verification record
            $verificationData = [
                'store_id' => $storeId,
                'verified_at' => date('Y-m-d H:i:s'),
                'verified_by' => $verifiedBy,
                'verification_status' => $verificationStatus,
                'admin_notes' => $adminNotes
            ];

            $db->table('business_verifications')->insert($verificationData);

            // Update store's last verified timestamp and count
            $currentCount = $db->query("SELECT verification_count FROM store WHERE id = ?", [$storeId])->getRow();
            $newCount = ($currentCount->verification_count ?? 0) + 1;
            
            $updateData = [
                'last_verified_at' => date('Y-m-d H:i:s'),
                'verification_count' => $newCount
            ];

            $db->table('store')->where('id', $storeId)->update($updateData);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Verification saved successfully',
                'verification_id' => $db->insertID()
            ]);

        } catch (\Exception $e) {
            log_message('error', 'VerificationTool verify error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error saving verification: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Save admin notes for multiple businesses
     */
    public function saveNotes()
    {
        try {
            $json = $this->request->getJSON(true);
            
            if (!isset($json['notes']) || !is_array($json['notes'])) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Invalid notes data provided'
                ]);
            }

            $db = $this->db;
            $savedCount = 0;

            foreach ($json['notes'] as $note) {
                if (!isset($note['business_id']) || !isset($note['notes'])) {
                    continue;
                }

                $businessId = (int) $note['business_id'];
                $noteText = trim($note['notes']);

                if (empty($noteText)) {
                    continue;
                }

                // Check if store exists
                $store = $this->db->query("SELECT id FROM store WHERE id = ?", [$businessId])->getRow();
                if (!$store) {
                    continue;
                }

                // Check if there's already a verification record for today
                $today = date('Y-m-d');
                $existingVerification = $db->query(
                    "SELECT id FROM business_verifications 
                     WHERE store_id = ? AND DATE(verified_at) = ? 
                     ORDER BY verified_at DESC LIMIT 1",
                    [$businessId, $today]
                )->getRow();

                if ($existingVerification) {
                    // Update existing verification with notes
                    $db->table('business_verifications')
                       ->where('id', $existingVerification->id)
                       ->update(['admin_notes' => $noteText]);
                } else {
                    // Create new verification record for notes only
                    $verificationData = [
                        'store_id' => $businessId,
                        'verified_at' => date('Y-m-d H:i:s'),
                        'verified_by' => 'Admin',
                        'verification_status' => 'verified',
                        'admin_notes' => $noteText
                    ];

                    $db->table('business_verifications')->insert($verificationData);

                    // Update store's last verified timestamp
                    $this->db->table('store')->where('id', $businessId)->update([
                        'last_verified_at' => date('Y-m-d H:i:s')
                    ]);
                }

                $savedCount++;
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => "Notes saved successfully",
                'saved_count' => $savedCount
            ]);

        } catch (\Exception $e) {
            log_message('error', 'VerificationTool saveNotes error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error saving notes: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Add a new business discovery
     */
    public function addBusiness()
    {
        try {
            $json = $this->request->getJSON(true);
            
            // Validate required fields
            $requiredFields = ['business_name', 'street_address', 'city', 'state', 'zip_code'];
            foreach ($requiredFields as $field) {
                if (!isset($json[$field]) || empty(trim($json[$field]))) {
                    return $this->response->setJSON([
                        'success' => false,
                        'message' => "Missing required field: $field"
                    ]);
                }
            }

            $db = $this->db;

            // Prepare new business discovery data
            $discoveryData = [
                'business_name' => trim($json['business_name']),
                'street_address' => trim($json['street_address']),
                'city' => trim($json['city']),
                'state' => strtoupper(trim($json['state'])),
                'zip_code' => trim($json['zip_code']),
                'phone' => isset($json['phone']) ? trim($json['phone']) : null,
                'business_type' => isset($json['business_type']) ? trim($json['business_type']) : null,
                'latitude' => isset($json['latitude']) ? (float) $json['latitude'] : null,
                'longitude' => isset($json['longitude']) ? (float) $json['longitude'] : null,
                'discovered_by' => $json['discovered_by'] ?? 'Admin',
                'discovered_at' => date('Y-m-d H:i:s'),
                'admin_notes' => isset($json['admin_notes']) ? trim($json['admin_notes']) : null,
                'status' => 'pending'
            ];

            // Check for potential duplicates
            $existingBusiness = $db->query(
                "SELECT id FROM store 
                 WHERE LOWER(name) LIKE ? 
                   AND LOWER(address) LIKE ? 
                   AND LOWER(city) = ? 
                   AND state = ?
                 LIMIT 1",
                [
                    '%' . strtolower($discoveryData['business_name']) . '%',
                    '%' . strtolower($discoveryData['street_address']) . '%',
                    strtolower($discoveryData['city']),
                    $discoveryData['state']
                ]
            )->getRow();

            if ($existingBusiness) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'A similar business already exists in the system'
                ]);
            }

            // Insert into new business discoveries table
            $db->table('new_business_discoveries')->insert($discoveryData);
            $discoveryId = $db->insertID();

            // Immediately add to main store table (as per requirement)
            $storeData = [
                'name' => $discoveryData['business_name'],
                'address' => $discoveryData['street_address'],
                'city' => $discoveryData['city'],
                'state' => $discoveryData['state'],
                'postal_code' => $discoveryData['zip_code'],
                'phone' => $discoveryData['phone'],
                'latitude' => $discoveryData['latitude'],
                'longitude' => $discoveryData['longitude'],
                'active' => 1,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            // Add to store table
            $this->db->table('store')->insert($storeData);
            $newStoreId = $this->db->insertID();

            // Update discovery record with the new store ID
            $db->table('new_business_discoveries')
               ->where('id', $discoveryId)
               ->update([
                   'store_id' => $newStoreId,
                   'status' => 'added_to_system',
                   'processed_at' => date('Y-m-d H:i:s'),
                   'processed_by' => 'Admin'
               ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'New business added successfully',
                'store_id' => $newStoreId,
                'discovery_id' => $discoveryId
            ]);

        } catch (\Exception $e) {
            log_message('error', 'VerificationTool addBusiness error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error adding business: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get verification statistics (for potential admin dashboard)
     */
    public function getStats()
    {
        try {
            $db = $this->db;

            // Get verification statistics
            $stats = [
                'total_businesses' => $db->table('store')->where('active', 1)->countAllResults(),
                'verified_businesses' => $db->table('store')->where('active', 1)->where('last_verified_at IS NOT NULL')->countAllResults(),
                'never_verified' => $db->table('store')->where('active', 1)->where('last_verified_at IS NULL')->countAllResults(),
                'verified_today' => $db->query("SELECT COUNT(DISTINCT store_id) as count FROM business_verifications WHERE DATE(verified_at) = CURDATE()")->getRow()->count,
                'verified_this_week' => $db->query("SELECT COUNT(DISTINCT store_id) as count FROM business_verifications WHERE verified_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK)")->getRow()->count,
                'new_discoveries_pending' => $db->table('new_business_discoveries')->where('status', 'pending')->countAllResults(),
                'new_discoveries_total' => $db->table('new_business_discoveries')->countAllResults()
            ];

            // Get recent verifications
            $recentVerifications = $db->query("
                SELECT 
                    s.name,
                    bv.verification_status,
                    bv.verified_at,
                    bv.verified_by
                FROM business_verifications bv
                JOIN store s ON bv.store_id = s.id
                ORDER BY bv.verified_at DESC
                LIMIT 10
            ")->getResultArray();

            return $this->response->setJSON([
                'success' => true,
                'stats' => $stats,
                'recent_verifications' => $recentVerifications
            ]);

        } catch (\Exception $e) {
            log_message('error', 'VerificationTool getStats error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error loading statistics: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Check password for verification tool access
     */
    public function checkPassword()
    {
        try {
            $json = $this->request->getJSON(true);
            $password = $json['password'] ?? '';
            
            // Simple password check - you can make this more secure
            $correctPassword = 'verify2025'; // Change this to your preferred password
            
            if ($password === $correctPassword) {
                // Set a session to remember authentication
                session()->set('verification_tool_authenticated', true);
                
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Access granted'
                ]);
            } else {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Incorrect password'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Password check error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error checking password'
            ]);
        }
    }

    /**
     * Mark a business as inactive (hide from public search)
     */
    public function markInactive()
    {
        try {
            // Check authentication
            if (!session()->get('verification_tool_authenticated')) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Not authenticated'
                ]);
            }

            $json = $this->request->getJSON(true);
            
            if (!isset($json['store_id'])) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Missing store ID'
                ]);
            }

            $storeId = (int) $json['store_id'];
            $reason = $json['reason'] ?? 'No reason provided';
            $markedBy = $json['marked_by'] ?? 'Admin';

            // Check if store exists
            $store = $this->db->query("SELECT id, name FROM store WHERE id = ?", [$storeId])->getRow();
            if (!$store) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Store not found'
                ]);
            }

            // Mark store as inactive
            $this->db->table('store')->where('id', $storeId)->update([
                'status' => 0,
                'update_date' => date('Y-m-d H:i:s')
            ]);

            // Log the verification that led to this action
            $verificationData = [
                'store_id' => $storeId,
                'verified_at' => date('Y-m-d H:i:s'),
                'verified_by' => $markedBy,
                'verification_status' => 'closed_permanently',
                'admin_notes' => "MARKED INACTIVE: $reason"
            ];

            $this->db->table('business_verifications')->insert($verificationData);

            // Log the action
            log_message('info', "Business marked inactive: {$store->name} (ID: $storeId) by $markedBy. Reason: $reason");

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Business marked as inactive successfully'
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Mark inactive error: ' . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error marking business inactive: ' . $e->getMessage()
            ]);
        }
    }
    private function getBusinessStatusFromCRON($storeId)
    {
        $statusRecords = $this->db->query("
            SELECT service_type, status, current_message, closes_at, next_open_datetime
            FROM store_current_status 
            WHERE store_id = ?
            ORDER BY service_type
        ", [$storeId])->getResultArray();

        $openServices = [];
        $anyOpen = false;

        foreach ($statusRecords as $record) {
            if (in_array($record['status'], ['open', 'closing_soon'])) {
                $openServices[] = [
                    'name' => $record['service_type'],
                    'status' => $record['status']
                ];
                $anyOpen = true;
            }
        }

        return [
            'any_open' => $anyOpen,
            'open_services' => $openServices
        ];
    }

    /**
     * Format status for verification display (from your existing code)
     */
    private function formatStatusForVerification($statusData)
    {
        if (!$statusData['any_open']) {
            return 'CLOSED';
        }

        $serviceNames = array_column($statusData['open_services'], 'name');
        return 'OPEN (' . implode(', ', $serviceNames) . ')';
    }
}