<?php

namespace App\Controllers\Frontend\Ajax;

header("Cache-Control: no-cache, must-revalidate");

use App\Controllers\BaseController;
use App\Models\StoreModel;

class Locator extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    public function index()
    {
        try {
            $data = json_decode($this->request->getBody(), true);
            
            $page = max((int)($data['page'] ?? 1), 1);
            $keyword = trim($data['keyword'] ?? '');
            $category = $data['category'] ?? '';
            $coords = $data['coords'] ?? null;
            $openOnly = $data['open_only'] ?? false;

            $limit = 12;
            $offset = max(0, ($page - 1) * $limit);

            // Log every request to track the problem
            log_message('debug', "=== LOCATOR REQUEST DEBUG ===");
            log_message('debug', "Page: $page");
            log_message('debug', "Open Only: " . ($openOnly ? 'TRUE' : 'FALSE'));
            log_message('debug', "Category: '$category'");
            log_message('debug', "Keyword: '$keyword'");
            log_message('debug', "Has Coordinates: " . (isset($coords['lat'], $coords['lng']) ? 'YES' : 'NO'));

            // Check if we have valid coordinates
            $hasCoords = isset($coords['lat'], $coords['lng']) 
                        && is_numeric($coords['lat']) 
                        && is_numeric($coords['lng'])
                        && $coords['lat'] != 0 
                        && $coords['lng'] != 0;
            
            $lat = $hasCoords ? floatval($coords['lat']) : null;
            $lng = $hasCoords ? floatval($coords['lng']) : null;

            log_message('debug', "Valid Coordinates: " . ($hasCoords ? "YES ($lat, $lng)" : 'NO'));

            // ALWAYS calculate distance for consistent "results near me" sorting
            $distanceCalc = $hasCoords ? 
                "(3959 * acos(cos(radians($lat)) * cos(radians(s.latitude)) * cos(radians(s.longitude) - radians($lng)) + sin(radians($lat)) * sin(radians(s.latitude))))" : 
                "999999"; // Large number for non-coordinate sorts to put them at bottom
            
            $distanceSelect = $hasCoords ? 
                "$distanceCalc AS distance" : 
                "NULL as distance";

            // Build category JOIN and WHERE conditions
            $categoryJoin = '';
            $categoryWhere = '';
            if (!empty($category)) {
                $categoryJoin = "INNER JOIN store_category sc ON s.store_id = sc.store_id 
                                INNER JOIN category c ON sc.category_id = c.category_id";
                $categoryWhere = "AND c.category_id = " . $this->db->escape($category) . " AND c.status = 1";
                log_message('debug', "Category filtering enabled for: $category");
            }

            // Build keyword search conditions
            $keywordWhere = '';
            if (!empty($keyword)) {
                $escapedKeyword = $this->db->escapeLikeString($keyword);
                $keywordWhere = "AND (
                    s.name LIKE '%{$escapedKeyword}%' OR 
                    s.description LIKE '%{$escapedKeyword}%' OR 
                    s.address LIKE '%{$escapedKeyword}%' OR 
                    s.city LIKE '%{$escapedKeyword}%' OR 
                    s.state LIKE '%{$escapedKeyword}%' OR 
                    s.postal_code LIKE '%{$escapedKeyword}%' OR 
                    s.phone LIKE '%{$escapedKeyword}%' OR 
                    s.website LIKE '%{$escapedKeyword}%'
                )";
                log_message('debug', "Keyword search enabled for: '$keyword'");
            }

            // Build the SQL query - ALWAYS sort by distance when available
            if ($openOnly) {
                // ONLY show open businesses, sorted by distance
                $sql = "
                    SELECT DISTINCT s.store_id, s.name, s.description, s.address, s.city, s.state, s.postal_code, 
                           s.latitude, s.longitude, s.website, s.phone, s.image, s.image_type, s.status,
                           $distanceSelect
                    FROM store s
                    INNER JOIN store_current_status scs ON s.store_id = scs.store_id
                    $categoryJoin
                    WHERE s.status = 1 
                    AND scs.status IN ('open', 'closing_soon')
                    $categoryWhere
                    $keywordWhere
                    " . ($hasCoords ? "AND $distanceCalc <= 8000" : "") . "
                    ORDER BY " . ($hasCoords ? "distance ASC" : "s.name ASC") . "
                    LIMIT $limit OFFSET $offset
                ";
            } else {
                // Show ALL businesses, sorted by distance  
                $sql = "
                    SELECT DISTINCT s.store_id, s.name, s.description, s.address, s.city, s.state, s.postal_code, 
                           s.latitude, s.longitude, s.website, s.phone, s.image, s.image_type, s.status,
                           $distanceSelect
                    FROM store s
                    $categoryJoin
                    WHERE s.status = 1
                    $categoryWhere
                    $keywordWhere
                    " . ($hasCoords ? "AND $distanceCalc <= 8000" : "") . "
                    ORDER BY " . ($hasCoords ? "distance ASC" : "s.name ASC") . "
                    LIMIT $limit OFFSET $offset
                ";
            }

            log_message('debug', "Final SQL Query:");
            log_message('debug', $sql);

            $stores = $this->db->query($sql)->getResultArray();

            log_message('debug', "Raw query returned: " . count($stores) . " stores");
            
            // If we're filtering for open only, let's verify each store's status
            if ($openOnly) {
                $verifiedStores = [];
                foreach ($stores as $store) {
                    $openStatus = $this->verifyStoreIsOpen($store['store_id']);
                    if ($openStatus) {
                        $verifiedStores[] = $store;
                        log_message('debug', "✓ VERIFIED OPEN: {$store['name']}");
                    } else {
                        log_message('debug', "✗ NOT OPEN: {$store['name']}");
                    }
                }
                $stores = $verifiedStores;
                log_message('debug', "After verification: " . count($stores) . " open stores");
            }

            // Check if more results exist by trying to get one more record
            $moreCheckSql = str_replace("LIMIT $limit OFFSET $offset", "LIMIT 1 OFFSET " . ($offset + $limit), $sql);
            $moreResults = $this->db->query($moreCheckSql)->getResultArray();
            $hasMoreResults = !empty($moreResults);

            log_message('debug', "Has more results: " . ($hasMoreResults ? 'YES' : 'NO'));

            // Enhance stores with status and hours data
            $enhancedStores = [];
            foreach ($stores as $store) {
                $store['id'] = $store['store_id']; // Set ID for modal functionality
                
                // Ensure distance is properly formatted
                if (isset($store['distance']) && is_numeric($store['distance'])) {
                    $store['distance'] = round((float)$store['distance'], 2);
                } else {
                    $store['distance'] = null;
                }
                
                // Get comprehensive status from CRON data
                $statusData = $this->getBusinessStatusFromCRON($store['store_id']);
                $store['open_status'] = $this->formatStatusForCard($statusData);
                $store['next_open'] = $this->getNextOpenFromCRON($statusData);
                
                // Check for active coupon
                $store['has_active_coupon'] = $this->hasActiveCoupon($store['store_id']);
                
                // Check for active jobs
                $store['has_active_jobs'] = $this->hasActiveJobs($store['store_id']);
                
                // Get actual jobs data for modal
                $store['active_jobs'] = $this->getActiveJobsForModal($store['store_id']);
                
                // Add hours data for modal
                $store['all_time'] = $this->getFormattedHours($store['store_id']);
                $store['moreHours'] = $this->getAdditionalHourTypes($store['store_id']);
                
                $enhancedStores[] = $store;
            }

            log_message('debug', "Enhanced stores count: " . count($enhancedStores));

            // Build HTML response
            $html = '';
            foreach ($enhancedStores as $store) {
                $html .= view('partials/store_card', ['store' => $store]);
            }

            // Get accurate total counts for proper display
            $totalCounts = $this->getTotalCounts($hasCoords, $lat, $lng, $openOnly, $category, $keyword);

            // Add metadata for JavaScript infinite scroll handling
            $metadata = [
                'has_more_results' => $hasMoreResults,
                'current_page' => $page,
                'open_only_mode' => $openOnly,
                'category_filter' => $category,
                'keyword_search' => $keyword,
                'results_count' => count($enhancedStores),
                'has_coordinates' => $hasCoords,
                'total_open_businesses' => $totalCounts['total_open'],
                'total_all_businesses' => $totalCounts['total_all'],
                'displayed_count' => $offset + count($enhancedStores)
            ];

            $html = "<!-- METADATA: " . json_encode($metadata) . " -->" . $html;

            log_message('debug', "Final response: " . count($enhancedStores) . " stores, has_more: " . ($hasMoreResults ? 'true' : 'false'));
            log_message('debug', "=== END LOCATOR REQUEST ===");

            return $this->response->setBody($html);

        } catch (\Exception $e) {
            log_message('error', "Locator controller error: " . $e->getMessage());
            log_message('error', "Stack trace: " . $e->getTraceAsString());
            return $this->response->setBody('<div class="alert alert-danger p-4">Error loading businesses: ' . esc($e->getMessage()) . '</div>');
        }
    }

    /**
     * Get accurate total counts from database for display
     */
    private function getTotalCounts($hasCoords, $lat, $lng, $currentlyFilteringOpen = false, $category = '', $keyword = '')
    {
        $distanceCondition = $hasCoords ? 
            "AND (3959 * acos(cos(radians($lat)) * cos(radians(s.latitude)) * cos(radians(s.longitude) - radians($lng)) + sin(radians($lat)) * sin(radians(s.latitude)))) <= 2500" : 
            "";

        // Build category JOIN and WHERE conditions for counts
        $categoryJoin = '';
        $categoryWhere = '';
        if (!empty($category)) {
            $categoryJoin = "INNER JOIN store_category sc ON s.store_id = sc.store_id 
                            INNER JOIN category c ON sc.category_id = c.category_id";
            $categoryWhere = "AND c.category_id = " . $this->db->escape($category) . " AND c.status = 1";
        }

        // Build keyword search conditions for counts
        $keywordWhere = '';
        if (!empty($keyword)) {
            $escapedKeyword = $this->db->escapeLikeString($keyword);
            $keywordWhere = "AND (
                s.name LIKE '%{$escapedKeyword}%' OR 
                s.description LIKE '%{$escapedKeyword}%' OR 
                s.address LIKE '%{$escapedKeyword}%' OR 
                s.city LIKE '%{$escapedKeyword}%' OR 
                s.state LIKE '%{$escapedKeyword}%' OR 
                s.postal_code LIKE '%{$escapedKeyword}%' OR 
                s.phone LIKE '%{$escapedKeyword}%' OR 
                s.website LIKE '%{$escapedKeyword}%'
            )";
        }

        // Count all businesses in range (with category and keyword filters if applicable)
        $totalAllSql = "
            SELECT COUNT(DISTINCT s.store_id) as total
            FROM store s
            $categoryJoin
            WHERE s.status = 1
            $categoryWhere
            $keywordWhere
            $distanceCondition
        ";
        
        $totalAllResult = $this->db->query($totalAllSql)->getRowArray();
        $totalAll = $totalAllResult['total'] ?? 0;

        // Count open businesses in range (with category and keyword filters if applicable)
        $totalOpenSql = "
            SELECT COUNT(DISTINCT s.store_id) as total
            FROM store s
            INNER JOIN store_current_status scs ON s.store_id = scs.store_id
            $categoryJoin
            WHERE s.status = 1 
            AND scs.status IN ('open', 'closing_soon')
            $categoryWhere
            $keywordWhere
            $distanceCondition
        ";
        
        $totalOpenResult = $this->db->query($totalOpenSql)->getRowArray();
        $totalOpen = $totalOpenResult['total'] ?? 0;

        return [
            'total_all' => $totalAll,
            'total_open' => $totalOpen
        ];
    }

    /**
     * Verify that a store is actually open by checking CRON status table
     */
    private function verifyStoreIsOpen($storeId)
    {
        $openServices = $this->db->query("
            SELECT COUNT(*) as open_count
            FROM store_current_status 
            WHERE store_id = ? AND status IN ('open', 'closing_soon')
        ", [$storeId])->getRowArray();

        return ($openServices['open_count'] > 0);
    }

    /**
     * Get business status from CRON-generated data
     */
    private function getBusinessStatusFromCRON($storeId)
    {
        $statusRecords = $this->db->query("
            SELECT service_type, status, current_message, closes_at, next_open_datetime
            FROM store_current_status 
            WHERE store_id = ?
            ORDER BY service_type
        ", [$storeId])->getResultArray();

        $openServices = [];
        $anyOpen = false;
        $nextOpenTime = null;

        foreach ($statusRecords as $record) {
            if (in_array($record['status'], ['open', 'closing_soon'])) {
                $openServices[] = [
                    'name' => $record['service_type'],
                    'status' => $record['status'],
                    'closes_at' => $record['closes_at'],
                    'message' => $record['current_message']
                ];
                $anyOpen = true;
            } else {
                // Track earliest next opening time
                if ($record['next_open_datetime'] && (!$nextOpenTime || $record['next_open_datetime'] < $nextOpenTime)) {
                    $nextOpenTime = $record['next_open_datetime'];
                }
            }
        }

        return [
            'any_open' => $anyOpen,
            'open_services' => $openServices,
            'next_open_datetime' => $nextOpenTime,
            'all_services' => $statusRecords
        ];
    }

    /**
     * Format status for display on business cards
     */
    private function formatStatusForCard($statusData)
    {
        if (!$statusData['any_open']) {
            return 'Closed';
        }

        $lines = [];
        foreach ($statusData['open_services'] as $service) {
            if ($service['status'] === 'closing_soon') {
                $lines[] = "{$service['name']} closing soon";
            } else if ($service['closes_at']) {
                $closeTime = date('g:i A', strtotime($service['closes_at']));
                $lines[] = "{$service['name']} until {$closeTime}";
            } else {
                $lines[] = "{$service['name']} open";
            }
        }

        return implode("\n", $lines);
    }

    /**
     * Get next opening time for closed businesses
     */
    private function getNextOpenFromCRON($statusData)
    {
        if ($statusData['any_open']) {
            return null; // Business is currently open
        }

        if ($statusData['next_open_datetime']) {
            try {
                // Parse the next opening datetime
                $nextOpen = new \DateTime($statusData['next_open_datetime']);
                
                // Get current time in Eastern timezone (business timezone)
                $now = new \DateTime('now', new \DateTimeZone('America/New_York'));
                
                // FIXED: Compare only the DATE parts, not full datetime
                $nextOpenDate = $nextOpen->format('Y-m-d');
                $todayDate = $now->format('Y-m-d');
                
                if ($nextOpenDate == $todayDate) {
                    // Same date - show "today"
                    return 'Opens today at ' . $nextOpen->format('g:i A');
                } else {
                    // Different dates - calculate days difference using date-only objects
                    $nextDateObj = new \DateTime($nextOpenDate);
                    $todayDateObj = new \DateTime($todayDate);
                    $interval = $todayDateObj->diff($nextDateObj);
                    
                    if ($interval->days == 1) {
                        return 'Opens tomorrow at ' . $nextOpen->format('g:i A');
                    } else {
                        return 'Opens ' . $nextOpen->format('l \a\t g:i A');
                    }
                }
                
            } catch (\Exception $e) {
                log_message('error', 'Error parsing next_open_datetime: ' . $e->getMessage());
                return 'Hours not available';
            }
        }

        return 'Hours not available';
    }

    /**
     * Get formatted hours for main service types (for modal)
     */
    private function getFormattedHours($storeId)
    {
        try {
            $mainHours = $this->db->query("
                SELECT sht.custom_name, ss.day_of_week, ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
                ORDER BY ss.day_of_week
            ", [$storeId])->getResultArray();

            $formattedHours = [];
            $daySchedules = [];

            // Group by day
            foreach ($mainHours as $hour) {
                $day = $hour['day_of_week'];
                if (!isset($daySchedules[$day])) {
                    $daySchedules[$day] = [
                        'is_closed' => true,
                        'times' => []
                    ];
                }

                if (!$hour['is_closed']) {
                    $daySchedules[$day]['is_closed'] = false;
                    
                    if ($hour['slot_1_open'] && $hour['slot_1_close']) {
                        $daySchedules[$day]['times'][] = [
                            'to' => $hour['slot_1_open'],
                            'from' => $hour['slot_1_close']
                        ];
                    }
                    if ($hour['slot_2_open'] && $hour['slot_2_close']) {
                        $daySchedules[$day]['times'][] = [
                            'to' => $hour['slot_2_open'],
                            'from' => $hour['slot_2_close']
                        ];
                    }
                }
            }

            // Create formatted hours for all 7 days
            for ($day = 1; $day <= 7; $day++) {
                $times = [];
                if (isset($daySchedules[$day]) && !$daySchedules[$day]['is_closed']) {
                    $times = $daySchedules[$day]['times'];
                }
                
                $formattedHours[] = [
                    'day' => $day,
                    'time' => json_encode($times)
                ];
            }

            return $formattedHours;
        } catch (\Exception $e) {
            log_message('error', 'Error formatting hours: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get additional hour types (non-main status affecting) for modal
     */
    private function getAdditionalHourTypes($storeId)
    {
        try {
            $additionalTypes = $this->db->query("
                SELECT sht.custom_name, ss.day_of_week, ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.affects_main_status = 0 AND sht.is_active = 1
                ORDER BY sht.display_order, ss.day_of_week
            ", [$storeId])->getResultArray();

            $moreHours = [];
            
            foreach ($additionalTypes as $type) {
                $typeName = $type['custom_name'];
                if (!isset($moreHours[$typeName])) {
                    $moreHours[$typeName] = [];
                }

                $times = [];
                if (!$type['is_closed']) {
                    if ($type['slot_1_open'] && $type['slot_1_close']) {
                        $times[] = [
                            'to' => $type['slot_1_open'],
                            'from' => $type['slot_1_close']
                        ];
                    }
                    if ($type['slot_2_open'] && $type['slot_2_close']) {
                        $times[] = [
                            'to' => $type['slot_2_open'],
                            'from' => $type['slot_2_close']
                        ];
                    }
                }

                $moreHours[$typeName][$type['day_of_week']] = [
                    'time' => $times
                ];
            }

            return $moreHours;
        } catch (\Exception $e) {
            log_message('error', 'Error formatting additional hours: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Check if store has active coupon
     */
    private function hasActiveCoupon($storeId)
    {
        try {
            $today = date('Y-m-d');
            
            $coupon = $this->db->query("
                SELECT COUNT(*) as has_coupon
                FROM business_coupons 
                WHERE store_id = ? 
                AND is_active = 1 
                AND starts_at <= ? 
                AND expires_at >= ?
                LIMIT 1
            ", [$storeId, $today, $today])->getRowArray();
            
            return ($coupon['has_coupon'] > 0);
        } catch (\Exception $e) {
            log_message('error', 'Error checking coupon: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if store has active job postings
     */
    private function hasActiveJobs($storeId)
    {
        try {
            $today = date('Y-m-d');
            
            $job = $this->db->query("
                SELECT COUNT(*) as has_jobs
                FROM business_jobs 
                WHERE store_id = ? 
                AND is_active = 1 
                AND (expires_at IS NULL OR expires_at >= ?)
                LIMIT 1
            ", [$storeId, $today])->getRowArray();
            
            log_message('debug', "Jobs query for $storeId returned: " . ($job['has_jobs'] ?? 'NULL'));
            
            return ($job['has_jobs'] > 0);
        } catch (\Exception $e) {
            log_message('error', 'Error checking jobs: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get active jobs for modal display
     */
    private function getActiveJobsForModal($storeId)
    {
        try {
            $today = date('Y-m-d');
            
            $jobs = $this->db->query("
                SELECT id, title, description, employment_type, salary_min, salary_max, salary_type, 
                       contact_method, contact_value, created_at
                FROM business_jobs 
                WHERE store_id = ? 
                AND is_active = 1 
                AND (expires_at IS NULL OR expires_at >= ?)
                ORDER BY created_at DESC
                LIMIT 3
            ", [$storeId, $today])->getResultArray();
            
            // Format each job for display
            $formattedJobs = [];
            foreach ($jobs as $job) {
                $formattedJobs[] = $this->formatJobForModal($job, $storeId);
            }
            
            return $formattedJobs;
        } catch (\Exception $e) {
            log_message('error', 'Error getting active jobs for modal: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Format job data for modal display
     */
    private function formatJobForModal($job, $storeId)
    {
        // Format employment type
        $employmentTypeMap = [
            'full_time' => 'Full Time',
            'part_time' => 'Part Time', 
            'contract' => 'Contract',
            'seasonal' => 'Seasonal'
        ];
        $job['employment_type_display'] = $employmentTypeMap[$job['employment_type']] ?? ucfirst($job['employment_type']);
        
        // Format salary display
        $job['salary_display'] = $this->formatSalaryDisplay($job['salary_min'], $job['salary_max'], $job['salary_type']);
        
        // Format contact method
        $job['contact_method_display'] = $this->formatContactMethodDisplay($job['contact_method'], $job['contact_value'], $storeId);
        
        // Format posted date
        $job['posted_date'] = date('M j, Y', strtotime($job['created_at']));
        
        return $job;
    }

    /**
     * Format salary for display
     */
    private function formatSalaryDisplay($salaryMin, $salaryMax, $salaryType)
    {
        $typeMap = [
            'hourly' => '/hour',
            'monthly' => '/month', 
            'yearly' => '/year'
        ];
        
        $suffix = $typeMap[$salaryType] ?? '';
        
        if ($salaryMax && $salaryMax > $salaryMin) {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                $minDisplay = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                $maxDisplay = $salaryMax >= 1000 ? number_format($salaryMax / 1000, 0) . 'K' : number_format($salaryMax, 0);
                return "$" . $minDisplay . "-" . $maxDisplay . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "-" . number_format($salaryMax, 2) . $suffix;
            }
        } else {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                $display = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                return "$" . $display . "+" . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "+" . $suffix;
            }
        }
    }

    /**
     * Format contact method for display
     */
    private function formatContactMethodDisplay($contactMethod, $contactValue, $storeId)
    {
        switch ($contactMethod) {
            case 'phone':
                // Get store phone number
                $storeInfo = $this->db->query("SELECT phone FROM store WHERE store_id = ?", [$storeId])->getRowArray();
                return $storeInfo ? "Call: " . $storeInfo['phone'] : "Call us";
                
            case 'email':
                // Get store email
                $storeInfo = $this->db->query("SELECT email FROM store WHERE store_id = ?", [$storeId])->getRowArray();
                return $storeInfo ? "Email: " . $storeInfo['email'] : "Email us";
                
            case 'walk_in':
                return "Visit us in person";
                
            case 'online':
                return $contactValue ? "Apply online" : "Apply online";
                
            default:
                return "Contact us";
        }
    }
}