<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Models\BusinessChangeRequestModel;
use CodeIgniter\HTTP\ResponseInterface;

class Dashboard extends BaseController
{
    protected $businessUserModel;
    protected $storeModel;
    protected $changeRequestModel;
    protected $session;

    public function __construct()
    {
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->changeRequestModel = new BusinessChangeRequestModel();
        $this->session = \Config\Services::session();
    }

    /**
     * Business dashboard home page
     */
    public function index()
    {
        // Check authentication
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $userId = $this->session->get('business_user_id');
        $storeId = $this->session->get('business_store_id');

        // Get user with store information
        $user = $this->businessUserModel->getUserWithStore($userId);
        
        if (!$user) {
            $this->session->destroy();
            return redirect()->to('/business/auth/login')->with('error', 'Account not found.');
        }

        // Get store details
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/auth/login')->with('error', 'Business not found.');
        }

        // Get recent change requests
        $recentRequests = $this->changeRequestModel->getRequestsByStore($storeId, 5);

        // Get pending change requests count
        $pendingCount = $this->changeRequestModel->getPendingRequestsCount($storeId);

        // Get dashboard statistics
        $stats = $this->getDashboardStats($storeId);

        $data = [
            'title' => 'Business Dashboard',
            'user' => $user,
            'store' => $store,
            'recent_requests' => $recentRequests,
            'pending_count' => $pendingCount,
            'stats' => $stats,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/dashboard/index', $data);
    }

    /**
     * Business profile overview
     */
    public function profile()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $userId = $this->session->get('business_user_id');
        $storeId = $this->session->get('business_store_id');

        // Get user and store information
        $user = $this->businessUserModel->getUserWithStore($userId);
        $store = $this->storeModel->getStoreById($storeId);

        if (!$user || !$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Profile information not found.');
        }

        $data = [
            'title' => 'Business Profile',
            'user' => $user,
            'store' => $store,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/dashboard/profile', $data);
    }

    /**
     * Change requests history
     */
    public function changeRequests()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');

        // Get all change requests for this store
        $requests = $this->changeRequestModel->getRequestsByStore($storeId);

        // Get summary stats
        $stats = [
            'total' => count($requests),
            'pending' => $this->changeRequestModel->getPendingRequestsCount($storeId),
            'approved' => $this->changeRequestModel->getApprovedRequestsCount($storeId),
            'rejected' => $this->changeRequestModel->getRejectedRequestsCount($storeId)
        ];

        $data = [
            'title' => 'Change Requests',
            'requests' => $requests,
            'stats' => $stats,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/dashboard/change_requests', $data);
    }

    /**
     * View specific change request details
     */
    public function viewChangeRequest($requestId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');

        // Get change request (ensure it belongs to this store)
        $request = $this->changeRequestModel->getRequestByIdAndStore($requestId, $storeId);

        if (!$request) {
            return redirect()->to('/business/dashboard/change-requests')->with('error', 'Change request not found.');
        }

        // Decode field changes for display
        $fieldChanges = json_decode($request['field_changes'], true);

        $data = [
            'title' => 'Change Request Details',
            'request' => $request,
            'field_changes' => $fieldChanges,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/dashboard/change_request_detail', $data);
    }

    /**
     * Account settings page
     */
    public function settings()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $userId = $this->session->get('business_user_id');
        $user = $this->businessUserModel->find($userId);

        if (!$user) {
            return redirect()->to('/business/dashboard')->with('error', 'User account not found.');
        }

        $data = [
            'title' => 'Account Settings',
            'user' => $user,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/dashboard/settings', $data);
    }

    /**
     * Update account information
     */
    public function updateAccount()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $rules = [
            'first_name' => 'required|min_length[2]|max_length[100]',
            'last_name' => 'required|min_length[2]|max_length[100]',
            'email' => 'required|valid_email|max_length[255]',
            'phone' => 'required|min_length[10]|max_length[20]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('error', 'Please fill in all required fields correctly.');
        }

        $userId = $this->session->get('business_user_id');
        
        $updateData = [
            'first_name' => $this->request->getPost('first_name'),
            'last_name' => $this->request->getPost('last_name'),
            'email' => $this->request->getPost('email'),
            'phone' => $this->request->getPost('phone')
        ];

        // Check if email is already used by another business user
        if ($this->businessUserModel->emailExists($updateData['email'], $userId)) {
            return redirect()->back()->withInput()->with('error', 'This email address is already in use.');
        }

        if ($this->businessUserModel->update($userId, $updateData)) {
            return redirect()->back()->with('success', 'Account information updated successfully.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to update account information. Please try again.');
        }
    }

    /**
     * Get dashboard statistics
     */
    private function getDashboardStats($storeId)
    {
        $stats = [
            'total_requests' => $this->changeRequestModel->getTotalRequestsCount($storeId),
            'pending_requests' => $this->changeRequestModel->getPendingRequestsCount($storeId),
            'approved_requests' => $this->changeRequestModel->getApprovedRequestsCount($storeId),
            'rejected_requests' => $this->changeRequestModel->getRejectedRequestsCount($storeId),
            'last_update' => $this->changeRequestModel->getLastRequestDate($storeId),
            'profile_completion' => $this->getProfileCompletionPercentage($storeId)
        ];

        return $stats;
    }

    /**
     * Calculate profile completion percentage
     */
    private function getProfileCompletionPercentage($storeId)
    {
        $store = $this->storeModel->find($storeId);
        
        if (!$store) {
            return 0;
        }

        $requiredFields = [
            'name', 'description', 'email', 'phone', 'website', 
            'address', 'city', 'state', 'postal_code'
        ];

        $completedFields = 0;
        $totalFields = count($requiredFields);

        foreach ($requiredFields as $field) {
            if (!empty($store[$field])) {
                $completedFields++;
            }
        }

        return round(($completedFields / $totalFields) * 100);
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && !empty($this->session->get('business_user_id'));
    }
}