<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessCouponModel;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Services\BusinessEmailService;

class Coupons extends BaseController
{
    protected $couponModel;
    protected $businessUserModel;
    protected $storeModel;
    protected $emailService;
    protected $session;

    public function __construct()
    {
        $this->couponModel = new BusinessCouponModel();
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->emailService = new BusinessEmailService();
        $this->session = \Config\Services::session();
    }

    /**
     * Coupons management page with calendar view
     */
    public function index()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Store not found.');
        }

        // Get coupons for calendar display (3 months range)
        $startDate = date('Y-m-01', strtotime('-1 month'));
        $endDate = date('Y-m-t', strtotime('+2 months'));
        $coupons = $this->couponModel->getStoreCoupons($storeId, $startDate, $endDate);

        // Format coupons for calendar
        $calendarEvents = [];
        foreach ($coupons as $coupon) {
            $formattedCoupon = $this->couponModel->formatCouponDisplay($coupon);
            $calendarEvents[] = [
                'id' => $coupon['id'],
                'title' => $formattedCoupon['discount_display'] . ' - ' . $coupon['title'],
                'start' => $coupon['starts_at'],
                'end' => date('Y-m-d', strtotime($coupon['expires_at'] . ' +1 day')), // FullCalendar needs end date exclusive
                'backgroundColor' => $coupon['is_active'] ? '#10b981' : '#6b7280',
                'borderColor' => $coupon['is_active'] ? '#059669' : '#4b5563',
                'extendedProps' => [
                    'description' => $coupon['description'],
                    'discount_type' => $coupon['discount_type'],
                    'discount_value' => $coupon['discount_value'],
                    'usage_count' => $coupon['usage_count'],
                    'usage_limit' => $coupon['usage_limit'],
                    'is_active' => $coupon['is_active']
                ]
            ];
        }

        // Get statistics
        $stats = $this->couponModel->getCouponStats($storeId);
        
        // Get current active coupon
        $activeCoupon = $this->couponModel->getActiveCoupon($storeId);
        if ($activeCoupon) {
            $activeCoupon = $this->couponModel->formatCouponDisplay($activeCoupon);
        }

        // Get upcoming coupons
        $upcomingCoupons = $this->couponModel->getUpcomingCoupons($storeId, 5);

        $data = [
            'title' => 'Manage Coupons',
            'store' => $store,
            'calendar_events' => json_encode($calendarEvents),
            'stats' => $stats,
            'active_coupon' => $activeCoupon,
            'upcoming_coupons' => $upcomingCoupons,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/coupons/index', $data);
    }

    /**
     * Create new coupon
     */
    public function create()
    {
        // DEBUG: Log every time this method is called
        log_message('debug', 'Coupon CREATE: Method called, request method: ' . $this->request->getMethod());
        
        if (!$this->isLoggedIn()) {
            log_message('debug', 'Coupon CREATE: User not logged in, redirecting');
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'post' || $this->request->getPost()) {
            // DEBUG: Log that we're entering the save process
            log_message('debug', 'Coupon CREATE: POST request received, calling save()');
            return $this->save();
        }

        // DEBUG: Log GET request (showing the form)
        log_message('debug', 'Coupon CREATE: GET request - showing form');
        
        $data = [
            'title' => 'Create New Coupon',
            'coupon' => null,
            'is_edit' => false
        ];

        return view('business/coupons/form', $data);
    }

    /**
     * Edit existing coupon
     */
    public function edit($id)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $coupon = $this->couponModel->where('store_id', $storeId)->find($id);

        if (!$coupon) {
            return redirect()->to('/business/coupons')->with('error', 'Coupon not found.');
        }

        if ($this->request->getMethod() === 'post' || $this->request->getPost()) {
            // DEBUG: Log that we're entering the save process for edit
            log_message('debug', 'Coupon EDIT: POST request received for ID ' . $id . ', calling save()');
            return $this->save($id);
        }

        $data = [
            'title' => 'Edit Coupon',
            'coupon' => $coupon,
            'is_edit' => true
        ];

        return view('business/coupons/form', $data);
    }

    /**
     * Save coupon (create or update)
     */
    private function save($id = null)
    {
        // DEBUG: Log entry to save method
        log_message('debug', 'Coupon SAVE: Method started. ID: ' . ($id ?? 'null'));
        
        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');
        
        // DEBUG: Log session data
        log_message('debug', 'Coupon SAVE: Store ID: ' . $storeId . ', User ID: ' . $userId);
        
        // DEBUG: Log all POST data
        log_message('debug', 'Coupon SAVE: POST data: ' . json_encode($this->request->getPost()));

        // Validation rules - simplified like Profile controller
        $rules = [
            'title' => 'required|min_length[3]|max_length[255]',
            'description' => 'required|min_length[10]',
            'discount_type' => 'required|in_list[percentage,fixed_amount,bogo,free_item]',
            'starts_at' => 'required',
            'expires_at' => 'required'
        ];

        // Only validate discount_value for types that need it
        $discountType = $this->request->getPost('discount_type');
        if (in_array($discountType, ['percentage', 'fixed_amount'])) {
            $rules['discount_value'] = 'required|numeric|greater_than[0]';
            if ($discountType === 'percentage') {
                $rules['discount_value'] .= '|less_than_equal_to[100]';
            }
        }

        // DEBUG: Log validation attempt
        log_message('debug', 'Coupon SAVE: Starting validation with rules: ' . json_encode($rules));

        if (!$this->validate($rules)) {
            // DEBUG: Log validation failure
            $errors = $this->validator->getErrors();
            log_message('debug', 'Coupon SAVE: Validation FAILED. Errors: ' . json_encode($errors));
            return redirect()->back()->withInput()->with('validation', $errors);
        }

        // DEBUG: Log validation success
        log_message('debug', 'Coupon SAVE: Validation PASSED');

        // Prepare data for insert/update - following Profile controller pattern
        $data = [
            'store_id' => (string)$storeId,
            'title' => trim($this->request->getPost('title')),
            'description' => trim($this->request->getPost('description')),
            'discount_type' => $discountType,
            'starts_at' => $this->request->getPost('starts_at'),
            'expires_at' => $this->request->getPost('expires_at'),
            'is_active' => $this->request->getPost('is_active') ? 1 : 0
        ];

        // Set discount_value based on type
        if (in_array($discountType, ['percentage', 'fixed_amount'])) {
            $data['discount_value'] = (float)$this->request->getPost('discount_value');
        } else {
            $data['discount_value'] = 1; // Default for BOGO and free_item
        }

        // Set optional fields
        $data['minimum_purchase'] = $this->request->getPost('minimum_purchase') ?: null;
        $data['usage_limit'] = $this->request->getPost('usage_limit') ?: null;

        if (!$id) {
            $data['created_by'] = (int)$userId;
            $data['usage_count'] = 0;
        } else {
            // For updates, ensure created_by is not overwritten
            unset($data['created_by']);
        }

        // DEBUG: Log data to be saved and verify required fields
        log_message('debug', 'Coupon SAVE: Data to save: ' . json_encode($data));
        log_message('debug', 'Coupon SAVE: Required field check - store_id: ' . $data['store_id'] . ' (type: ' . gettype($data['store_id']) . ')');
        log_message('debug', 'Coupon SAVE: Required field check - created_by: ' . ($data['created_by'] ?? 'NOT SET') . ' (for new records only)');

        // Save the coupon - let's add comprehensive debugging
        try {
            // Check database connection first
            $db = \Config\Database::connect();
            log_message('debug', 'Coupon SAVE: Database connection status: ' . ($db->connID ? 'CONNECTED' : 'FAILED'));
            
            // Test a simple query to verify connection
            $testQuery = $db->query("SELECT 1 as test");
            log_message('debug', 'Coupon SAVE: Test query result: ' . ($testQuery ? 'SUCCESS' : 'FAILED'));
            
            if ($id) {
                log_message('debug', 'Coupon SAVE: Attempting UPDATE for ID: ' . $id);
                
                // Check if record exists first
                $existingRecord = $this->couponModel->find($id);
                log_message('debug', 'Coupon SAVE: Existing record found: ' . ($existingRecord ? 'YES' : 'NO'));
                
                $success = $this->couponModel->update($id, $data);
                $action = 'updated';
                log_message('debug', 'Coupon SAVE: Update result: ' . ($success ? 'SUCCESS' : 'FAILED'));
                
                // Get last query for debugging
                log_message('debug', 'Coupon SAVE: Last query: ' . $this->couponModel->getLastQuery());
                
            } else {
                log_message('debug', 'Coupon SAVE: Attempting INSERT');
                
                // Try direct SQL insert for debugging
                $sql = "INSERT INTO business_coupons (store_id, title, description, discount_type, discount_value, starts_at, expires_at, is_active, created_by, usage_count) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $directInsert = $db->query($sql, [
                    $data['store_id'],
                    $data['title'], 
                    $data['description'],
                    $data['discount_type'],
                    $data['discount_value'],
                    $data['starts_at'],
                    $data['expires_at'],
                    $data['is_active'],
                    $data['created_by'],
                    $data['usage_count']
                ]);
                log_message('debug', 'Coupon SAVE: Direct SQL insert result: ' . ($directInsert ? 'SUCCESS' : 'FAILED'));
                
                if ($directInsert) {
                    $insertId = $db->insertID();
                    log_message('debug', 'Coupon SAVE: Direct insert ID: ' . $insertId);
                    $success = true;
                    $id = $insertId;
                    $action = 'created';
                } else {
                    // Try with the model
                    $insertResult = $this->couponModel->insert($data);
                    $success = $insertResult !== false;
                    $insertId = $this->couponModel->getInsertID();
                    $action = 'created';
                    log_message('debug', 'Coupon SAVE: Model insert result: ' . ($success ? 'SUCCESS' : 'FAILED') . ', Insert ID: ' . $insertId);
                    log_message('debug', 'Coupon SAVE: Model last query: ' . $this->couponModel->getLastQuery());
                    
                    if ($success) {
                        $id = $insertId;
                    }
                }
            }
            
            // Check for database errors
            $dbError = $db->error();
            if ($dbError['code'] != 0) {
                log_message('error', 'Coupon SAVE: Database error: ' . json_encode($dbError));
            }
            
            // DEBUG: Check for model errors
            $modelErrors = $this->couponModel->errors();
            if (!empty($modelErrors)) {
                log_message('error', 'Coupon SAVE: Model errors: ' . json_encode($modelErrors));
            }
            
        } catch (\Exception $e) {
            log_message('error', 'Coupon SAVE: Exception during save: ' . $e->getMessage());
            log_message('error', 'Coupon SAVE: Exception trace: ' . $e->getTraceAsString());
            $success = false;
        }

        if ($success) {
            log_message('debug', 'Coupon SAVE: Save successful');
            return redirect()->to(base_url('business/coupons'))->with('success', "Coupon {$action} successfully!");
        } else {
            log_message('error', 'Coupon SAVE: Save FAILED - redirecting with error');
            return redirect()->back()->withInput()->with('error', "Failed to save coupon. Please try again.");
        }
    }

    /**
     * Toggle coupon active status
     */
    public function toggle($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $coupon = $this->couponModel->where('store_id', $storeId)->find($id);

        if (!$coupon) {
            return $this->response->setJSON(['success' => false, 'message' => 'Coupon not found']);
        }

        $newStatus = $coupon['is_active'] ? 0 : 1;
        
        // If activating, check for overlaps
        if ($newStatus === 1) {
            $hasOverlap = $this->couponModel->hasActiveCouponInRange(
                $storeId, 
                $coupon['starts_at'], 
                $coupon['expires_at'], 
                $id
            );
            
            if ($hasOverlap) {
                return $this->response->setJSON([
                    'success' => false, 
                    'message' => 'Another coupon is already active during this period.'
                ]);
            }
        }

        if ($this->couponModel->update($id, ['is_active' => $newStatus])) {
            // Notify admin
            $this->notifyAdminCouponChange($id, $newStatus ? 'activated' : 'deactivated');
            
            return $this->response->setJSON([
                'success' => true, 
                'message' => 'Coupon ' . ($newStatus ? 'activated' : 'deactivated') . ' successfully!'
            ]);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to update coupon status']);
    }

    /**
     * Delete coupon
     */
    public function delete($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $coupon = $this->couponModel->where('store_id', $storeId)->find($id);

        if (!$coupon) {
            return $this->response->setJSON(['success' => false, 'message' => 'Coupon not found']);
        }

        if ($this->couponModel->delete($id)) {
            // Notify admin
            $this->notifyAdminCouponChange($id, 'deleted', $coupon);
            
            return $this->response->setJSON(['success' => true, 'message' => 'Coupon deleted successfully!']);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to delete coupon']);
    }

    /**
     * Get coupon details via AJAX
     */
    public function details($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $coupon = $this->couponModel->where('store_id', $storeId)->find($id);

        if (!$coupon) {
            return $this->response->setJSON(['success' => false, 'message' => 'Coupon not found']);
        }

        $coupon = $this->couponModel->formatCouponDisplay($coupon);

        return $this->response->setJSON(['success' => true, 'coupon' => $coupon]);
    }

    /**
     * Preview coupon as it will appear to customers
     */
    public function preview($id)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $coupon = $this->couponModel->where('store_id', $storeId)->find($id);

        if (!$coupon) {
            return redirect()->to('/business/coupons')->with('error', 'Coupon not found.');
        }

        $store = $this->storeModel->getStoreById($storeId);
        $coupon = $this->couponModel->formatCouponDisplay($coupon);

        $data = [
            'title' => 'Coupon Preview',
            'store' => $store,
            'coupon' => $coupon
        ];

        return view('business/coupons/preview', $data);
    }

    /**
     * Send email notification to admin about coupon changes
     */
    private function notifyAdminCouponChange($couponId, $action, $couponData = null)
    {
        try {
            // Get coupon with creator info if not provided
            if (!$couponData) {
                $couponData = $this->couponModel->getCouponWithCreator($couponId);
            }
            
            if (!$couponData) {
                return false;
            }

            // Get store info
            $store = $this->storeModel->getStoreById($couponData['store_id']);
            if (!$store) {
                return false;
            }

            // Get user info
            $user = $this->businessUserModel->find($this->session->get('business_user_id'));
            
            $subject = "Coupon {$action} - {$store['name']}";
            
            $data = [
                'action' => $action,
                'store' => $store,
                'coupon' => $couponData,
                'user' => $user,
                'admin_url' => base_url('admin/coupons'),
                'timestamp' => date('Y-m-d H:i:s')
            ];

            // Use existing email service pattern
            $this->emailService->notifyAdminCouponChange($data);
            
            return true;

        } catch (\Exception $e) {
            log_message('error', "Failed to send coupon notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && 
               !empty($this->session->get('business_user_id'));
    }
}