<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\StoreModel;
use App\Models\CategoryModel;

class Stores extends BaseController
{
    protected $categoryModel;

    public function __construct()
    {
        $this->categoryModel = new CategoryModel();
    }

    public function index()
    {
        // Check admin authentication
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            $db = \Config\Database::connect();
            
            // Add pagination parameters
            $perPage = 20;
            $page = $this->request->getGet('page') ?: 1;
            $search = $this->request->getGet('search') ?: '';
            $statusFilter = $this->request->getGet('status') ?: '';
            $categoryFilter = $this->request->getGet('category') ?: '';

            // Build WHERE clause with category filtering
            $whereClause = "WHERE 1=1";
            $params = [];
            
            if (!empty($search)) {
                $whereClause .= " AND (s.name LIKE ? OR s.address LIKE ? OR s.city LIKE ?)";
                $searchParam = '%' . $search . '%';
                $params = [$searchParam, $searchParam, $searchParam];
            }

            if (!empty($statusFilter)) {
                switch ($statusFilter) {
                    case 'active':
                        $whereClause .= " AND s.status = 1 AND s.is_closed = 0";
                        break;
                    case 'inactive':
                        $whereClause .= " AND s.status = 0";
                        break;
                    case 'closed':
                        $whereClause .= " AND s.is_closed = 1";
                        break;
                }
            }

            // Category filtering
            $joinClause = '';
            if (!empty($categoryFilter)) {
                $joinClause = "INNER JOIN store_category sc ON s.store_id = sc.store_id";
                $whereClause .= " AND sc.category_id = ?";
                $params[] = $categoryFilter;
            }

            // Get total count
            $countQuery = "
                SELECT COUNT(DISTINCT s.id) as total 
                FROM store s 
                {$joinClause}
                {$whereClause}
            ";
            $totalResult = $db->query($countQuery, $params)->getRow();
            $total = $totalResult->total;

            // Get stores with category information
            $offset = ($page - 1) * $perPage;
            $dataQuery = "
                SELECT 
                    s.id, s.store_id, s.name, s.address, s.city, s.state, 
                    s.phone, s.email, s.website, s.status, s.is_closed,
                    GROUP_CONCAT(c.title ORDER BY c.title SEPARATOR ', ') as categories,
                    COUNT(DISTINCT sc.category_id) as category_count
                FROM store s
                LEFT JOIN store_category sc ON s.store_id = sc.store_id
                LEFT JOIN category c ON sc.category_id = c.category_id AND c.status = 1
                {$whereClause}
                GROUP BY s.id, s.store_id, s.name, s.address, s.city, s.state, 
                         s.phone, s.email, s.website, s.status, s.is_closed
                ORDER BY s.name ASC 
                LIMIT ? OFFSET ?
            ";
            
            $dataParams = $params;
            $dataParams[] = $perPage;
            $dataParams[] = $offset;
            
            $stores = $db->query($dataQuery, $dataParams)->getResultArray();

            // Get all categories for filter dropdown
            $categories = $this->categoryModel->getActiveCategories();

            $data = [
                'title' => 'Store Management',
                'stores' => $stores,
                'categories' => $categories,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => $total,
                    'total_pages' => ceil($total / $perPage)
                ],
                'search' => $search,
                'statusFilter' => $statusFilter,
                'categoryFilter' => $categoryFilter
            ];

            return view('admin/stores/index', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Stores index error: ' . $e->getMessage());
            return redirect()->to('/admin/dashboard')->with('error', 'Error loading stores: ' . $e->getMessage());
        }
    }

    // Store view method with categories
    public function view($id) {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            $db = \Config\Database::connect();
            $store = $db->query("SELECT * FROM store WHERE id = ?", [$id])->getRowArray();
            
            if (!$store) {
                return redirect()->to('/admin/stores')->with('error', 'Store not found');
            }

            // Get categories for this store
            $storeCategories = $this->categoryModel->getCategoriesForStore($store['store_id']);

            $data = [
                'title' => 'Store Details: ' . $store['name'],
                'store' => $store,
                'storeCategories' => $storeCategories
            ];

            return view('admin/stores/view', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Store view error: ' . $e->getMessage());
            return redirect()->to('/admin/stores')->with('error', 'Error loading store: ' . $e->getMessage());
        }
    }
    
    // Store edit method with category assignment
    public function edit($id) {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            $db = \Config\Database::connect();
            $store = $db->query("SELECT * FROM store WHERE id = ?", [$id])->getRowArray();
            
            if (!$store) {
                return redirect()->to('/admin/stores')->with('error', 'Store not found');
            }

            if ($this->request->getPost()) {
                // Handle comprehensive form submission
                $updateData = [
                    // Basic Information
                    'name' => $this->request->getPost('name'),
                    'description' => $this->request->getPost('description'),
                    'status' => $this->request->getPost('status') ?: 1,
                    'is_closed' => $this->request->getPost('is_closed') ? 1 : 0,
                    'close_message' => $this->request->getPost('close_message'),
                    
                    // Contact Information
                    'email' => $this->request->getPost('email'),
                    'phone' => $this->request->getPost('phone'),
                    'fax' => $this->request->getPost('fax'),
                    'website' => $this->request->getPost('website'),
                    
                    // Location Information
                    'address' => $this->request->getPost('address'),
                    'city' => $this->request->getPost('city'),
                    'state' => $this->request->getPost('state'),
                    'postal_code' => $this->request->getPost('postal_code'),
                    'store_timezone' => $this->request->getPost('store_timezone'),
                    
                    // Social Media
                    'facebook' => $this->request->getPost('facebook'),
                    'instagram' => $this->request->getPost('instagram'),
                    'twitter' => $this->request->getPost('twitter'),
                    'linkedin' => $this->request->getPost('linkedin'),
                    'youtube' => $this->request->getPost('youtube'),
                    'pinterest' => $this->request->getPost('pinterest')
                ];

                // Handle Vimeo video
                $vimeoId = $this->request->getPost('vimeo_id');
                if (!empty($vimeoId)) {
                    // Clean the ID and store it
                    $cleanVimeoId = preg_replace('/[^0-9]/', '', $vimeoId);
                    $updateData['additional_info'] = $cleanVimeoId;
                } else {
                    $updateData['additional_info'] = '';
                }

                // Clear closure message if not closed
                if (!$updateData['is_closed']) {
                    $updateData['close_message'] = '';
                }

                // Handle category assignments
                $categoryIds = $this->request->getPost('categories') ?: [];

                try {
                    // Start transaction
                    $db->transStart();

                    // Update store data
                    $db->table('store')->where('id', $id)->update($updateData);

                    // Update category assignments
                    $this->categoryModel->updateStoreCategories($store['store_id'], $categoryIds);

                    $db->transComplete();

                    if ($db->transStatus() === false) {
                        throw new \Exception('Transaction failed');
                    }

                    return redirect()->to('/admin/stores/view/' . $id)->with('success', 'Store updated successfully');

                } catch (\Exception $e) {
                    $db->transRollback();
                    log_message('error', 'Store update failed: ' . $e->getMessage());
                    return redirect()->back()->with('error', 'Failed to update store: ' . $e->getMessage());
                }
            }

            // Get all categories for assignment
            $allCategories = $this->categoryModel->getActiveCategories();
            
            // Get currently assigned categories
            $assignedCategories = $this->categoryModel->getCategoriesForStore($store['store_id']);
            $assignedCategoryIds = array_column($assignedCategories, 'category_id');

            $data = [
                'title' => 'Edit Store: ' . $store['name'],
                'store' => $store,
                'allCategories' => $allCategories,
                'assignedCategoryIds' => $assignedCategoryIds
            ];

            return view('admin/stores/edit', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Store edit error: ' . $e->getMessage());
            return redirect()->to('/admin/stores')->with('error', 'Error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX endpoint for bulk category assignment
     */
    public function bulkAssignCategories()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        $storeIds = $this->request->getPost('store_ids');
        $categoryIds = $this->request->getPost('category_ids');

        if (empty($storeIds) || empty($categoryIds)) {
            return $this->response->setJSON(['success' => false, 'message' => 'Missing store IDs or category IDs']);
        }

        try {
            $result = $this->categoryModel->bulkAssignCategories($storeIds, $categoryIds);

            if ($result) {
                return $this->response->setJSON([
                    'success' => true, 
                    'message' => 'Categories assigned to ' . count($storeIds) . ' stores successfully'
                ]);
            } else {
                throw new \Exception('Bulk assignment failed');
            }

        } catch (\Exception $e) {
            log_message('error', 'Bulk category assignment failed: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'message' => 'Assignment failed: ' . $e->getMessage()]);
        }
    }

    /**
     * DELETE BUSINESS COMPLETELY
     */
    public function deleteBusiness()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Not authorized'
            ]);
        }
        
        $storeId = $this->request->getPost('store_id');
        
        if (!$storeId) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'No store ID provided'
            ]);
        }
        
        try {
            $db = \Config\Database::connect();
            
            // Get business info for logging
            $business = $db->query("SELECT name FROM store WHERE store_id = ?", [$storeId])->getRowArray();
            $businessName = $business['name'] ?? "Store ID $storeId";
            
            // Start transaction
            $db->transStart();
            
            // Delete in proper order to respect foreign keys
            $db->query("DELETE FROM store_schedule_exceptions WHERE store_id = ?", [$storeId]);
            $db->query("DELETE FROM store_schedules WHERE store_id = ?", [$storeId]);
            $db->query("DELETE FROM store_hour_types WHERE store_id = ?", [$storeId]);
            $db->query("DELETE FROM store_schedule_templates WHERE store_id = ?", [$storeId]);
            $db->query("DELETE FROM store_category WHERE store_id = ?", [$storeId]);
            $db->query("DELETE FROM store WHERE store_id = ?", [$storeId]);
            
            $db->transComplete();
            
            if ($db->transStatus() === false) {
                throw new \Exception('Transaction failed');
            }
            
            log_message('info', "Business completely deleted: $businessName (Store ID: $storeId) by admin: " . session()->get('admin_name'));
            
            return $this->response->setJSON([
                'success' => true,
                'message' => "Business '$businessName' has been completely deleted from the system"
            ]);
            
        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', "Failed to delete business $storeId: " . $e->getMessage());
            
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to delete business: ' . $e->getMessage()
            ]);
        }
    }
}