<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\StoreModel;

class Hours extends BaseController
{
    protected $storeModel;
    protected $db;

    public function __construct()
    {
        $this->storeModel = new StoreModel();
        $this->db = \Config\Database::connect();
    }

    /**
     * Hours management dashboard - shows all businesses with their current status
     */
    public function index()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        // Get all stores with their hour types and current status
        $query = "
            SELECT 
                s.id,
                s.store_id,
                s.name,
                s.city,
                s.state,
                COUNT(sht.id) as hour_types_count,
                GROUP_CONCAT(sht.custom_name ORDER BY sht.display_order SEPARATOR ', ') as hour_types,
                MAX(sht.updated_at) as last_updated
            FROM store s
            LEFT JOIN store_hour_types sht ON s.store_id = sht.store_id AND sht.is_active = 1
            GROUP BY s.id, s.store_id, s.name, s.city, s.state
            ORDER BY s.name
        ";
        
        $stores = $this->db->query($query)->getResultArray();

        // Calculate current status for each store
        foreach ($stores as &$store) {
            $store['current_status'] = $this->calculateCurrentStatus($store['store_id']);
            $store['status_badge'] = $this->getStatusBadge($store['current_status']);
        }

        $data = [
            'title' => 'Business Hours Management',
            'stores' => $stores,
            'total_stores' => count($stores),
            'stores_with_hours' => count(array_filter($stores, fn($s) => $s['hour_types_count'] > 0))
        ];

        return view('admin/hours/index', $data);
    }

    /**
     * Edit hours for a specific business - the main scheduling interface
     */
    public function store($storeId)
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        // Get store details
        $store = $this->db->query("SELECT * FROM store WHERE store_id = ?", [$storeId])->getRowArray();
        if (!$store) {
            return redirect()->to('/admin/hours')->with('error', 'Business not found');
        }

        // Get all global hour types for selection
        $globalTypes = $this->db->query("SELECT * FROM hours_type ORDER BY name")->getResultArray();

        // Get this store's configured hour types
        $storeHourTypes = $this->db->query("
            SELECT sht.*, ht.name as global_name 
            FROM store_hour_types sht 
            LEFT JOIN hours_type ht ON sht.global_type_id = ht.id
            WHERE sht.store_id = ? AND sht.is_active = 1 
            ORDER BY sht.display_order, sht.custom_name
        ", [$storeId])->getResultArray();

        // Get schedules for each hour type
        $schedules = [];
        foreach ($storeHourTypes as $hourType) {
            $weekSchedule = $this->db->query("
                SELECT day_of_week, slot_1_open, slot_1_close, slot_2_open, slot_2_close, is_closed
                FROM store_schedules 
                WHERE store_id = ? AND hour_type_id = ? 
                AND (effective_start_date IS NULL OR effective_start_date <= CURDATE())
                AND (effective_end_date IS NULL OR effective_end_date >= CURDATE())
                ORDER BY day_of_week
            ", [$storeId, $hourType['id']])->getResultArray();
            
            // Organize by day of week (1-7)
            $schedules[$hourType['id']] = [];
            foreach ($weekSchedule as $day) {
                $schedules[$hourType['id']][$day['day_of_week']] = $day;
            }
            
            // Fill in missing days with defaults
            for ($day = 1; $day <= 7; $day++) {
                if (!isset($schedules[$hourType['id']][$day])) {
                    $schedules[$hourType['id']][$day] = [
                        'day_of_week' => $day,
                        'slot_1_open' => null,
                        'slot_1_close' => null,
                        'slot_2_open' => null,
                        'slot_2_close' => null,
                        'is_closed' => 0
                    ];
                }
            }
        }

        // Get recent exceptions/holidays
        $exceptions = $this->db->query("
            SELECT sse.*, sht.custom_name as hour_type_name
            FROM store_schedule_exceptions sse
            LEFT JOIN store_hour_types sht ON sse.hour_type_id = sht.id
            WHERE sse.store_id = ? AND sse.exception_date >= CURDATE() - INTERVAL 30 DAY
            ORDER BY sse.exception_date DESC
        ", [$storeId])->getResultArray();

        // Get saved templates
        $templates = $this->db->query("
            SELECT * FROM store_schedule_templates 
            WHERE store_id = ? 
            ORDER BY is_default DESC, template_name
        ", [$storeId])->getResultArray();

        $data = [
            'title' => 'Edit Hours: ' . $store['name'],
            'store' => $store,
            'globalTypes' => $globalTypes,
            'storeHourTypes' => $storeHourTypes,
            'schedules' => $schedules,
            'exceptions' => $exceptions,
            'templates' => $templates,
            'days' => [
                1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday', 4 => 'Thursday',
                5 => 'Friday', 6 => 'Saturday', 7 => 'Sunday'
            ],
            'current_status' => $this->calculateCurrentStatus($storeId)
        ];

        return view('admin/hours/edit', $data);
    }

    /**
     * Add a new hour type to a business
     */
    public function addHourType()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        $storeId = $this->request->getPost('store_id');
        $globalTypeId = $this->request->getPost('global_type_id');
        $customName = $this->request->getPost('custom_name');
        $affectsStatus = $this->request->getPost('affects_main_status') ? 1 : 0;

        if (!$storeId || !$customName) {
            return $this->response->setJSON(['success' => false, 'message' => 'Missing required fields']);
        }

        try {
            // Get next display order
            $maxOrder = $this->db->query("
                SELECT COALESCE(MAX(display_order), 0) + 1 as next_order 
                FROM store_hour_types WHERE store_id = ?
            ", [$storeId])->getRowArray();

            // Insert new hour type
            $this->db->query("
                INSERT INTO store_hour_types (store_id, global_type_id, custom_name, display_order, affects_main_status) 
                VALUES (?, ?, ?, ?, ?)
            ", [$storeId, $globalTypeId, $customName, $maxOrder['next_order'], $affectsStatus]);

            $hourTypeId = $this->db->insertID();

            // Create default closed schedule for all days
            for ($day = 1; $day <= 7; $day++) {
                $this->db->query("
                    INSERT INTO store_schedules (store_id, hour_type_id, day_of_week, is_closed) 
                    VALUES (?, ?, ?, 1)
                ", [$storeId, $hourTypeId, $day]);
            }

            return $this->response->setJSON([
                'success' => true, 
                'message' => 'Hour type added successfully',
                'hour_type_id' => $hourTypeId
            ]);

        } catch (\Exception $e) {
            return $this->response->setJSON(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        }
    }

    /**
     * Save schedule data for a specific hour type
     */
    public function saveSchedule()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        $storeId = $this->request->getPost('store_id');
        $hourTypeId = $this->request->getPost('hour_type_id');
        $scheduleData = $this->request->getPost('schedule');

        if (!$storeId || !$hourTypeId || !$scheduleData) {
            return redirect()->back()->with('error', 'Missing required data');
        }

        try {
            // Validate hour type belongs to store
            $hourType = $this->db->query("
                SELECT id FROM store_hour_types 
                WHERE id = ? AND store_id = ?
            ", [$hourTypeId, $storeId])->getRowArray();

            if (!$hourType) {
                return redirect()->back()->with('error', 'Invalid hour type');
            }

            // Save schedule for each day
            foreach ($scheduleData as $day => $dayData) {
                if (!is_numeric($day) || $day < 1 || $day > 7) continue;

                // Clean up time inputs
                $slot1Open = !empty($dayData['slot_1_open']) ? $dayData['slot_1_open'] : null;
                $slot1Close = !empty($dayData['slot_1_close']) ? $dayData['slot_1_close'] : null;
                $slot2Open = !empty($dayData['slot_2_open']) ? $dayData['slot_2_open'] : null;
                $slot2Close = !empty($dayData['slot_2_close']) ? $dayData['slot_2_close'] : null;
                $isClosed = isset($dayData['is_closed']) ? 1 : 0;

                // Update or insert schedule
                $existing = $this->db->query("
                    SELECT id FROM store_schedules 
                    WHERE store_id = ? AND hour_type_id = ? AND day_of_week = ?
                    AND (effective_start_date IS NULL OR effective_start_date <= CURDATE())
                    AND (effective_end_date IS NULL OR effective_end_date >= CURDATE())
                ", [$storeId, $hourTypeId, $day])->getRowArray();

                if ($existing) {
                    $this->db->query("
                        UPDATE store_schedules SET 
                            slot_1_open = ?, slot_1_close = ?, 
                            slot_2_open = ?, slot_2_close = ?, 
                            is_closed = ?, updated_at = NOW()
                        WHERE id = ?
                    ", [$slot1Open, $slot1Close, $slot2Open, $slot2Close, $isClosed, $existing['id']]);
                } else {
                    $this->db->query("
                        INSERT INTO store_schedules 
                        (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, slot_2_open, slot_2_close, is_closed) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    ", [$storeId, $hourTypeId, $day, $slot1Open, $slot1Close, $slot2Open, $slot2Close, $isClosed]);
                }
            }

            return redirect()->to('/admin/hours/store/' . $storeId)->with('success', 'Schedule saved successfully!');

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to save schedule: ' . $e->getMessage());
        }
    }

    /**
     * Add holiday/exception override
     */
    public function addException()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        $storeId = $this->request->getPost('store_id');
        $date = $this->request->getPost('exception_date');
        $type = $this->request->getPost('exception_type');
        $message = $this->request->getPost('message');
        $hourTypeId = $this->request->getPost('hour_type_id') ?: null;

        try {
            $this->db->query("
                INSERT INTO store_schedule_exceptions (store_id, exception_date, hour_type_id, exception_type, message) 
                VALUES (?, ?, ?, ?, ?)
            ", [$storeId, $date, $hourTypeId, $type, $message]);

            return $this->response->setJSON(['success' => true, 'message' => 'Exception added successfully']);

        } catch (\Exception $e) {
            return $this->response->setJSON(['success' => false, 'message' => 'Failed to add exception: ' . $e->getMessage()]);
        }
    }

    /**
     * Calculate current open/closed status for a business
     */
    private function calculateCurrentStatus($storeId)
    {
        // Get business timezone from existing store_timezone field
        $storeData = $this->db->query("
            SELECT s.name, s.store_timezone, tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York'; // Default to Eastern
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            // Fallback to Eastern if invalid timezone
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
            error_log("Invalid timezone '$businessTimezone' for store $storeId, using Eastern Time");
        }
        
        $dayOfWeek = $now->format('N'); // 1 = Monday, 7 = Sunday
        $currentTime = $now->format('H:i:s');
        $currentDate = $now->format('Y-m-d');
        
        // Debug output
        error_log("Status Check - Store: $storeId ({$storeData['name']}) | TZ: $businessTimezone | Time: " . $now->format('Y-m-d H:i:s T') . " | Day: $dayOfWeek");

        // Check for exceptions first
        $exception = $this->db->query("
            SELECT * FROM store_schedule_exceptions 
            WHERE store_id = ? AND exception_date = ? 
            ORDER BY hour_type_id IS NULL DESC
            LIMIT 1
        ", [$storeId, $currentDate])->getRowArray();

        if ($exception && $exception['exception_type'] === 'closed') {
            return ['status' => 'closed', 'message' => $exception['message'] ?? 'Closed today'];
        }

        // Get all hour types that affect main status
        $statusTypes = $this->db->query("
            SELECT sht.id, sht.custom_name,
                   ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
            AND ss.day_of_week = ?
            AND (ss.effective_start_date IS NULL OR ss.effective_start_date <= ?)
            AND (ss.effective_end_date IS NULL OR ss.effective_end_date >= ?)
        ", [$storeId, $dayOfWeek, $currentDate, $currentDate])->getResultArray();

        $openServices = [];
        $closingSoon = [];

        foreach ($statusTypes as $type) {
            if ($type['is_closed']) continue;

            $isOpen = false;
            $closingSoonCheck = false;

            // Check slot 1 with midnight handling
            if ($type['slot_1_open'] && $type['slot_1_close']) {
                $openTime = $type['slot_1_open'];
                $closeTime = $type['slot_1_close'];
                
                // Handle cross-midnight schedules (e.g., 18:00 - 03:00)
                if ($closeTime <= $openTime) {
                    // Crosses midnight: open time to 23:59:59 OR 00:00:00 to close time
                    if ($currentTime >= $openTime || $currentTime <= $closeTime) {
                        $isOpen = true;
                        // Check if closing within 30 minutes
                        if ($currentTime >= $openTime) {
                            // Still same day, check if close to 23:59:59 or close time
                            $minutesToMidnight = (23 * 60 + 59) - (intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2)));
                            if ($minutesToMidnight <= 30) {
                                $closingSoonCheck = true;
                            }
                        } else {
                            // Next day, check if close to actual close time
                            $closeTimeMinutes = intval(substr($closeTime, 0, 2)) * 60 + intval(substr($closeTime, 3, 2));
                            $currentTimeMinutes = intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2));
                            if ($closeTimeMinutes - $currentTimeMinutes <= 30) {
                                $closingSoonCheck = true;
                            }
                        }
                    }
                } else {
                    // Normal same-day schedule
                    if ($currentTime >= $openTime && $currentTime <= $closeTime) {
                        $isOpen = true;
                        // Check if closing within 30 minutes
                        $closeTimeMinutes = intval(substr($closeTime, 0, 2)) * 60 + intval(substr($closeTime, 3, 2));
                        $currentTimeMinutes = intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2));
                        if ($closeTimeMinutes - $currentTimeMinutes <= 30) {
                            $closingSoonCheck = true;
                        }
                    }
                }
            }

            // Check slot 2 with midnight handling
            if (!$isOpen && $type['slot_2_open'] && $type['slot_2_close']) {
                $openTime = $type['slot_2_open'];
                $closeTime = $type['slot_2_close'];
                
                // Handle cross-midnight schedules
                if ($closeTime <= $openTime) {
                    // Crosses midnight
                    if ($currentTime >= $openTime || $currentTime <= $closeTime) {
                        $isOpen = true;
                        // Check if closing within 30 minutes (similar logic as slot 1)
                        if ($currentTime >= $openTime) {
                            $minutesToMidnight = (23 * 60 + 59) - (intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2)));
                            if ($minutesToMidnight <= 30) {
                                $closingSoonCheck = true;
                            }
                        } else {
                            $closeTimeMinutes = intval(substr($closeTime, 0, 2)) * 60 + intval(substr($closeTime, 3, 2));
                            $currentTimeMinutes = intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2));
                            if ($closeTimeMinutes - $currentTimeMinutes <= 30) {
                                $closingSoonCheck = true;
                            }
                        }
                    }
                } else {
                    // Normal same-day schedule
                    if ($currentTime >= $openTime && $currentTime <= $closeTime) {
                        $isOpen = true;
                        $closeTimeMinutes = intval(substr($closeTime, 0, 2)) * 60 + intval(substr($closeTime, 3, 2));
                        $currentTimeMinutes = intval(substr($currentTime, 0, 2)) * 60 + intval(substr($currentTime, 3, 2));
                        if ($closeTimeMinutes - $currentTimeMinutes <= 30) {
                            $closingSoonCheck = true;
                        }
                    }
                }
            }

            if ($isOpen) {
                $openServices[] = $type['custom_name'];
                if ($closingSoonCheck) {
                    $closingSoon[] = $type['custom_name'];
                }
            }
        }

        if (empty($openServices)) {
            return ['status' => 'closed', 'services' => []];
        }

        if (!empty($closingSoon)) {
            return ['status' => 'closing_soon', 'services' => $openServices, 'closing_soon' => $closingSoon];
        }

        return ['status' => 'open', 'services' => $openServices];
    }

    /**
     * Generate status badge HTML
     */
    private function getStatusBadge($statusData)
    {
        $status = $statusData['status'];
        $services = $statusData['services'] ?? [];

        switch ($status) {
            case 'open':
                $badge = '<span class="badge bg-success">Open</span>';
                if (!empty($services)) {
                    $badge .= ' <small class="text-muted">(' . implode(', ', $services) . ')</small>';
                }
                return $badge;

            case 'closing_soon':
                $badge = '<span class="badge bg-warning">Closing Soon</span>';
                if (!empty($services)) {
                    $badge .= ' <small class="text-muted">(' . implode(', ', $services) . ')</small>';
                }
                return $badge;

            case 'closed':
            default:
                $message = $statusData['message'] ?? 'Closed';
                return '<span class="badge bg-danger">Closed</span> <small class="text-muted">(' . $message . ')</small>';
        }
    }

    /**
     * Quick status API for frontend - includes timezone debug info
     */
    public function apiStatus($storeId)
    {
        // Get business timezone from existing store_timezone field  
        $storeData = $this->db->query("
            SELECT s.name, s.store_timezone, tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }
        
        $status = $this->calculateCurrentStatus($storeId);
        
        // Add comprehensive debug information
        $status['debug'] = [
            'business_name' => $storeData['name'],
            'timezone_id' => $storeData['store_timezone'],
            'business_timezone' => $businessTimezone,
            'server_time_utc' => gmdate('Y-m-d H:i:s') . ' UTC',
            'business_local_time' => $now->format('Y-m-d H:i:s T'),
            'day_of_week' => $now->format('N') . ' (' . $now->format('l') . ')',
            'current_time' => $now->format('H:i:s'),
            'timezone_offset' => $now->format('P')
        ];
        
        return $this->response->setJSON($status);
    }

    /**
     * Bulk operations for multiple stores
     */
    public function bulkUpdate()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        $action = $this->request->getPost('bulk_action');
        $storeIds = $this->request->getPost('store_ids');

        if (!$action || empty($storeIds)) {
            return redirect()->back()->with('error', 'No action or stores selected');
        }

        try {
            switch ($action) {
                case 'add_regular_hours':
                    foreach ($storeIds as $storeId) {
                        $this->addDefaultHours($storeId);
                    }
                    $message = 'Regular hours added to selected businesses';
                    break;

                case 'copy_template':
                    $templateId = $this->request->getPost('template_id');
                    if ($templateId) {
                        foreach ($storeIds as $storeId) {
                            $this->applyTemplate($storeId, $templateId);
                        }
                        $message = 'Template applied to selected businesses';
                    } else {
                        throw new \Exception('No template selected');
                    }
                    break;

                default:
                    throw new \Exception('Unknown bulk action');
            }

            return redirect()->back()->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Bulk operation failed: ' . $e->getMessage());
        }
    }

    /**
     * Add default "Regular Hours" to a business
     */
    private function addDefaultHours($storeId)
    {
        // Check if already has regular hours
        $existing = $this->db->query("
            SELECT id FROM store_hour_types 
            WHERE store_id = ? AND custom_name = 'Regular Hours'
        ", [$storeId])->getRowArray();

        if ($existing) return;

        // Add regular hours type
        $this->db->query("
            INSERT INTO store_hour_types (store_id, global_type_id, custom_name, display_order, affects_main_status) 
            VALUES (?, 1, 'Regular Hours', 1, 1)
        ", [$storeId]);

        $hourTypeId = $this->db->insertID();

        // Add default 9-5 weekday schedule
        for ($day = 1; $day <= 7; $day++) {
            $isWeekday = $day <= 5;
            $this->db->query("
                INSERT INTO store_schedules (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, is_closed) 
                VALUES (?, ?, ?, ?, ?, ?)
            ", [
                $storeId, 
                $hourTypeId, 
                $day, 
                $isWeekday ? '09:00:00' : null,
                $isWeekday ? '17:00:00' : null,
                $isWeekday ? 0 : 1
            ]);
        }
    }
}