<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\StoreModel;
use App\Models\CategoryModel;

class Dashboard extends BaseController
{
    public function index()
    {
        // Check admin authentication
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        // Load models
        $storeModel = new StoreModel();
        $categoryModel = new CategoryModel();

        try {
            $db = \Config\Database::connect();

            // Store statistics
            $storeCount = $storeModel->countAllResults();
            $activeStores = $storeModel->where('status', 1)->countAllResults();
            $closedStores = $storeModel->where('is_closed', 1)->countAllResults();

            // Category statistics
            $categoryStats = $categoryModel->getCategoryStats();

            // Hours coverage - stores that have hours configured
            $storesWithHours = $db->query("
                SELECT COUNT(DISTINCT s.id) as count
                FROM store s
                INNER JOIN store_hour_types sht ON s.store_id = sht.store_id
                WHERE s.status = 1 AND sht.is_active = 1
            ")->getRowArray();

            // Recent activity - last 10 updated stores
            $recentUpdates = $db->query("
                SELECT s.id, s.name, s.city, s.state, s.update_date,
                       COUNT(sc.category_id) as category_count
                FROM store s
                LEFT JOIN store_category sc ON s.store_id = sc.store_id
                WHERE s.status = 1
                GROUP BY s.id, s.name, s.city, s.state, s.update_date
                ORDER BY s.update_date DESC
                LIMIT 10
            ")->getResultArray();

            // System health checks
            $systemHealth = [
                'stores_without_categories' => $categoryStats['stores_without_categories'],
                'stores_without_hours' => $activeStores - ($storesWithHours['count'] ?? 0),
                'inactive_categories' => $categoryStats['total_categories'] - $categoryStats['active_categories'],
                'total_issues' => 0
            ];

            $systemHealth['total_issues'] = $systemHealth['stores_without_categories'] + 
                                           $systemHealth['stores_without_hours'] + 
                                           $systemHealth['inactive_categories'];

            // Quick stats for dashboard cards
            $dashboardStats = [
                'total_stores' => $storeCount,
                'active_stores' => $activeStores,
                'closed_stores' => $closedStores,
                'total_categories' => $categoryStats['total_categories'],
                'active_categories' => $categoryStats['active_categories'],
                'stores_with_categories' => $categoryStats['stores_with_categories'],
                'stores_with_hours' => $storesWithHours['count'] ?? 0,
                'category_coverage' => $categoryStats['category_coverage_percent']
            ];

            $data = [
                'title' => 'Admin Dashboard',
                'stats' => $dashboardStats,
                'categoryStats' => $categoryStats,
                'systemHealth' => $systemHealth,
                'recentUpdates' => $recentUpdates
            ];

            return view('admin/dashboard', $data);

        } catch (\Exception $e) {
            log_message('error', 'Dashboard error: ' . $e->getMessage());
            
            // Fallback data if database queries fail
            $data = [
                'title' => 'Admin Dashboard',
                'stats' => [
                    'total_stores' => $storeCount ?? 0,
                    'active_stores' => 0,
                    'closed_stores' => 0,
                    'total_categories' => 0,
                    'active_categories' => 0,
                    'stores_with_categories' => 0,
                    'stores_with_hours' => 0,
                    'category_coverage' => 0
                ],
                'categoryStats' => [
                    'total_categories' => 0,
                    'active_categories' => 0,
                    'stores_with_categories' => 0,
                    'stores_without_categories' => $storeCount ?? 0,
                    'category_coverage_percent' => 0
                ],
                'systemHealth' => [
                    'stores_without_categories' => $storeCount ?? 0,
                    'stores_without_hours' => $storeCount ?? 0,
                    'inactive_categories' => 0,
                    'total_issues' => $storeCount ?? 0
                ],
                'recentUpdates' => [],
                'error' => 'Unable to load complete statistics'
            ];

            return view('admin/dashboard', $data);
        }
    }
}