<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\CategoryModel;
use App\Models\StoreModel;

class Categories extends BaseController
{
    protected $categoryModel;
    protected $storeModel;
    protected $db;

    public function __construct()
    {
        $this->categoryModel = new CategoryModel();
        $this->storeModel = new StoreModel();
        $this->db = \Config\Database::connect();
    }

    /**
     * Category management dashboard
     */
    public function index()
    {
        // Check admin authentication
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            // Get all categories with store counts
            $categories = $this->categoryModel->getCategoriesWithStoreCounts();
            
            // Get category statistics
            $stats = $this->categoryModel->getCategoryStats();
            
            // Get search filters
            $search = $this->request->getGet('search') ?: '';
            $statusFilter = $this->request->getGet('status') ?: '';
            
            // Apply filters if needed
            if (!empty($search) || !empty($statusFilter)) {
                $categories = $this->filterCategories($categories, $search, $statusFilter);
            }

            $data = [
                'title' => 'Category Management',
                'categories' => $categories,
                'stats' => $stats,
                'search' => $search,
                'statusFilter' => $statusFilter,
                'totalCategories' => count($categories)
            ];

            return view('admin/categories/index', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Categories index error: ' . $e->getMessage());
            return redirect()->to('/admin/dashboard')->with('error', 'Error loading categories');
        }
    }

    /**
     * Create new category form
     */
    public function create()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        if ($this->request->getPost()) {
            // Handle form submission
            return $this->store();
        }

        $data = [
            'title' => 'Create New Category',
            'category' => null,
            'fontAwesomeIcons' => $this->getFontAwesomeIcons()
        ];

        return view('admin/categories/create', $data);
    }

    /**
     * Store new category
     */
    public function store()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        $validation = \Config\Services::validation();
        $validation->setRules([
            'title' => 'required|min_length[2]|max_length[255]',
            'icon' => 'permit_empty|max_length[255]',
            'status' => 'required|in_list[0,1]',
            'top_category' => 'permit_empty|in_list[0,1]'
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Please fix the validation errors and try again.');
        }

        try {
            $data = [
                'title' => trim($this->request->getPost('title')),
                'icon' => trim($this->request->getPost('icon')) ?: 'fa fa-tag',
                'image' => $this->request->getPost('image') ?: '',
                'type' => $this->request->getPost('type') ?: 0,
                'top_category' => $this->request->getPost('top_category') ? 1 : 0,
                'status' => $this->request->getPost('status') ?: 1
            ];

            $result = $this->categoryModel->createCategory($data);

            if ($result) {
                return redirect()->to('/admin/categories')
                               ->with('success', 'Category "' . $data['title'] . '" created successfully!');
            } else {
                throw new \Exception('Failed to create category');
            }

        } catch (\Exception $e) {
            log_message('error', 'Category creation failed: ' . $e->getMessage());
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Failed to create category: ' . $e->getMessage());
        }
    }

    /**
     * Edit category form
     */
    public function edit($id)
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        if ($this->request->getPost()) {
            // Handle form submission
            return $this->update($id);
        }

        try {
            $category = $this->categoryModel->find($id);
            
            if (!$category) {
                return redirect()->to('/admin/categories')->with('error', 'Category not found');
            }

            // Get stores assigned to this category
            $assignedStores = $this->categoryModel->getStoresByCategory($category['category_id']);

            $data = [
                'title' => 'Edit Category: ' . $category['title'],
                'category' => $category,
                'assignedStores' => $assignedStores,
                'fontAwesomeIcons' => $this->getFontAwesomeIcons()
            ];

            return view('admin/categories/create', $data); // Reuse create view
            
        } catch (\Exception $e) {
            log_message('error', 'Category edit error: ' . $e->getMessage());
            return redirect()->to('/admin/categories')->with('error', 'Error loading category');
        }
    }

    /**
     * Update category
     */
    public function update($id)
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        $validation = \Config\Services::validation();
        $validation->setRules([
            'title' => 'required|min_length[2]|max_length[255]',
            'icon' => 'permit_empty|max_length[255]',
            'status' => 'required|in_list[0,1]'
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Please fix the validation errors and try again.');
        }

        try {
            $category = $this->categoryModel->find($id);
            if (!$category) {
                return redirect()->to('/admin/categories')->with('error', 'Category not found');
            }

            $data = [
                'title' => trim($this->request->getPost('title')),
                'icon' => trim($this->request->getPost('icon')) ?: 'fa fa-tag',
                'image' => $this->request->getPost('image') ?: '',
                'type' => $this->request->getPost('type') ?: 0,
                'top_category' => $this->request->getPost('top_category') ? 1 : 0,
                'status' => $this->request->getPost('status') ?: 1
            ];

            $result = $this->categoryModel->update($id, $data);

            if ($result) {
                return redirect()->to('/admin/categories')
                               ->with('success', 'Category "' . $data['title'] . '" updated successfully!');
            } else {
                throw new \Exception('Failed to update category');
            }

        } catch (\Exception $e) {
            log_message('error', 'Category update failed: ' . $e->getMessage());
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Failed to update category: ' . $e->getMessage());
        }
    }

    /**
     * Delete category
     */
    public function delete($id)
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        try {
            $category = $this->categoryModel->find($id);
            if (!$category) {
                return $this->response->setJSON(['success' => false, 'message' => 'Category not found']);
            }

            // Check if category has assigned stores
            $assignedStores = $this->categoryModel->getStoresByCategory($category['category_id']);
            
            if (!empty($assignedStores)) {
                return $this->response->setJSON([
                    'success' => false, 
                    'message' => 'Cannot delete category with ' . count($assignedStores) . ' assigned stores. Remove store assignments first.'
                ]);
            }

            // Delete the category
            $result = $this->categoryModel->delete($id);

            if ($result) {
                return $this->response->setJSON([
                    'success' => true, 
                    'message' => 'Category "' . $category['title'] . '" deleted successfully'
                ]);
            } else {
                throw new \Exception('Failed to delete category');
            }

        } catch (\Exception $e) {
            log_message('error', 'Category deletion failed: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'message' => 'Error deleting category']);
        }
    }

    /**
     * Bulk operations on categories
     */
    public function bulkAction()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        $action = $this->request->getPost('bulk_action');
        $categoryIds = $this->request->getPost('category_ids');

        if (!$action || empty($categoryIds)) {
            return redirect()->back()->with('error', 'No action or categories selected');
        }

        try {
            $result = false;
            $message = '';

            switch ($action) {
                case 'activate':
                    $result = $this->categoryModel->bulkUpdateStatus($categoryIds, 1);
                    $message = count($categoryIds) . ' categories activated successfully';
                    break;

                case 'deactivate':
                    $result = $this->categoryModel->bulkUpdateStatus($categoryIds, 0);
                    $message = count($categoryIds) . ' categories deactivated successfully';
                    break;

                case 'delete':
                    $result = $this->bulkDeleteCategories($categoryIds);
                    $message = $result['message'];
                    $result = $result['success'];
                    break;

                default:
                    throw new \Exception('Unknown bulk action');
            }

            if ($result) {
                return redirect()->back()->with('success', $message);
            } else {
                throw new \Exception('Bulk operation failed');
            }

        } catch (\Exception $e) {
            log_message('error', 'Bulk action failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Bulk operation failed: ' . $e->getMessage());
        }
    }

    /**
     * Category assignment interface
     */
    public function assignments()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            // Get all categories
            $categories = $this->categoryModel->getActiveCategories();
            
            // Get stores with their categories
            $stores = $this->db->query("
                SELECT 
                    s.id,
                    s.store_id,
                    s.name,
                    s.city,
                    s.state,
                    GROUP_CONCAT(c.title ORDER BY c.title SEPARATOR ', ') as categories,
                    COUNT(sc.category_id) as category_count
                FROM store s
                LEFT JOIN store_category sc ON s.store_id = sc.store_id
                LEFT JOIN category c ON sc.category_id = c.category_id AND c.status = 1
                WHERE s.status = 1
                GROUP BY s.id, s.store_id, s.name, s.city, s.state
                ORDER BY s.name ASC
                LIMIT 100
            ")->getResultArray();

            $data = [
                'title' => 'Category Assignments',
                'categories' => $categories,
                'stores' => $stores,
                'totalStores' => count($stores)
            ];

            return view('admin/categories/assignments', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Category assignments error: ' . $e->getMessage());
            return redirect()->to('/admin/categories')->with('error', 'Error loading assignments');
        }
    }

    /**
     * Enhanced API endpoint for category data with live open counts
     * Supports both normal and open-only filtering
     */
    public function apiCategories()
    {
        // Allow public access for frontend use
        // if (!session()->get('admin_logged_in')) {
        //     return $this->response->setJSON(['error' => 'Not authorized']);
        // }

        try {
            // Get request parameters
            $openOnly = $this->request->getGet('open_only') === 'true';
            $coords = $this->request->getGet('coords'); // Optional lat,lng for distance filtering
            
            log_message('debug', "Categories API called - Open Only: " . ($openOnly ? 'YES' : 'NO'));
            
            // Parse coordinates if provided
            $hasCoords = false;
            $lat = null;
            $lng = null;
            
            if ($coords && strpos($coords, ',') !== false) {
                list($lat, $lng) = explode(',', $coords);
                $lat = floatval($lat);
                $lng = floatval($lng);
                $hasCoords = ($lat != 0 && $lng != 0);
            }
            
            $distanceCondition = $hasCoords ? 
                "AND (3959 * acos(cos(radians($lat)) * cos(radians(s.latitude)) * cos(radians(s.longitude) - radians($lng)) + sin(radians($lat)) * sin(radians(s.latitude)))) <= 100" : 
                "";
            
            if ($openOnly) {
                // Get categories with ONLY open business counts
                $sql = "
                    SELECT 
                        c.category_id,
                        c.title,
                        c.icon,
                        c.top_category,
                        c.status,
                        COUNT(DISTINCT s.store_id) as total_stores,
                        COUNT(DISTINCT CASE WHEN scs.status IN ('open', 'closing_soon') THEN s.store_id END) as open_stores
                    FROM category c
                    LEFT JOIN store_category sc ON c.category_id = sc.category_id
                    LEFT JOIN store s ON sc.store_id = s.store_id AND s.status = 1
                    LEFT JOIN store_current_status scs ON s.store_id = scs.store_id
                    WHERE c.status = 1
                    $distanceCondition
                    GROUP BY c.category_id, c.title, c.icon, c.top_category, c.status
                    HAVING total_stores > 0
                    ORDER BY c.top_category DESC, open_stores DESC, c.title ASC
                ";
            } else {
                // Get categories with total business counts
                $sql = "
                    SELECT 
                        c.category_id,
                        c.title,
                        c.icon,
                        c.top_category,
                        c.status,
                        COUNT(DISTINCT s.store_id) as total_stores,
                        COUNT(DISTINCT CASE WHEN scs.status IN ('open', 'closing_soon') THEN s.store_id END) as open_stores
                    FROM category c
                    LEFT JOIN store_category sc ON c.category_id = sc.category_id
                    LEFT JOIN store s ON sc.store_id = s.store_id AND s.status = 1
                    LEFT JOIN store_current_status scs ON s.store_id = scs.store_id
                    WHERE c.status = 1
                    $distanceCondition
                    GROUP BY c.category_id, c.title, c.icon, c.top_category, c.status
                    HAVING total_stores > 0
                    ORDER BY c.top_category DESC, total_stores DESC, c.title ASC
                ";
            }
            
            log_message('debug', "Categories SQL: " . $sql);
            
            $categories = $this->db->query($sql)->getResultArray();
            
            // Process categories for frontend
            $processedCategories = [];
            foreach ($categories as $category) {
                $totalStores = (int)$category['total_stores'];
                $openStores = (int)$category['open_stores'];
                
                // Determine availability status
                $isAvailable = $openOnly ? ($openStores > 0) : ($totalStores > 0);
                
                $processedCategories[] = [
                    'category_id' => $category['category_id'],
                    'title' => $category['title'],
                    'icon' => $category['icon'],
                    'top_category' => (int)$category['top_category'],
                    'total_stores' => $totalStores,
                    'open_stores' => $openStores,
                    'store_count' => $openOnly ? $openStores : $totalStores, // For backward compatibility
                    'is_available' => $isAvailable,
                    'status_text' => $this->generateStatusText($totalStores, $openStores, $openOnly)
                ];
            }
            
            // Filter to only top categories if we have too many
            $topCategories = array_filter($processedCategories, function($cat) {
                return $cat['top_category'] == 1;
            });
            
            // If we have 8+ top categories, use only those. Otherwise use all available.
            $finalCategories = (count($topCategories) >= 8) ? $topCategories : $processedCategories;
            
            // Limit to max 12 categories total for clean UX
            $finalCategories = array_slice($finalCategories, 0, 12);
            
            log_message('debug', "Returning " . count($finalCategories) . " categories");
            
            return $this->response->setJSON([
                'categories' => $finalCategories,
                'open_only_mode' => $openOnly,
                'has_coordinates' => $hasCoords,
                'total_categories' => count($finalCategories)
            ]);
            
        } catch (\Exception $e) {
            log_message('error', 'Categories API error: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Failed to load categories']);
        }
    }

    /**
     * Get top categories only (for focused category management)
     */
    public function apiTopCategories()
    {
        try {
            $categories = $this->categoryModel->where('status', 1)
                                             ->where('top_category', 1)
                                             ->orderBy('title', 'ASC')
                                             ->findAll();
            
            return $this->response->setJSON(['categories' => $categories]);
            
        } catch (\Exception $e) {
            log_message('error', 'Top categories API error: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Failed to load top categories']);
        }
    }

    /**
     * Suggest category consolidation based on business distribution
     */
    public function suggestConsolidation()
    {
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }
        
        try {
            // Find categories with low business counts
            $sql = "
                SELECT 
                    c.category_id,
                    c.title,
                    c.icon,
                    c.top_category,
                    COUNT(sc.store_id) as business_count
                FROM category c
                LEFT JOIN store_category sc ON c.category_id = sc.category_id
                WHERE c.status = 1
                GROUP BY c.category_id, c.title, c.icon, c.top_category
                ORDER BY business_count ASC, c.title ASC
            ";
            
            $categories = $this->db->query($sql)->getResultArray();
            
            $suggestions = [
                'candidates_for_removal' => array_filter($categories, function($cat) {
                    return $cat['business_count'] == 0;
                }),
                'candidates_for_merging' => array_filter($categories, function($cat) {
                    return $cat['business_count'] > 0 && $cat['business_count'] < 3;
                }),
                'strong_categories' => array_filter($categories, function($cat) {
                    return $cat['business_count'] >= 5;
                }),
                'total_categories' => count($categories)
            ];
            
            return $this->response->setJSON($suggestions);
            
        } catch (\Exception $e) {
            log_message('error', 'Consolidation suggestions error: ' . $e->getMessage());
            return $this->response->setJSON(['error' => 'Failed to generate suggestions']);
        }
    }

    /**
     * Get stores by category (public endpoint for frontend)
     */
    public function storesByCategory($categoryId)
    {
        try {
            $stores = $this->categoryModel->getStoresByCategory($categoryId);
            return $this->response->setJSON(['stores' => $stores]);
        } catch (\Exception $e) {
            return $this->response->setJSON(['error' => 'Failed to load stores']);
        }
    }

    /**
     * Generate user-friendly status text for categories
     */
    private function generateStatusText($totalStores, $openStores, $openOnly)
    {
        if ($openOnly) {
            if ($openStores == 0) {
                return "None open now";
            } else if ($openStores == 1) {
                return "1 open now";
            } else {
                return "$openStores open now";
            }
        } else {
            if ($totalStores == 1) {
                return "1 business";
            } else {
                return "$totalStores businesses";
            }
        }
    }

    /**
     * Filter categories based on search and status
     */
    private function filterCategories($categories, $search, $statusFilter)
    {
        return array_filter($categories, function($category) use ($search, $statusFilter) {
            $matchesSearch = empty($search) || 
                           stripos($category['title'], $search) !== false ||
                           stripos($category['icon'], $search) !== false;
            
            $matchesStatus = empty($statusFilter) || 
                           ($statusFilter === 'active' && $category['status'] == 1) ||
                           ($statusFilter === 'inactive' && $category['status'] == 0) ||
                           ($statusFilter === 'top' && $category['top_category'] == 1);
            
            return $matchesSearch && $matchesStatus;
        });
    }

    /**
     * Bulk delete categories with validation
     */
    private function bulkDeleteCategories($categoryIds)
    {
        $deletedCount = 0;
        $skippedCount = 0;
        
        foreach ($categoryIds as $categoryId) {
            // Check if category has assigned stores
            $storeCount = $this->db->query("
                SELECT COUNT(*) as count 
                FROM store_category sc
                INNER JOIN category c ON sc.category_id = c.category_id
                WHERE c.category_id = ?
            ", [$categoryId])->getRowArray();
            
            if ($storeCount['count'] > 0) {
                $skippedCount++;
                continue;
            }
            
            // Safe to delete
            $category = $this->categoryModel->where('category_id', $categoryId)->first();
            if ($category && $this->categoryModel->delete($category['id'])) {
                $deletedCount++;
            }
        }
        
        $message = "Deleted {$deletedCount} categories";
        if ($skippedCount > 0) {
            $message .= ", skipped {$skippedCount} categories with assigned stores";
        }
        
        return [
            'success' => $deletedCount > 0,
            'message' => $message
        ];
    }

    /**
     * Get Font Awesome icons for category selection
     */
    private function getFontAwesomeIcons()
    {
        return [
            'fas fa-home' => 'Home',
            'fas fa-building' => 'Building',
            'fas fa-store' => 'Store',
            'fas fa-cart-shopping' => 'Shopping Cart',
            'fas fa-utensils' => 'Restaurant',
            'fas fa-car' => 'Automotive',
            'fas fa-wrench' => 'Repair',
            'fas fa-kit-medical' => 'Healthcare',
            'fas fa-graduation-cap' => 'Education',
            'fas fa-money-bill' => 'Finance',
            'fas fa-gavel' => 'Legal',
            'fas fa-paintbrush' => 'Art & Design',
            'fas fa-music' => 'Entertainment',
            'fas fa-camera' => 'Photography',
            'fas fa-laptop' => 'Technology',
            'fas fa-leaf' => 'Garden & Landscaping',
            'fas fa-heart' => 'Health & Beauty',
            'fas fa-paw' => 'Pet Services',
            'fas fa-tag' => 'General',
            'fas fa-star' => 'Featured'
        ];
    }
}