<?php
/**
 * Simple Date Test Script
 * Save as: public/test_dates.php
 * Access at: https://ringgoldroad.com/test_dates.php
 */

echo "<h1>Date Comparison Test</h1>";
echo "<style>
    body { font-family: Arial, sans-serif; margin: 20px; }
    .error { color: red; font-weight: bold; }
    .success { color: green; font-weight: bold; }
    .info { color: blue; font-weight: bold; }
    table { border-collapse: collapse; width: 100%; margin: 20px 0; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background-color: #f2f2f2; }
</style>";

// Current time in Eastern timezone
$now = new DateTime('now', new DateTimeZone('America/New_York'));
echo "<p class='info'><strong>Current Eastern Time:</strong> " . $now->format('Y-m-d H:i:s T l') . "</p>";

// Test the problematic dates from your debug output
$testDates = [
    '2025-07-31 09:00:00',  // This is what's in your database
    '2025-07-31 08:00:00',
    '2025-07-31 10:00:00'
];

echo "<table>";
echo "<tr><th>Database DateTime</th><th>Parsed Date</th><th>Today's Date</th><th>Comparison</th><th>Should Show</th><th>Current Logic Shows</th></tr>";

foreach ($testDates as $testDate) {
    // Parse the datetime from database
    $nextOpen = new DateTime($testDate);
    
    // Get dates for comparison
    $nextOpenDate = $nextOpen->format('Y-m-d');
    $todayDate = $now->format('Y-m-d');
    
    // Current (buggy) logic comparison
    $diff = $now->diff($nextOpen);
    $daysDiff = $diff->days;
    
    // What should it show?
    $shouldShow = "";
    $currentLogicShows = "";
    
    if ($nextOpenDate == $todayDate) {
        $shouldShow = "Opens today at " . $nextOpen->format('g:i A');
    } else {
        // Calculate days properly
        $nextDateObj = new DateTime($nextOpenDate);
        $todayDateObj = new DateTime($todayDate);
        $properDiff = $todayDateObj->diff($nextDateObj);
        
        if ($properDiff->days == 1 && $nextDateObj > $todayDateObj) {
            $shouldShow = "Opens tomorrow at " . $nextOpen->format('g:i A');
        } else {
            $shouldShow = "Opens " . $nextOpen->format('l \a\t g:i A');
        }
    }
    
    // What current logic might show (this is the bug)
    if ($diff->days == 0) {
        $currentLogicShows = "Opens today at " . $nextOpen->format('g:i A');
    } elseif ($diff->days == 1) {
        $currentLogicShows = "Opens tomorrow at " . $nextOpen->format('g:i A');
    } else {
        $currentLogicShows = "Opens " . $nextOpen->format('l \a\t g:i A');
    }
    
    // Determine row color
    $rowClass = ($shouldShow != $currentLogicShows) ? 'style="background-color: #ffcccc;"' : '';
    
    echo "<tr $rowClass>";
    echo "<td>" . $testDate . "</td>";
    echo "<td>" . $nextOpenDate . "</td>";
    echo "<td>" . $todayDate . "</td>";
    echo "<td>" . ($nextOpenDate == $todayDate ? "SAME" : "DIFFERENT") . "</td>";
    echo "<td class='success'>" . $shouldShow . "</td>";
    echo "<td class='" . ($shouldShow != $currentLogicShows ? "error" : "success") . "'>" . $currentLogicShows . "</td>";
    echo "</tr>";
}

echo "</table>";

echo "<h2>Detailed Analysis</h2>";

// Show what the current DateTime->diff() method is doing
$testDateTime = new DateTime('2025-07-31 09:00:00');
$diff = $now->diff($testDateTime);

echo "<p><strong>Testing DateTime('2025-07-31 09:00:00') vs now:</strong></p>";
echo "<ul>";
echo "<li>DateTime->diff()->days: " . $diff->days . "</li>";
echo "<li>DateTime->diff()->h: " . $diff->h . "</li>";
echo "<li>DateTime->diff()->invert: " . $diff->invert . "</li>";
echo "<li>Formatted: " . $diff->format('%r%a days, %h hours') . "</li>";
echo "</ul>";

echo "<h2>The Root Cause</h2>";
echo "<p class='error'><strong>Problem:</strong> The DateTime->diff() method calculates the difference between 'July 30 7:32 PM' and 'July 31 9:00 AM' as 0 days because it's less than 24 hours apart.</p>";
echo "<p class='success'><strong>Solution:</strong> Compare only the DATE parts (Y-m-d), not the full datetime objects.</p>";

echo "<h2>Fixed Logic Test</h2>";

// Test the fixed logic
echo "<table>";
echo "<tr><th>Database DateTime</th><th>Fixed Logic Result</th><th>Status</th></tr>";

foreach ($testDates as $testDate) {
    $nextOpen = new DateTime($testDate);
    
    // FIXED LOGIC: Compare only dates
    $nextOpenDate = $nextOpen->format('Y-m-d');
    $todayDate = $now->format('Y-m-d');
    
    if ($nextOpenDate == $todayDate) {
        $result = "Opens today at " . $nextOpen->format('g:i A');
        $status = "error"; // This should not happen with your data
    } else {
        // Calculate days difference using date objects only
        $nextDateObj = new DateTime($nextOpenDate);
        $todayDateObj = new DateTime($todayDate);
        $interval = $todayDateObj->diff($nextDateObj);
        
        if ($interval->days == 1) {
            $result = "Opens tomorrow at " . $nextOpen->format('g:i A');
            $status = "success";
        } else {
            $result = "Opens " . $nextOpen->format('l \a\t g:i A');
            $status = "success";
        }
    }
    
    echo "<tr>";
    echo "<td>" . $testDate . "</td>";
    echo "<td class='$status'>" . $result . "</td>";
    echo "<td>" . ($status == "success" ? "✓ CORRECT" : "✗ WRONG") . "</td>";
    echo "</tr>";
}

echo "</table>";

echo "<p><em>Test completed at: " . $now->format('Y-m-d H:i:s T') . "</em></p>";
?>