<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Debug Tester</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .debug-container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.15);
        }
        
        h1 {
            text-align: center;
            color: #333;
            margin-bottom: 30px;
            font-size: 2rem;
        }
        
        .action-controls {
            text-align: center;
            margin-bottom: 30px;
            padding: 25px;
            background: linear-gradient(135deg, #e8f5e8 0%, #d4edda 100%);
            border-radius: 15px;
            border: 3px solid #28a745;
            box-shadow: 0 5px 15px rgba(40, 167, 69, 0.2);
        }
        
        .action-controls h2 {
            color: #155724;
            margin-bottom: 20px;
            font-size: 1.3rem;
        }
        
        .test-section {
            margin-bottom: 40px;
            border: 2px solid #e9ecef;
            border-radius: 15px;
            padding: 25px;
            background: #f8f9fa;
        }
        
        .test-section h2 {
            color: #667eea;
            margin-bottom: 20px;
            font-size: 1.4rem;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .test-form {
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
            margin-bottom: 20px;
        }
        
        .test-input {
            padding: 12px 16px;
            border: 2px solid #ddd;
            border-radius: 10px;
            font-size: 1rem;
            min-width: 200px;
            transition: all 0.3s ease;
        }
        
        .test-input:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .test-button {
            padding: 12px 24px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-size: 1rem;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .test-button:hover {
            background: #5a6fd8;
            transform: translateY(-2px);
        }
        
        .action-button {
            padding: 15px 30px;
            margin: 0 8px;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            font-size: 1.1rem;
            font-weight: 700;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .export-btn {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
        }
        
        .export-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
        }
        
        .run-all-btn {
            background: linear-gradient(135deg, #17a2b8 0%, #007bff 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(23, 162, 184, 0.3);
        }
        
        .run-all-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(23, 162, 184, 0.4);
        }
        
        .clear-btn {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
        }
        
        .clear-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(220, 53, 69, 0.4);
        }
        
        .debug-output {
            background: #1e1e1e;
            color: #00ff00;
            font-family: 'Courier New', monospace;
            padding: 20px;
            border-radius: 10px;
            max-height: 300px;
            overflow-y: auto;
            font-size: 0.9rem;
            line-height: 1.4;
            white-space: pre-wrap;
        }
        
        .debug-output .error {
            color: #ff6b6b;
        }
        
        .debug-output .warning {
            color: #ffc107;
        }
        
        .debug-output .success {
            color: #28a745;
        }
        
        .debug-output .info {
            color: #17a2b8;
        }
        
        .zip-digits {
            display: flex;
            gap: 8px;
        }
        
        .zip-digit {
            width: 40px;
            height: 40px;
            text-align: center;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 1.2rem;
            font-weight: bold;
        }
        
        .zip-digit:focus {
            border-color: #667eea;
        }
        
        .clear-all {
            background: #dc3545;
            margin-left: auto;
        }
        
        .clear-all:hover {
            background: #c82333;
        }
        
        .status-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }
        
        .status-idle { background: #6c757d; }
        .status-testing { background: #ffc107; animation: pulse 1s infinite; }
        .status-success { background: #28a745; }
        .status-error { background: #dc3545; }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .comparison-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        
        .comparison-card {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 20px;
        }
        
        .comparison-card h3 {
            color: #495057;
            margin-bottom: 15px;
            font-size: 1.1rem;
        }
        
        .data-field {
            margin-bottom: 10px;
            padding: 8px 12px;
            background: #f8f9fa;
            border-radius: 6px;
            font-family: monospace;
            font-size: 0.9rem;
        }
        
        .data-field strong {
            color: #495057;
        }
        
        .data-field.different {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
        }
    </style>
</head>
<body>
    <div class="debug-container">
        <h1>🔍 Three-Tab Search Debug Tester</h1>
        
        <!-- PROMINENT ACTION CONTROLS -->
        <div class="action-controls">
            <h2>🚀 Quick Actions & Export Controls</h2>
            <button class="action-button export-btn" onclick="exportDebugLog()">
                📄 Export Debug Report
            </button>
            <button class="action-button run-all-btn" onclick="runAllTests()">
                🚀 Run All Tests
            </button>
            <button class="action-button clear-btn" onclick="clearAllLogs()">
                🗑️ Clear All
            </button>
        </div>
        
        <!-- ZIP Code Search Test -->
        <div class="test-section">
            <h2>
                <span class="status-indicator status-idle" id="zip-status"></span>
                📍 ZIP Code Search Test
            </h2>
            <div class="test-form">
                <div class="zip-digits">
                    <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" id="zip-1" placeholder="0">
                    <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" id="zip-2" placeholder="0">
                    <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" id="zip-3" placeholder="0">
                    <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" id="zip-4" placeholder="0">
                    <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" id="zip-5" placeholder="0">
                </div>
                <button class="test-button" onclick="testZipSearch()">Test ZIP Search</button>
                <button class="test-button clear-all" onclick="clearOutput('zip-output')">Clear</button>
            </div>
            <div class="debug-output" id="zip-output">Ready to test ZIP code search...\n</div>
        </div>
        
        <!-- Address Search Test -->
        <div class="test-section">
            <h2>
                <span class="status-indicator status-idle" id="address-status"></span>
                🏠 Address Search Test
            </h2>
            <div class="test-form">
                <input type="text" class="test-input" id="address-input" placeholder="Enter address (e.g., 123 Main St, Nashville, TN)" value="123 Main St, Nashville, TN">
                <button class="test-button" onclick="testAddressSearch()">Test Address Search</button>
                <button class="test-button clear-all" onclick="clearOutput('address-output')">Clear</button>
            </div>
            <div class="debug-output" id="address-output">Ready to test address search...\n</div>
        </div>
        
        <!-- Keyword Search Test -->
        <div class="test-section">
            <h2>
                <span class="status-indicator status-idle" id="keyword-status"></span>
                🔎 Keyword Search Test
            </h2>
            <div class="test-form">
                <input type="text" class="test-input" id="keyword-input" placeholder="Enter keyword (e.g., plumber, electrician, HVAC)" value="plumber">
                <button class="test-button" onclick="testKeywordSearch()">Test Keyword Search</button>
                <button class="test-button clear-all" onclick="clearOutput('keyword-output')">Clear</button>
            </div>
            <div class="debug-output" id="keyword-output">Ready to test keyword search...\n</div>
        </div>
        
        <!-- Comparison Analysis -->
        <div class="comparison-grid" id="comparison-grid" style="display: none;">
            <div class="comparison-card">
                <h3>ZIP Search Data</h3>
                <div id="zip-data"></div>
            </div>
            <div class="comparison-card">
                <h3>Address Search Data</h3>
                <div id="address-data"></div>
            </div>
            <div class="comparison-card">
                <h3>Keyword Search Data</h3>
                <div id="keyword-data"></div>
            </div>
        </div>
    </div>

    <script>
        // Global test results storage
        let testResults = {
            zip: null,
            address: null,
            keyword: null
        };

        // Hidden form fields simulation
        let formFields = {
            latitude: '',
            longitude: '',
            keyword: '',
            zipcode: '',
            selected_category: '',
            search_mode: ''
        };

        // Global debug log for export
        let debugLog = [];
        let actualResults = {
            zip: null,
            address: null,
            keyword: null
        };

        function addToDebugLog(section, message, type = 'info') {
            const timestamp = new Date().toISOString();
            debugLog.push({
                timestamp,
                section,
                type,
                message
            });
        }

        // Mock the current system's functions for testing
        function updateLocationStatus(message, type) {
            return { message, type, timestamp: new Date().toISOString() };
        }

        function setStatus(type, status) {
            const indicator = document.getElementById(`${type}-status`);
            indicator.className = `status-indicator status-${status}`;
        }

        function log(outputId, message, type = 'info') {
            const output = document.getElementById(outputId);
            const timestamp = new Date().toLocaleTimeString();
            const className = type;
            
            // Add to visible output
            output.innerHTML += `<span class="${className}">[${timestamp}] ${message}</span>\n`;
            output.scrollTop = output.scrollHeight;
            
            // Add to debug log for export
            const section = outputId.replace('-output', '').toUpperCase();
            addToDebugLog(section, message, type);
        }

        function clearOutput(outputId) {
            document.getElementById(outputId).innerHTML = 'Cleared...\n';
        }

        // ZIP Code Search Test
        async function testZipSearch() {
            setStatus('zip', 'testing');
            log('zip-output', '=== STARTING ZIP CODE SEARCH TEST ===', 'info');
            
            try {
                // Collect ZIP digits
                const zipDigits = [1,2,3,4,5].map(i => document.getElementById(`zip-${i}`).value);
                const zipCode = zipDigits.join('');
                
                log('zip-output', `Input ZIP: "${zipCode}"`, 'info');
                
                if (zipCode.length !== 5) {
                    log('zip-output', 'VALIDATION: ZIP code must be 5 digits', 'error');
                    setStatus('zip', 'error');
                    return;
                }

                // Simulate form field updates
                formFields.zipcode = zipCode;
                formFields.search_mode = 'zip';
                formFields.keyword = ''; // Clear keyword for location search
                
                log('zip-output', `Form fields updated:`, 'success');
                log('zip-output', `  - zipcode: "${formFields.zipcode}"`, 'success');
                log('zip-output', `  - search_mode: "${formFields.search_mode}"`, 'success');
                log('zip-output', `  - keyword: "${formFields.keyword}" (cleared)`, 'success');

                // Simulate geocoding
                log('zip-output', 'Calling geocoding API...', 'info');
                const locationStatus = updateLocationStatus(`Finding businesses near ZIP ${zipCode}...`, 'default');
                log('zip-output', `Location status: ${locationStatus.message}`, 'info');

                // Mock geocoding response
                await new Promise(resolve => setTimeout(resolve, 1000));
                
                const mockCoords = { lat: 36.1627, lng: -86.7816 }; // Nashville
                formFields.latitude = mockCoords.lat;
                formFields.longitude = mockCoords.lng;
                
                log('zip-output', `Geocoding successful:`, 'success');
                log('zip-output', `  - latitude: ${formFields.latitude}`, 'success');
                log('zip-output', `  - longitude: ${formFields.longitude}`, 'success');

                // Simulate search request and make ACTUAL API CALL
                const searchData = {
                    keyword: formFields.keyword,
                    location: '',
                    category: formFields.selected_category,
                    coords: { lat: formFields.latitude, lng: formFields.longitude },
                    page: 1,
                    open_only: false,
                    search_mode: 'zip'
                };

                log('zip-output', 'Search payload:', 'info');
                log('zip-output', JSON.stringify(searchData, null, 2), 'info');

                // MAKE ACTUAL API CALL
                log('zip-output', 'Making REAL API call to /ajax/locator...', 'warning');
                try {
                    const response = await fetch('/ajax/locator', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(searchData)
                    });

                    if (response.ok) {
                        const htmlResponse = await response.text();
                        const results = parseSearchResults(htmlResponse);
                        
                        actualResults.zip = results;
                        
                        log('zip-output', `API Response received: ${results.businesses.length} businesses found`, 'success');
                        log('zip-output', `Sample businesses: ${results.businesses.slice(0, 3).map(b => b.name).join(', ')}`, 'info');
                        
                        // Log search result details
                        addToDebugLog('ZIP_API_RESULTS', `Total businesses returned: ${results.businesses.length}`, 'info');
                        addToDebugLog('ZIP_API_RESULTS', `Metadata: ${JSON.stringify(results.metadata)}`, 'info');
                        if (results.businesses.length > 0) {
                            addToDebugLog('ZIP_API_RESULTS', `First 5 business names: ${results.businesses.slice(0, 5).map(b => b.name).join(', ')}`, 'info');
                        }
                    } else {
                        log('zip-output', `API call failed: ${response.status} ${response.statusText}`, 'error');
                        addToDebugLog('ZIP_API_RESULTS', `API call failed: ${response.status}`, 'error');
                    }
                } catch (apiError) {
                    log('zip-output', `API Error: ${apiError.message}`, 'error');
                    addToDebugLog('ZIP_API_RESULTS', `API Error: ${apiError.message}`, 'error');
                }

                testResults.zip = {
                    success: true,
                    formFields: { ...formFields },
                    searchData: { ...searchData },
                    timestamp: new Date()
                };

                // Add comprehensive summary to debug log
                addToDebugLog('ZIP_SUMMARY', '=== ZIP SEARCH COMPLETE ===', 'success');
                addToDebugLog('ZIP_SUMMARY', `Input: ${zipCode}`, 'info');
                addToDebugLog('ZIP_SUMMARY', `Final form_keyword: "${formFields.keyword}"`, 'info');
                addToDebugLog('ZIP_SUMMARY', `Final coordinates: ${formFields.latitude}, ${formFields.longitude}`, 'info');
                addToDebugLog('ZIP_SUMMARY', `Search payload keyword: "${searchData.keyword}"`, 'info');
                addToDebugLog('ZIP_SUMMARY', `Search path: Location-based (coordinates used)`, 'info');

                log('zip-output', '=== ZIP SEARCH TEST COMPLETED SUCCESSFULLY ===', 'success');
                setStatus('zip', 'success');
                updateComparison();

            } catch (error) {
                log('zip-output', `ERROR: ${error.message}`, 'error');
                setStatus('zip', 'error');
            }
        }

        // Address Search Test
        async function testAddressSearch() {
            setStatus('address', 'testing');
            log('address-output', '=== STARTING ADDRESS SEARCH TEST ===', 'info');
            
            try {
                const address = document.getElementById('address-input').value.trim();
                
                log('address-output', `Input Address: "${address}"`, 'info');
                
                if (!address) {
                    log('address-output', 'VALIDATION: Address cannot be empty', 'error');
                    setStatus('address', 'error');
                    return;
                }

                // Simulate form field updates
                formFields.search_mode = 'address';
                formFields.keyword = ''; // Clear keyword for location search
                formFields.zipcode = ''; // Clear ZIP
                
                log('address-output', `Form fields updated:`, 'success');
                log('address-output', `  - search_mode: "${formFields.search_mode}"`, 'success');
                log('address-output', `  - keyword: "${formFields.keyword}" (cleared)`, 'success');
                log('address-output', `  - zipcode: "${formFields.zipcode}" (cleared)`, 'success');

                // Simulate geocoding
                log('address-output', 'Calling geocoding API...', 'info');
                const locationStatus = updateLocationStatus(`Finding businesses near ${address}...`, 'default');
                log('address-output', `Location status: ${locationStatus.message}`, 'info');

                // Mock geocoding response
                await new Promise(resolve => setTimeout(resolve, 1200));
                
                const mockCoords = { lat: 36.1627, lng: -86.7816 }; // Nashville
                formFields.latitude = mockCoords.lat;
                formFields.longitude = mockCoords.lng;
                
                log('address-output', `Geocoding successful:`, 'success');
                log('address-output', `  - latitude: ${formFields.latitude}`, 'success');
                log('address-output', `  - longitude: ${formFields.longitude}`, 'success');

                // Simulate search request and make ACTUAL API CALL
                const searchData = {
                    keyword: formFields.keyword,
                    location: '',
                    category: formFields.selected_category,
                    coords: { lat: formFields.latitude, lng: formFields.longitude },
                    page: 1,
                    open_only: false,
                    search_mode: 'address'
                };

                log('address-output', 'Search payload:', 'info');
                log('address-output', JSON.stringify(searchData, null, 2), 'info');

                // MAKE ACTUAL API CALL
                log('address-output', 'Making REAL API call to /ajax/locator...', 'warning');
                try {
                    const response = await fetch('/ajax/locator', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(searchData)
                    });

                    if (response.ok) {
                        const htmlResponse = await response.text();
                        const results = parseSearchResults(htmlResponse);
                        
                        actualResults.address = results;
                        
                        log('address-output', `API Response received: ${results.businesses.length} businesses found`, 'success');
                        log('address-output', `Sample businesses: ${results.businesses.slice(0, 3).map(b => b.name).join(', ')}`, 'info');
                        
                        // Log search result details
                        addToDebugLog('ADDRESS_API_RESULTS', `Total businesses returned: ${results.businesses.length}`, 'info');
                        addToDebugLog('ADDRESS_API_RESULTS', `Metadata: ${JSON.stringify(results.metadata)}`, 'info');
                        if (results.businesses.length > 0) {
                            addToDebugLog('ADDRESS_API_RESULTS', `First 5 business names: ${results.businesses.slice(0, 5).map(b => b.name).join(', ')}`, 'info');
                        }
                    } else {
                        log('address-output', `API call failed: ${response.status} ${response.statusText}`, 'error');
                        addToDebugLog('ADDRESS_API_RESULTS', `API call failed: ${response.status}`, 'error');
                    }
                } catch (apiError) {
                    log('address-output', `API Error: ${apiError.message}`, 'error');
                    addToDebugLog('ADDRESS_API_RESULTS', `API Error: ${apiError.message}`, 'error');
                }

                testResults.address = {
                    success: true,
                    formFields: { ...formFields },
                    searchData: { ...searchData },
                    timestamp: new Date()
                };

                // Add comprehensive summary to debug log
                addToDebugLog('ADDRESS_SUMMARY', '=== ADDRESS SEARCH COMPLETE ===', 'success');
                addToDebugLog('ADDRESS_SUMMARY', `Input: ${address}`, 'info');
                addToDebugLog('ADDRESS_SUMMARY', `Final form_keyword: "${formFields.keyword}"`, 'info');
                addToDebugLog('ADDRESS_SUMMARY', `Final coordinates: ${formFields.latitude}, ${formFields.longitude}`, 'info');
                addToDebugLog('ADDRESS_SUMMARY', `Search payload keyword: "${searchData.keyword}"`, 'info');
                addToDebugLog('ADDRESS_SUMMARY', `Search path: Location-based (coordinates used)`, 'info');

                log('address-output', '=== ADDRESS SEARCH TEST COMPLETED SUCCESSFULLY ===', 'success');
                setStatus('address', 'success');
                updateComparison();

            } catch (error) {
                log('address-output', `ERROR: ${error.message}`, 'error');
                setStatus('address', 'error');
            }
        }

        // Keyword Search Test
        async function testKeywordSearch() {
            setStatus('keyword', 'testing');
            log('keyword-output', '=== STARTING KEYWORD SEARCH TEST ===', 'info');
            
            try {
                const keyword = document.getElementById('keyword-input').value.trim();
                
                log('keyword-output', `Input Keyword: "${keyword}"`, 'info');
                
                if (!keyword) {
                    log('keyword-output', 'VALIDATION: Keyword cannot be empty', 'error');
                    setStatus('keyword', 'error');
                    return;
                }

                // Simulate form field updates - CRITICAL DIFFERENCE
                formFields.keyword = keyword; // SET the keyword
                formFields.search_mode = 'keyword';
                formFields.zipcode = ''; // Clear ZIP
                
                log('keyword-output', `Form fields updated:`, 'success');
                log('keyword-output', `  - keyword: "${formFields.keyword}" (SET TO KEYWORD!)`, 'success');
                log('keyword-output', `  - search_mode: "${formFields.search_mode}"`, 'success');
                log('keyword-output', `  - zipcode: "${formFields.zipcode}" (cleared)`, 'success');

                // Check if we have coordinates (optional for keyword search)
                const hasCoords = formFields.latitude && formFields.longitude;
                log('keyword-output', `Has coordinates: ${hasCoords}`, hasCoords ? 'success' : 'warning');
                
                if (hasCoords) {
                    log('keyword-output', `Using existing coordinates:`, 'info');
                    log('keyword-output', `  - latitude: ${formFields.latitude}`, 'info');
                    log('keyword-output', `  - longitude: ${formFields.longitude}`, 'info');
                } else {
                    log('keyword-output', `No coordinates - will search without location bias`, 'warning');
                }

                // Simulate keyword search request and make ACTUAL API CALL
                const searchData = {
                    keyword: formFields.keyword, // KEYWORD IS THE STAR HERE
                    location: '',
                    category: formFields.selected_category,
                    coords: hasCoords ? { lat: formFields.latitude, lng: formFields.longitude } : { lat: null, lng: null },
                    page: 1,
                    open_only: false,
                    search_mode: 'keyword'
                };

                log('keyword-output', 'Search payload:', 'info');
                log('keyword-output', JSON.stringify(searchData, null, 2), 'info');

                // Simulate location status update
                const locationStatus = updateLocationStatus(`Searching for "${keyword}"...`, 'default');
                log('keyword-output', `Location status: ${locationStatus.message}`, 'info');

                // MAKE ACTUAL API CALL
                log('keyword-output', 'Making REAL API call to /ajax/locator...', 'warning');
                try {
                    const response = await fetch('/ajax/locator', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify(searchData)
                    });

                    if (response.ok) {
                        const htmlResponse = await response.text();
                        const results = parseSearchResults(htmlResponse);
                        
                        actualResults.keyword = results;
                        
                        log('keyword-output', `API Response received: ${results.businesses.length} businesses found`, 'success');
                        log('keyword-output', `Sample businesses: ${results.businesses.slice(0, 3).map(b => b.name).join(', ')}`, 'info');
                        
                        // CRITICAL: Check relevance to keyword
                        const keywordRelevance = analyzeKeywordRelevance(keyword, results.businesses);
                        log('keyword-output', `RELEVANCE ANALYSIS: ${keywordRelevance.score}/10 relevance score`, keywordRelevance.score >= 7 ? 'success' : 'error');
                        log('keyword-output', `Relevant matches: ${keywordRelevance.relevantCount}/${results.businesses.length}`, 'info');
                        
                        // Log search result details
                        addToDebugLog('KEYWORD_API_RESULTS', `Total businesses returned: ${results.businesses.length}`, 'info');
                        addToDebugLog('KEYWORD_API_RESULTS', `Keyword relevance score: ${keywordRelevance.score}/10`, keywordRelevance.score >= 7 ? 'success' : 'error');
                        addToDebugLog('KEYWORD_API_RESULTS', `Relevant matches: ${keywordRelevance.relevantCount}/${results.businesses.length}`, 'info');
                        addToDebugLog('KEYWORD_API_RESULTS', `Metadata: ${JSON.stringify(results.metadata)}`, 'info');
                        if (results.businesses.length > 0) {
                            addToDebugLog('KEYWORD_API_RESULTS', `First 5 business names: ${results.businesses.slice(0, 5).map(b => b.name).join(', ')}`, 'info');
                            addToDebugLog('KEYWORD_API_RESULTS', `First 5 business categories: ${results.businesses.slice(0, 5).map(b => b.category || 'N/A').join(', ')}`, 'info');
                        }
                    } else {
                        log('keyword-output', `API call failed: ${response.status} ${response.statusText}`, 'error');
                        addToDebugLog('KEYWORD_API_RESULTS', `API call failed: ${response.status}`, 'error');
                    }
                } catch (apiError) {
                    log('keyword-output', `API Error: ${apiError.message}`, 'error');
                    addToDebugLog('KEYWORD_API_RESULTS', `API Error: ${apiError.message}`, 'error');
                }

                await new Promise(resolve => setTimeout(resolve, 500));

                testResults.keyword = {
                    success: true,
                    formFields: { ...formFields },
                    searchData: { ...searchData },
                    timestamp: new Date()
                };

                // Add comprehensive summary to debug log
                addToDebugLog('KEYWORD_SUMMARY', '=== KEYWORD SEARCH COMPLETE ===', 'success');
                addToDebugLog('KEYWORD_SUMMARY', `Input: ${keyword}`, 'info');
                addToDebugLog('KEYWORD_SUMMARY', `Final form_keyword: "${formFields.keyword}"`, 'info');
                addToDebugLog('KEYWORD_SUMMARY', `Has coordinates: ${hasCoords}`, 'info');
                if (hasCoords) {
                    addToDebugLog('KEYWORD_SUMMARY', `Coordinates: ${formFields.latitude}, ${formFields.longitude}`, 'info');
                }
                addToDebugLog('KEYWORD_SUMMARY', `Search payload keyword: "${searchData.keyword}"`, 'info');
                addToDebugLog('KEYWORD_SUMMARY', `Search path: ${hasCoords ? 'Keyword with location bias' : 'Pure keyword search'}`, 'info');

                log('keyword-output', '=== KEYWORD SEARCH TEST COMPLETED SUCCESSFULLY ===', 'success');
                setStatus('keyword', 'success');
                updateComparison();

            } catch (error) {
                log('keyword-output', `ERROR: ${error.message}`, 'error');
                setStatus('keyword', 'error');
            }
        }

        // Comparison Analysis
        function updateComparison() {
            const hasResults = testResults.zip || testResults.address || testResults.keyword;
            
            if (!hasResults) return;

            document.getElementById('comparison-grid').style.display = 'grid';

            // Update individual cards
            updateComparisonCard('zip', testResults.zip);
            updateComparisonCard('address', testResults.address);
            updateComparisonCard('keyword', testResults.keyword);
        }

        function updateComparisonCard(type, result) {
            const container = document.getElementById(`${type}-data`);
            
            if (!result) {
                container.innerHTML = '<div class="data-field">No test data yet</div>';
                return;
            }

            const fields = result.formFields;
            const search = result.searchData;

            let html = '<h4>Form Fields:</h4>';
            Object.keys(fields).forEach(key => {
                const value = fields[key] || '(empty)';
                const isDifferent = checkIfDifferent(key, value, type);
                html += `<div class="data-field ${isDifferent ? 'different' : ''}"><strong>${key}:</strong> ${value}</div>`;
            });

            html += '<h4>Search Payload:</h4>';
            html += `<div class="data-field"><strong>keyword:</strong> ${search.keyword || '(empty)'}</div>`;
            html += `<div class="data-field"><strong>coords:</strong> ${search.coords.lat ? `${search.coords.lat}, ${search.coords.lng}` : '(none)'}</div>`;
            html += `<div class="data-field"><strong>search_mode:</strong> ${search.search_mode}</div>`;

            container.innerHTML = html;
        }

        function checkIfDifferent(key, value, currentType) {
            // Check if this field value is different from other test types
            const otherTypes = ['zip', 'address', 'keyword'].filter(t => t !== currentType);
            
            for (let otherType of otherTypes) {
                if (testResults[otherType] && testResults[otherType].formFields[key] !== value) {
                    return true;
                }
            }
            return false;
        }

        // Parse HTML search results into structured data
        function parseSearchResults(htmlResponse) {
            const businesses = [];
            let metadata = {};
            
            // Parse metadata if present
            const metadataMatch = htmlResponse.match(/<!-- METADATA: (.+?) -->/);
            if (metadataMatch) {
                try {
                    metadata = JSON.parse(metadataMatch[1]);
                } catch (e) {
                    metadata = { parsing_error: e.message };
                }
            }
            
            // Create a temporary DOM to parse the HTML
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = htmlResponse;
            
            // Look for business cards (adjust selectors based on your actual HTML structure)
            const businessCards = tempDiv.querySelectorAll('.stores_list_modal, .appstore-card, .business-card');
            
            businessCards.forEach((card, index) => {
                const business = {
                    id: card.getAttribute('data-val') || card.dataset.val || `unknown-${index}`,
                    name: '',
                    category: '',
                    address: '',
                    phone: '',
                    description: '',
                    rawHtml: card.outerHTML.substring(0, 200) + '...' // First 200 chars for debugging
                };
                
                // Extract business name (try multiple selectors)
                const nameElement = card.querySelector('h3, h4, h5, .business-name, .store-name') || 
                                   card.querySelector('[class*="name"]') ||
                                   card.querySelector('[class*="title"]');
                if (nameElement) {
                    business.name = nameElement.textContent.trim();
                }
                
                // Extract category
                const categoryElement = card.querySelector('.category, .business-category, [class*="category"]');
                if (categoryElement) {
                    business.category = categoryElement.textContent.trim();
                }
                
                // Extract address
                const addressElement = card.querySelector('.address, .business-address, [class*="address"]');
                if (addressElement) {
                    business.address = addressElement.textContent.trim();
                }
                
                // Extract phone
                const phoneElement = card.querySelector('.phone, .business-phone, [class*="phone"]');
                if (phoneElement) {
                    business.phone = phoneElement.textContent.trim();
                }
                
                // Extract description
                const descElement = card.querySelector('.description, .business-description, p');
                if (descElement) {
                    business.description = descElement.textContent.trim().substring(0, 100);
                }
                
                businesses.push(business);
            });
            
            return { businesses, metadata, totalResults: businesses.length };
        }

        // Analyze keyword relevance
        function analyzeKeywordRelevance(keyword, businesses) {
            if (!businesses.length) return { score: 0, relevantCount: 0, details: 'No businesses to analyze' };
            
            const keywordLower = keyword.toLowerCase();
            let relevantCount = 0;
            const analysisDetails = [];
            
            businesses.forEach((business, index) => {
                const searchText = `${business.name} ${business.category} ${business.description}`.toLowerCase();
                const isRelevant = searchText.includes(keywordLower) || 
                                 business.name.toLowerCase().includes(keywordLower) ||
                                 business.category.toLowerCase().includes(keywordLower);
                
                if (isRelevant) {
                    relevantCount++;
                    analysisDetails.push(`${business.name} (${business.category})`);
                }
            });
            
            const score = Math.round((relevantCount / businesses.length) * 10);
            
            return {
                score,
                relevantCount,
                totalCount: businesses.length,
                details: analysisDetails.slice(0, 5).join(', ') + (analysisDetails.length > 5 ? '...' : '')
            };
        }

        // Export Debug Log
        function exportDebugLog() {
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
            const filename = `search-debug-report-${timestamp}.txt`;
            
            let report = '='.repeat(80) + '\n';
            report += 'BESTCALL THREE-TAB SEARCH DEBUG REPORT\n';
            report += '='.repeat(80) + '\n';
            report += `Generated: ${new Date().toString()}\n`;
            report += `Browser: ${navigator.userAgent}\n`;
            report += `URL: ${window.location.href}\n\n`;

            // Executive Summary
            report += 'EXECUTIVE SUMMARY\n';
            report += '-'.repeat(40) + '\n';
            const zipStatus = testResults.zip ? 'COMPLETED' : 'NOT RUN';
            const addressStatus = testResults.address ? 'COMPLETED' : 'NOT RUN';
            const keywordStatus = testResults.keyword ? 'COMPLETED' : 'NOT RUN';
            
            report += `ZIP Search Test: ${zipStatus}\n`;
            report += `Address Search Test: ${addressStatus}\n`;
            report += `Keyword Search Test: ${keywordStatus}\n\n`;

            // Detailed Test Results
            if (testResults.zip || testResults.address || testResults.keyword) {
                report += 'DETAILED COMPARISON\n';
                report += '-'.repeat(40) + '\n';
                
                ['zip', 'address', 'keyword'].forEach(type => {
                    if (testResults[type]) {
                        report += `\n${type.toUpperCase()} SEARCH RESULTS:\n`;
                        report += `  Test Completed: ${testResults[type].timestamp}\n`;
                        report += `  Form Fields:\n`;
                        Object.entries(testResults[type].formFields).forEach(([key, value]) => {
                            report += `    ${key}: "${value}"\n`;
                        });
                        report += `  Search Payload:\n`;
                        Object.entries(testResults[type].searchData).forEach(([key, value]) => {
                            if (typeof value === 'object') {
                                report += `    ${key}: ${JSON.stringify(value)}\n`;
                            } else {
                                report += `    ${key}: "${value}"\n`;
                            }
                        });
                    }
                });

                // ACTUAL SEARCH RESULTS ANALYSIS
                report += '\nACTUAL SEARCH RESULTS ANALYSIS\n';
                report += '-'.repeat(40) + '\n';
                
                ['zip', 'address', 'keyword'].forEach(type => {
                    if (actualResults[type]) {
                        const results = actualResults[type];
                        report += `\n${type.toUpperCase()} SEARCH API RESULTS:\n`;
                        report += `  Total Businesses Found: ${results.totalResults}\n`;
                        report += `  Metadata: ${JSON.stringify(results.metadata)}\n`;
                        
                        if (results.businesses.length > 0) {
                            report += `  Sample Business Names:\n`;
                            results.businesses.slice(0, 10).forEach((business, index) => {
                                report += `    ${index + 1}. ${business.name} (${business.category || 'No category'})\n`;
                                if (business.address) report += `       Address: ${business.address}\n`;
                                if (business.description) report += `       Desc: ${business.description}\n`;
                            });
                            
                            if (results.businesses.length > 10) {
                                report += `    ... and ${results.businesses.length - 10} more businesses\n`;
                            }
                        } else {
                            report += `  NO BUSINESSES RETURNED\n`;
                        }
                        
                        // Keyword relevance analysis
                        if (type === 'keyword' && results.businesses.length > 0) {
                            const keyword = testResults.keyword.formFields.keyword;
                            const relevance = analyzeKeywordRelevance(keyword, results.businesses);
                            report += `  KEYWORD RELEVANCE ANALYSIS:\n`;
                            report += `    Search Term: "${keyword}"\n`;
                            report += `    Relevance Score: ${relevance.score}/10\n`;
                            report += `    Relevant Matches: ${relevance.relevantCount}/${relevance.totalCount}\n`;
                            report += `    Sample Relevant: ${relevance.details}\n`;
                            
                            if (relevance.score < 5) {
                                report += `    ⚠️  LOW RELEVANCE WARNING: Results may not match search intent\n`;
                            }
                        }
                    } else {
                        report += `\n${type.toUpperCase()} SEARCH: No API results (test may have failed)\n`;
                    }
                });

                // Key Differences Analysis
                report += '\nKEY DIFFERENCES ANALYSIS\n';
                report += '-'.repeat(40) + '\n';
                
                if (testResults.zip && testResults.keyword) {
                    report += 'ZIP vs KEYWORD:\n';
                    report += `  ZIP keyword field: "${testResults.zip.formFields.keyword}"\n`;
                    report += `  KEYWORD keyword field: "${testResults.keyword.formFields.keyword}"\n`;
                    report += `  ZIP coordinates: ${testResults.zip.formFields.latitude ? 'SET' : 'EMPTY'}\n`;
                    report += `  KEYWORD coordinates: ${testResults.keyword.formFields.latitude ? 'SET' : 'EMPTY'}\n`;
                    
                    if (actualResults.zip && actualResults.keyword) {
                        report += `  ZIP results count: ${actualResults.zip.totalResults}\n`;
                        report += `  KEYWORD results count: ${actualResults.keyword.totalResults}\n`;
                        
                        // Compare business overlap
                        const zipNames = new Set(actualResults.zip.businesses.map(b => b.name.toLowerCase()));
                        const keywordNames = new Set(actualResults.keyword.businesses.map(b => b.name.toLowerCase()));
                        const overlap = [...zipNames].filter(name => keywordNames.has(name));
                        report += `  Business overlap: ${overlap.length} businesses appear in both searches\n`;
                        
                        if (overlap.length > 0) {
                            report += `  Common businesses: ${overlap.slice(0, 5).join(', ')}\n`;
                        }
                    }
                    report += '\n';
                }
                
                if (testResults.address && testResults.keyword) {
                    report += 'ADDRESS vs KEYWORD:\n';
                    report += `  ADDRESS keyword field: "${testResults.address.formFields.keyword}"\n`;
                    report += `  KEYWORD keyword field: "${testResults.keyword.formFields.keyword}"\n`;
                    report += `  ADDRESS coordinates: ${testResults.address.formFields.latitude ? 'SET' : 'EMPTY'}\n`;
                    report += `  KEYWORD coordinates: ${testResults.keyword.formFields.latitude ? 'SET' : 'EMPTY'}\n`;
                    
                    if (actualResults.address && actualResults.keyword) {
                        report += `  ADDRESS results count: ${actualResults.address.totalResults}\n`;
                        report += `  KEYWORD results count: ${actualResults.keyword.totalResults}\n`;
                    }
                    report += '\n';
                }
            }

            // Full Debug Log
            report += '\nFULL DEBUG LOG\n';
            report += '-'.repeat(40) + '\n';
            
            debugLog.forEach(entry => {
                const timestamp = new Date(entry.timestamp).toLocaleTimeString();
                report += `[${timestamp}] ${entry.section} (${entry.type.toUpperCase()}): ${entry.message}\n`;
            });

            // Conclusions and Recommendations
            report += '\n' + '='.repeat(80) + '\n';
            report += 'CONCLUSIONS AND RECOMMENDATIONS\n';
            report += '='.repeat(80) + '\n';
            
            if (testResults.keyword && (testResults.zip || testResults.address)) {
                report += '\nCRITICAL FINDINGS:\n';
                
                const keywordHasKeyword = testResults.keyword.formFields.keyword;
                const zipHasKeyword = testResults.zip ? testResults.zip.formFields.keyword : null;
                const addressHasKeyword = testResults.address ? testResults.address.formFields.keyword : null;
                
                if (keywordHasKeyword && (!zipHasKeyword && !addressHasKeyword)) {
                    report += '✓ KEYWORD search properly sets form_keyword field\n';
                    report += '✓ ZIP/ADDRESS searches properly clear form_keyword field\n';
                    report += '→ This suggests input collection is working correctly\n\n';
                } else {
                    report += '✗ INCONSISTENT keyword field handling detected\n';
                    report += '→ This may indicate form field management issues\n\n';
                }
                
                // SEARCH RESULT QUALITY ANALYSIS
                if (actualResults.keyword) {
                    const keywordResults = actualResults.keyword;
                    report += 'SEARCH RESULT QUALITY ANALYSIS:\n';
                    
                    if (keywordResults.totalResults === 0) {
                        report += '✗ CRITICAL: Keyword search returned ZERO results\n';
                        report += '→ This indicates the search mechanism is broken\n';
                    } else {
                        const keyword = testResults.keyword.formFields.keyword;
                        const relevance = analyzeKeywordRelevance(keyword, keywordResults.businesses);
                        
                        if (relevance.score >= 7) {
                            report += `✓ GOOD: Keyword search returned ${keywordResults.totalResults} relevant results\n`;
                            report += `✓ RELEVANCE: ${relevance.score}/10 relevance score is acceptable\n`;
                        } else if (relevance.score >= 3) {
                            report += `⚠️  MIXED: Keyword search returned results but relevance is questionable\n`;
                            report += `⚠️  RELEVANCE: ${relevance.score}/10 relevance score suggests poor matching\n`;
                        } else {
                            report += `✗ POOR: Keyword search returned irrelevant results\n`;
                            report += `✗ RELEVANCE: ${relevance.score}/10 relevance score indicates broken keyword matching\n`;
                        }
                        
                        report += `→ Search term: "${keyword}"\n`;
                        report += `→ Total results: ${keywordResults.totalResults}\n`;
                        report += `→ Relevant matches: ${relevance.relevantCount}/${relevance.totalCount}\n`;
                        
                        // Compare with location-based searches
                        if (actualResults.zip) {
                            const zipCount = actualResults.zip.totalResults;
                            if (keywordResults.totalResults === zipCount) {
                                report += `⚠️  WARNING: Keyword and ZIP searches returned identical counts (${zipCount})\n`;
                                report += `→ This suggests keyword search may be falling back to location search\n`;
                            }
                        }
                        
                        if (actualResults.address) {
                            const addressCount = actualResults.address.totalResults;
                            if (keywordResults.totalResults === addressCount) {
                                report += `⚠️  WARNING: Keyword and Address searches returned identical counts (${addressCount})\n`;
                                report += `→ This suggests keyword search may be falling back to location search\n`;
                            }
                        }
                    }
                    report += '\n';
                }
                
                report += 'NEXT DEBUGGING STEPS:\n';
                
                if (actualResults.keyword && actualResults.keyword.totalResults === 0) {
                    report += '1. URGENT: Check why keyword search returns zero results\n';
                    report += '2. Verify backend is receiving the keyword parameter correctly\n';
                    report += '3. Check database query construction for keyword searches\n';
                } else if (actualResults.keyword) {
                    const relevance = analyzeKeywordRelevance(testResults.keyword.formFields.keyword, actualResults.keyword.businesses);
                    if (relevance.score < 5) {
                        report += '1. PRIORITY: Investigate why keyword search returns irrelevant results\n';
                        report += '2. Check if backend is using keyword for actual text matching\n';
                        report += '3. Verify keyword search is not falling back to location-only sorting\n';
                    } else {
                        report += '1. Keyword search appears to be working correctly\n';
                        report += '2. Focus on integrating this working logic into hero section\n';
                    }
                }
                
                report += '4. Test with actual CodeIgniter backend error logging\n';
                report += '5. Compare sticky header success with hero tab attempts\n';
                report += '6. Check browser network tab for actual payload differences\n';
                report += '7. Verify hero form submission path vs sticky header path\n\n';
            }
            
            report += 'END OF REPORT\n';
            report += '='.repeat(80);

            // Download the file
            const blob = new Blob([report], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);

            // Also log the export
            addToDebugLog('EXPORT', `Debug report exported as ${filename}`, 'success');
            console.log('Debug report exported:', filename);
        }

        // Run All Tests
        async function runAllTests() {
            addToDebugLog('AUTO_TEST', 'Starting automated test sequence', 'info');
            
            try {
                await testZipSearch();
                await new Promise(resolve => setTimeout(resolve, 1000));
                
                await testAddressSearch();
                await new Promise(resolve => setTimeout(resolve, 1000));
                
                await testKeywordSearch();
                
                addToDebugLog('AUTO_TEST', 'All automated tests completed successfully', 'success');
                
                // Auto-export after all tests
                setTimeout(() => {
                    exportDebugLog();
                }, 1000);
                
            } catch (error) {
                addToDebugLog('AUTO_TEST', `Automated test sequence failed: ${error.message}`, 'error');
            }
        }

        // Clear All Logs
        function clearAllLogs() {
            debugLog = [];
            testResults = { zip: null, address: null, keyword: null };
            actualResults = { zip: null, address: null, keyword: null };
            
            ['zip-output', 'address-output', 'keyword-output'].forEach(id => {
                document.getElementById(id).innerHTML = 'Cleared...\n';
            });
            
            ['zip', 'address', 'keyword'].forEach(type => {
                setStatus(type, 'idle');
            });
            
            document.getElementById('comparison-grid').style.display = 'none';
            
            // Reset form fields
            formFields = {
                latitude: '',
                longitude: '',
                keyword: '',
                zipcode: '',
                selected_category: '',
                search_mode: ''
            };
        }

        // ZIP digit handling
        document.addEventListener('DOMContentLoaded', function() {
            const zipDigits = document.querySelectorAll('.zip-digit');
            
            zipDigits.forEach((digit, index) => {
                digit.addEventListener('input', function(e) {
                    const value = e.target.value;
                    
                    if (!/^\d$/.test(value) && value !== '') {
                        e.target.value = '';
                        return;
                    }
                    
                    if (value && index < zipDigits.length - 1) {
                        zipDigits[index + 1].focus();
                    }
                });
                
                digit.addEventListener('keydown', function(e) {
                    if (e.key === 'Backspace' && !e.target.value && index > 0) {
                        zipDigits[index - 1].focus();
                    }
                });
            });

            // Auto-fill some test data
            setTimeout(() => {
                document.getElementById('zip-1').value = '3';
                document.getElementById('zip-2').value = '7';
                document.getElementById('zip-3').value = '2';
                document.getElementById('zip-4').value = '0';
                document.getElementById('zip-5').value = '1';
            }, 500);
        });
    </script>
</body>
</html>