<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $title ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }

        .admin-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
            padding: 1rem 0;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.1);
        }

        .nav-container {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 2rem;
        }

        .nav-brand {
            font-size: 1.5rem;
            font-weight: 700;
            color: #4a5568;
            text-decoration: none;
        }

        .nav-links {
            display: flex;
            gap: 0;
        }

        .nav-links a {
            color: #4a5568;
            text-decoration: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            transition: all 0.3s ease;
            font-weight: 500;
            position: relative;
        }

        .nav-links a:hover {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
        }

        .nav-links a.active {
            background: #667eea;
            color: white;
            box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
        }

        .user-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-info {
            font-size: 0.9rem;
            color: #666;
        }

        .logout-btn {
            background: #e53e3e;
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            text-decoration: none;
            transition: all 0.3s ease;
        }

        .logout-btn:hover {
            background: #c53030;
            transform: translateY(-1px);
        }

        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .page-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 0.5rem;
        }

        .page-subtitle {
            color: #718096;
            font-size: 1.1rem;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-left: 4px solid;
        }

        .stat-card.primary { border-left-color: #667eea; }
        .stat-card.success { border-left-color: #38a169; }
        .stat-card.warning { border-left-color: #ed8936; }
        .stat-card.info { border-left-color: #4299e1; }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }

        .stat-label {
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: #718096;
        }

        .stat-icon {
            font-size: 2rem;
            opacity: 0.3;
        }

        .stat-value {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2d3748;
        }

        .search-panel {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .search-grid {
            display: grid;
            grid-template-columns: 2fr 1fr auto auto;
            gap: 1.5rem;
            align-items: end;
        }

        .form-group {
            position: relative;
        }

        .form-label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #4a5568;
            font-size: 0.9rem;
        }

        .form-input, .form-select {
            width: 100%;
            padding: 0.875rem 1rem;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .form-input:focus, .form-select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .search-input {
            padding-left: 3rem;
        }

        .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #a0aec0;
            font-size: 1.1rem;
        }

        .btn {
            padding: 0.875rem 1.5rem;
            border: none;
            border-radius: 10px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
            font-size: 0.95rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: #f7fafc;
            color: #4a5568;
            border: 2px solid #e2e8f0;
        }

        .btn-secondary:hover {
            background: #edf2f7;
            border-color: #cbd5e0;
        }

        .businesses-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .businesses-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }

        .businesses-table th {
            background: #f8fafc;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: #4a5568;
            border-bottom: 2px solid #e2e8f0;
            font-size: 0.9rem;
        }

        .businesses-table td {
            padding: 1rem;
            border-bottom: 1px solid #e2e8f0;
            vertical-align: top;
        }

        .businesses-table tr:hover {
            background: rgba(102, 126, 234, 0.02);
        }

        .business-name {
            font-weight: 600;
            color: #2d3748;
            font-size: 1.1rem;
            margin-bottom: 0.25rem;
        }

        .business-location {
            color: #718096;
            font-size: 0.9rem;
        }

        .status-badge {
            padding: 0.375rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-block;
        }

        .status-open {
            background: #c6f6d5;
            color: #22543d;
        }

        .status-closed {
            background: #fed7d7;
            color: #742a2a;
        }

        .status-closing-soon {
            background: #fef5e7;
            color: #975a16;
        }

        .status-no-hours {
            background: #e2e8f0;
            color: #4a5568;
        }

        .hour-types-badge {
            background: #bee3f8;
            color: #2b6cb0;
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .action-buttons {
            display: flex;
            gap: 0.5rem;
        }

        .btn-sm {
            padding: 0.5rem 0.75rem;
            font-size: 0.8rem;
            border-radius: 6px;
        }

        .btn-edit {
            background: #667eea;
            color: white;
        }

        .btn-view {
            background: #4299e1;
            color: white;
        }

        .btn-sm:hover {
            transform: translateY(-1px);
            opacity: 0.9;
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 10px;
            margin-bottom: 1.5rem;
            border-left: 4px solid;
            font-weight: 500;
        }

        .alert-success {
            background: #f0fff4;
            border-color: #38a169;
            color: #22543d;
        }

        .alert-error {
            background: #fffaf0;
            border-color: #e53e3e;
            color: #742a2a;
        }

        .bulk-controls {
            background: #f7fafc;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            padding: 1rem;
            margin-bottom: 1.5rem;
            display: none;
        }

        .bulk-controls.show {
            display: block;
        }

        @media (max-width: 1200px) {
            .search-grid {
                grid-template-columns: 1fr 1fr;
                gap: 1rem;
            }
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 1rem;
            }
            
            .search-grid {
                grid-template-columns: 1fr;
            }
            
            .businesses-table {
                font-size: 0.9rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <header class="admin-header">
        <div class="nav-container">
            <a href="/admin/dashboard" class="nav-brand">
                <i class="fas fa-store"></i> Business Directory Admin
            </a>
            
            <nav class="nav-links">
                <a href="/admin/dashboard">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="/admin/stores">
                    <i class="fas fa-building"></i> Stores
                </a>
                <a href="/admin/hours" class="active">
                    <i class="fas fa-clock"></i> Hours
                </a>
                <a href="/admin/settings">
                    <i class="fas fa-cog"></i> Settings
                </a>
            </nav>
            
            <div class="user-menu">
                <div class="user-info">
                    <i class="fas fa-user-circle"></i>
                    <?= session()->get('admin_name') ?>
                </div>
                <a href="/admin/logout" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </header>

    <main class="main-container">
        <div class="page-header">
            <h1 class="page-title">
                <i class="fas fa-clock"></i> Business Hours Management
            </h1>
            <p class="page-subtitle">
                Manage operating hours for all businesses in your directory
            </p>
        </div>

        <!-- Stats Cards -->
        <div class="stats-grid">
            <div class="stat-card primary">
                <div class="stat-header">
                    <div class="stat-label">Total Businesses</div>
                    <i class="fas fa-building stat-icon"></i>
                </div>
                <div class="stat-value"><?= number_format($total_stores) ?></div>
            </div>

            <div class="stat-card success">
                <div class="stat-header">
                    <div class="stat-label">With Hours Set</div>
                    <i class="fas fa-clock stat-icon"></i>
                </div>
                <div class="stat-value"><?= number_format($stores_with_hours) ?></div>
            </div>

            <div class="stat-card warning">
                <div class="stat-header">
                    <div class="stat-label">Need Hours Setup</div>
                    <i class="fas fa-exclamation-triangle stat-icon"></i>
                </div>
                <div class="stat-value"><?= number_format($total_stores - $stores_with_hours) ?></div>
            </div>

            <div class="stat-card info">
                <div class="stat-header">
                    <div class="stat-label">Currently Open</div>
                    <i class="fas fa-door-open stat-icon"></i>
                </div>
                <div class="stat-value" id="currently-open-count">
                    <i class="fas fa-spinner fa-spin"></i>
                </div>
            </div>
        </div>

        <!-- Flash Messages -->
        <?php if (session()->getFlashdata('success')): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?= session()->getFlashdata('success') ?>
            </div>
        <?php endif; ?>

        <?php if (session()->getFlashdata('error')): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i> <?= session()->getFlashdata('error') ?>
            </div>
        <?php endif; ?>

        <!-- Search and Filter -->
        <div class="search-panel">
            <div class="search-grid">
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-search"></i> Search Businesses
                    </label>
                    <div style="position: relative;">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" id="searchInput" class="form-input search-input"
                               placeholder="Business name, location, hour types...">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">
                        <i class="fas fa-filter"></i> Status Filter
                    </label>
                    <select id="statusFilter" class="form-select">
                        <option value="">All Statuses</option>
                        <option value="open">Currently Open</option>
                        <option value="closed">Currently Closed</option>
                        <option value="no-hours">No Hours Set</option>
                    </select>
                </div>
                
                <div>
                    <button class="btn btn-primary" onclick="filterBusinesses()">
                        <i class="fas fa-search"></i> Filter
                    </button>
                </div>
                
                <div>
                    <button class="btn btn-secondary" onclick="clearFilters()">
                        <i class="fas fa-times"></i> Clear
                    </button>
                </div>
            </div>
        </div>

        <!-- Bulk Selection Controls -->
        <div class="bulk-controls" id="bulk-controls">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <strong><span id="selected-count">0</span></strong> businesses selected
                </div>
                <div style="display: flex; gap: 1rem;">
                    <button class="btn btn-primary btn-sm" onclick="addDefaultHours()">
                        <i class="fas fa-clock"></i> Add Default Hours
                    </button>
                    <button class="btn btn-secondary btn-sm" onclick="clearSelection()">
                        <i class="fas fa-times"></i> Clear Selection
                    </button>
                </div>
            </div>
        </div>

        <div class="businesses-container">
            <?php if (!empty($stores)): ?>
                <table class="businesses-table">
                    <thead>
                        <tr>
                            <th width="40">
                                <input type="checkbox" id="select-all" onchange="toggleSelectAll()">
                            </th>
                            <th>Business Name</th>
                            <th>Location</th>
                            <th>Hour Types</th>
                            <th>Current Status</th>
                            <th>Last Updated</th>
                            <th width="120">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stores as $store): ?>
                        <tr data-store-id="<?= $store['store_id'] ?>" class="business-row">
                            <td>
                                <input type="checkbox" class="store-checkbox" value="<?= $store['store_id'] ?>" onchange="updateBulkControls()">
                            </td>
                            <td>
                                <div class="business-name"><?= esc($store['name']) ?></div>
                                <small style="color: #666;">ID: <?= esc($store['store_id']) ?></small>
                            </td>
                            <td>
                                <div class="business-location">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <?= esc($store['city']) ?>, <?= esc($store['state']) ?>
                                </div>
                            </td>
                            <td>
                                <?php if ($store['hour_types_count'] > 0): ?>
                                    <div class="hour-types-badge"><?= $store['hour_types_count'] ?> types</div>
                                    <br><small style="color: #666; font-size: 0.8rem;"><?= esc($store['hour_types']) ?></small>
                                <?php else: ?>
                                    <span class="status-badge status-no-hours">No hours set</span>
                                <?php endif; ?>
                            </td>
                            <td class="status-cell" data-store-id="<?= $store['store_id'] ?>">
                                <div class="status-loading">
                                    <i class="fas fa-spinner fa-spin" style="color: #666;"></i> Checking...
                                </div>
                            </td>
                            <td>
                                <?php if ($store['last_updated']): ?>
                                    <small style="color: #666;">
                                        <?= date('M j, Y g:i A', strtotime($store['last_updated'])) ?>
                                    </small>
                                <?php else: ?>
                                    <small style="color: #999;">Never</small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <a href="/admin/hours/store/<?= $store['store_id'] ?>" 
                                       class="btn btn-edit btn-sm" 
                                       title="Edit Hours">
                                        <i class="fas fa-clock"></i>
                                    </a>
                                    <a href="/admin/stores/view/<?= $store['id'] ?>" 
                                       class="btn btn-view btn-sm" 
                                       title="View Store">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <div style="text-align: center; padding: 4rem; color: #718096;">
                    <i class="fas fa-clock" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.5;"></i>
                    <h3 style="margin-bottom: 0.5rem;">No businesses found</h3>
                    <p>Add some businesses to get started with hours management.</p>
                    <a href="/admin/stores" class="btn btn-primary" style="margin-top: 1rem;">
                        <i class="fas fa-plus"></i> Manage Stores
                    </a>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <script>
    let currentlyOpenCount = 0;
    
    // Load real-time status for all businesses on page load
    document.addEventListener('DOMContentLoaded', function() {
        loadAllStatuses();
        
        // Auto-refresh every 60 seconds
        setInterval(loadAllStatuses, 60000);
    });
    
    function loadAllStatuses() {
        const businessRows = document.querySelectorAll('.business-row');
        currentlyOpenCount = 0;
        
        businessRows.forEach(row => {
            const storeId = row.dataset.storeId;
            const statusCell = row.querySelector('.status-cell');
            
            fetch(`/admin/hours/api-status/${storeId}`)
                .then(response => response.json())
                .then(data => {
                    statusCell.innerHTML = getStatusBadge(data);
                    if (data.status === 'open') {
                        currentlyOpenCount++;
                    }
                })
                .catch(error => {
                    statusCell.innerHTML = '<span class="status-badge status-no-hours">Unknown</span>';
                })
                .finally(() => {
                    document.getElementById('currently-open-count').textContent = currentlyOpenCount;
                });
        });
    }
    
    function getStatusBadge(statusData) {
        const status = statusData.status;
        const services = statusData.services || [];
        
        switch (status) {
            case 'open':
                let badge = '<span class="status-badge status-open"><i class="fas fa-check-circle"></i> Open</span>';
                if (services.length > 0) {
                    badge += '<br><small style="color: #666;">(' + services.join(', ') + ')</small>';
                }
                return badge;
                
            case 'closing_soon':
                let closingBadge = '<span class="status-badge status-closing-soon"><i class="fas fa-clock"></i> Closing Soon</span>';
                if (services.length > 0) {
                    closingBadge += '<br><small style="color: #666;">(' + services.join(', ') + ')</small>';
                }
                return closingBadge;
                
            case 'closed':
            default:
                const message = statusData.message || 'Closed';
                return '<span class="status-badge status-closed"><i class="fas fa-times-circle"></i> Closed</span><br><small style="color: #666;">(' + message + ')</small>';
        }
    }
    
    function filterBusinesses() {
        const searchTerm = document.getElementById('searchInput').value.toLowerCase();
        const statusFilterValue = document.getElementById('statusFilter').value;
        const businessRows = document.querySelectorAll('.business-row');
        
        businessRows.forEach(row => {
            const businessName = row.querySelector('td:nth-child(2)').textContent.toLowerCase();
            const location = row.querySelector('td:nth-child(3)').textContent.toLowerCase();
            const hoursInfo = row.querySelector('td:nth-child(4)').textContent.toLowerCase();
            
            const matchesSearch = businessName.includes(searchTerm) || 
                                location.includes(searchTerm) || 
                                hoursInfo.includes(searchTerm);
            
            let matchesStatus = true;
            if (statusFilterValue) {
                const statusCell = row.querySelector('.status-cell');
                const statusText = statusCell.textContent.toLowerCase();
                
                switch (statusFilterValue) {
                    case 'open':
                        matchesStatus = statusText.includes('open') && !statusText.includes('closed');
                        break;
                    case 'closed':
                        matchesStatus = statusText.includes('closed');
                        break;
                    case 'no-hours':
                        matchesStatus = hoursInfo.includes('no hours set');
                        break;
                }
            }
            
            row.style.display = (matchesSearch && matchesStatus) ? '' : 'none';
        });
    }
    
    function clearFilters() {
        document.getElementById('searchInput').value = '';
        document.getElementById('statusFilter').value = '';
        filterBusinesses();
    }
    
    function toggleSelectAll() {
        const selectAllCheckbox = document.getElementById('select-all');
        const storeCheckboxes = document.querySelectorAll('.store-checkbox');
        
        storeCheckboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
        });
        
        updateBulkControls();
    }
    
    function updateBulkControls() {
        const selectedCheckboxes = document.querySelectorAll('.store-checkbox:checked');
        const count = selectedCheckboxes.length;
        const bulkControls = document.getElementById('bulk-controls');
        
        document.getElementById('selected-count').textContent = count;
        
        if (count > 0) {
            bulkControls.classList.add('show');
        } else {
            bulkControls.classList.remove('show');
        }
    }
    
    function clearSelection() {
        document.getElementById('select-all').checked = false;
        document.querySelectorAll('.store-checkbox').forEach(checkbox => {
            checkbox.checked = false;
        });
        updateBulkControls();
    }
    
    function addDefaultHours() {
        const selectedIds = Array.from(document.querySelectorAll('.store-checkbox:checked')).map(cb => cb.value);
        
        if (selectedIds.length === 0) {
            alert('Please select businesses first.');
            return;
        }
        
        if (confirm(`Add default hours (9 AM - 5 PM, Mon-Fri) to ${selectedIds.length} selected businesses?`)) {
            // Create form and submit
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/hours/bulk-update';
            
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'bulk_action';
            actionInput.value = 'add_regular_hours';
            
            const idsInput = document.createElement('input');
            idsInput.type = 'hidden';
            idsInput.name = 'store_ids';
            idsInput.value = JSON.stringify(selectedIds);
            
            form.appendChild(actionInput);
            form.appendChild(idsInput);
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Auto-filter as user types
    document.getElementById('searchInput').addEventListener('keyup', filterBusinesses);
    document.getElementById('statusFilter').addEventListener('change', filterBusinesses);
    </script>
</body>
</html>