<?php

namespace App\Models;

use CodeIgniter\Model;

class CategoryModel extends Model
{
    protected $table = 'category';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    protected $allowedFields = [
        'category_id', 'title', 'icon', 'image', 'type', 
        'top_category', 'status', 'create_date', 'update_date'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'create_date';
    protected $updatedField = 'update_date';

    // Validation
    protected $validationRules = [
        'title' => 'required|min_length[2]|max_length[255]',
        'icon' => 'permit_empty|max_length[255]',
        'status' => 'required|in_list[0,1]'
    ];

    protected $validationMessages = [
        'title' => [
            'required' => 'Category title is required',
            'min_length' => 'Category title must be at least 2 characters',
            'max_length' => 'Category title cannot exceed 255 characters'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    /**
     * Get all active categories
     */
    public function getActiveCategories($orderBy = 'title')
    {
        return $this->where('status', 1)
                    ->orderBy($orderBy, 'ASC')
                    ->findAll();
    }

    /**
     * Get top categories (featured categories)
     */
    public function getTopCategories()
    {
        return $this->where(['status' => 1, 'top_category' => 1])
                    ->orderBy('title', 'ASC')
                    ->findAll();
    }

    /**
     * Get category with store count
     */
    public function getCategoriesWithStoreCounts()
    {
        $db = \Config\Database::connect();
        
        return $db->query("
            SELECT 
                c.*,
                COUNT(sc.store_id) as store_count
            FROM category c
            LEFT JOIN store_category sc ON c.category_id = sc.category_id
            WHERE c.status = 1
            GROUP BY c.id, c.category_id, c.title, c.icon, c.image, c.type, c.top_category, c.status
            ORDER BY c.title ASC
        ")->getResultArray();
    }

    /**
     * Get stores by category
     */
    public function getStoresByCategory($categoryId)
    {
        $db = \Config\Database::connect();
        
        return $db->query("
            SELECT 
                s.*,
                c.title as category_title
            FROM store s
            INNER JOIN store_category sc ON s.store_id = sc.store_id
            INNER JOIN category c ON sc.category_id = c.category_id
            WHERE c.category_id = ? AND c.status = 1 AND s.status = 1
            ORDER BY s.name ASC
        ", [$categoryId])->getResultArray();
    }

    /**
     * Get categories for a specific store
     */
    public function getCategoriesForStore($storeId)
    {
        $db = \Config\Database::connect();
        
        return $db->query("
            SELECT 
                c.*
            FROM category c
            INNER JOIN store_category sc ON c.category_id = sc.category_id
            WHERE sc.store_id = ? AND c.status = 1
            ORDER BY c.title ASC
        ", [$storeId])->getResultArray();
    }

    /**
     * Assign category to store
     */
    public function assignCategoryToStore($storeId, $categoryId)
    {
        $db = \Config\Database::connect();
        
        // Check if assignment already exists
        $existing = $db->query("
            SELECT id FROM store_category 
            WHERE store_id = ? AND category_id = ?
        ", [$storeId, $categoryId])->getRowArray();
        
        if (!$existing) {
            return $db->query("
                INSERT INTO store_category (store_id, category_id) 
                VALUES (?, ?)
            ", [$storeId, $categoryId]);
        }
        
        return true; // Already exists
    }

    /**
     * Remove category from store
     */
    public function removeCategoryFromStore($storeId, $categoryId)
    {
        $db = \Config\Database::connect();
        
        return $db->query("
            DELETE FROM store_category 
            WHERE store_id = ? AND category_id = ?
        ", [$storeId, $categoryId]);
    }

    /**
     * Update store categories (replaces all categories for a store)
     */
    public function updateStoreCategories($storeId, $categoryIds = [])
    {
        $db = \Config\Database::connect();
        
        // Start transaction
        $db->transStart();
        
        try {
            // Remove all existing categories for this store
            $db->query("DELETE FROM store_category WHERE store_id = ?", [$storeId]);
            
            // Add new categories
            if (!empty($categoryIds)) {
                foreach ($categoryIds as $categoryId) {
                    $db->query("
                        INSERT INTO store_category (store_id, category_id) 
                        VALUES (?, ?)
                    ", [$storeId, $categoryId]);
                }
            }
            
            $db->transComplete();
            
            return $db->transStatus();
            
        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Failed to update store categories: ' . $e->getMessage());
            return false;
        }
    }

   /**
 * Generate unique category ID
 */
public function generateCategoryId()
{
    // Use the same UUID generation method as your import tool
    $data = random_bytes(16);
    
    // Set version (4) and variant bits according to RFC 4122
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant 10
    
    return sprintf('%08s-%04s-%04s-%04s-%012s',
        bin2hex(substr($data, 0, 4)),
        bin2hex(substr($data, 4, 2)),
        bin2hex(substr($data, 6, 2)),
        bin2hex(substr($data, 8, 2)),
        bin2hex(substr($data, 10, 6))
    );
}

    /**
     * Create new category with auto-generated ID
     */
    public function createCategory($data)
    {
        // Generate unique category_id if not provided
        if (!isset($data['category_id']) || empty($data['category_id'])) {
            $data['category_id'] = $this->generateCategoryId();
        }
        
        // Set default values
        $data['status'] = $data['status'] ?? 1;
        $data['type'] = $data['type'] ?? 0;
        $data['top_category'] = $data['top_category'] ?? 0;
        
        return $this->insert($data);
    }

    /**
     * Search categories
     */
    public function searchCategories($searchTerm, $limit = 50)
    {
        return $this->like('title', $searchTerm)
                    ->where('status', 1)
                    ->orderBy('title', 'ASC')
                    ->limit($limit)
                    ->findAll();
    }

    /**
 * Get category statistics - FIXED for meaningful coverage calculation
 */
public function getCategoryStats()
{
    $db = \Config\Database::connect();
    
    $stats = $db->query("
        SELECT 
            COUNT(*) as total_categories,
            SUM(CASE WHEN status = 1 THEN 1 ELSE 0 END) as active_categories,
            SUM(CASE WHEN top_category = 1 THEN 1 ELSE 0 END) as top_categories
        FROM category
    ")->getRowArray();
    
    // Get meaningful store assignment counts
    $storeStats = $db->query("
        SELECT 
            COUNT(DISTINCT s.store_id) as total_active_stores,
            COUNT(DISTINCT sc.store_id) as stores_with_assignments,
            COUNT(DISTINCT CASE WHEN c.status = 1 THEN sc.store_id END) as stores_with_active_categories
        FROM store s
        LEFT JOIN store_category sc ON s.store_id = sc.store_id
        LEFT JOIN category c ON sc.category_id = c.category_id
        WHERE s.status = 1
    ")->getRowArray();
    
    // Check if most assignments are just the default import category
    $defaultCategoryCheck = $db->query("
        SELECT 
            COUNT(DISTINCT sc.store_id) as stores_with_default_only
        FROM store_category sc
        INNER JOIN store s ON sc.store_id = s.store_id
        WHERE s.status = 1
        AND sc.category_id = 'b71ff455-4573-11ea-bef6-f0d4e6234c'
        AND sc.store_id NOT IN (
            SELECT DISTINCT sc2.store_id 
            FROM store_category sc2 
            WHERE sc2.category_id != 'b71ff455-4573-11ea-bef6-f0d4e6234c'
        )
    ")->getRowArray();
    
    // Merge stats
    $stats = array_merge($stats, $storeStats);
    
    // Calculate meaningful coverage (excludes stores with only default import category)
    $meaningfullyAssigned = $stats['stores_with_active_categories'] - ($defaultCategoryCheck['stores_with_default_only'] ?? 0);
    
    $stats['stores_without_categories'] = $stats['total_active_stores'] - $meaningfullyAssigned;
    $stats['category_coverage_percent'] = $stats['total_active_stores'] > 0 
        ? round(($meaningfullyAssigned / $stats['total_active_stores']) * 100, 1)
        : 0;
    
    // Add helpful context
    $stats['default_only_assignments'] = $defaultCategoryCheck['stores_with_default_only'] ?? 0;
        
    return $stats;
}
    /**
     * Bulk operations for categories
     */
    public function bulkUpdateStatus($categoryIds, $status)
    {
        if (empty($categoryIds)) {
            return false;
        }
        
        $placeholders = str_repeat('?,', count($categoryIds) - 1) . '?';
        $db = \Config\Database::connect();
        
        return $db->query("
            UPDATE category 
            SET status = ?, update_date = NOW() 
            WHERE category_id IN ($placeholders)
        ", array_merge([$status], $categoryIds));
    }

    /**
     * Bulk assign categories to stores
     */
    public function bulkAssignCategories($storeIds, $categoryIds)
    {
        if (empty($storeIds) || empty($categoryIds)) {
            return false;
        }
        
        $db = \Config\Database::connect();
        $db->transStart();
        
        try {
            foreach ($storeIds as $storeId) {
                foreach ($categoryIds as $categoryId) {
                    // Check if assignment already exists
                    $existing = $db->query("
                        SELECT id FROM store_category 
                        WHERE store_id = ? AND category_id = ?
                    ", [$storeId, $categoryId])->getRowArray();
                    
                    if (!$existing) {
                        $db->query("
                            INSERT INTO store_category (store_id, category_id) 
                            VALUES (?, ?)
                        ", [$storeId, $categoryId]);
                    }
                }
            }
            
            $db->transComplete();
            return $db->transStatus();
            
        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Bulk category assignment failed: ' . $e->getMessage());
            return false;
        }
    }
}