<?php

namespace App\Models;

use CodeIgniter\Model;

class BusinessUserModel extends Model
{
    protected $table = 'business_users';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'store_id',
        'username',
        'email',
        'password',
        'first_name',
        'last_name',
        'phone',
        'role',
        'status',
        'email_verified',
        'verification_token',
        'last_login'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    protected $deletedField = 'deleted_at';

    // Validation
    protected $validationRules = [
        'store_id' => 'required|max_length[100]',
        'username' => 'required|min_length[3]|max_length[50]|is_unique[business_users.username,id,{id}]',
        'email' => 'required|valid_email|max_length[255]|is_unique[business_users.email,id,{id}]',
        'password' => 'required|min_length[6]|max_length[255]',
        'first_name' => 'permit_empty|max_length[100]',
        'last_name' => 'permit_empty|max_length[100]',
        'phone' => 'permit_empty|max_length[20]',
        'role' => 'required|in_list[owner,manager]',
        'status' => 'required|in_list[0,1]'
    ];

    protected $validationMessages = [
        'store_id' => [
            'required' => 'Store ID is required.'
        ],
        'username' => [
            'required' => 'Username is required.',
            'min_length' => 'Username must be at least 3 characters long.',
            'max_length' => 'Username cannot exceed 50 characters.',
            'is_unique' => 'This username is already taken.'
        ],
        'email' => [
            'required' => 'Email address is required.',
            'valid_email' => 'Please enter a valid email address.',
            'is_unique' => 'This email address is already registered.'
        ],
        'password' => [
            'required' => 'Password is required.',
            'min_length' => 'Password must be at least 6 characters long.'
        ],
        'role' => [
            'required' => 'Role is required.',
            'in_list' => 'Role must be either owner or manager.'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = ['hashPassword'];
    protected $beforeUpdate = ['hashPassword'];

    /**
     * Hash password before saving to database
     */
    protected function hashPassword(array $data)
    {
        if (isset($data['data']['password'])) {
            $data['data']['password'] = password_hash($data['data']['password'], PASSWORD_DEFAULT);
        }
        return $data;
    }

    /**
     * Get business user by username
     */
    public function getUserByUsername($username)
    {
        return $this->where('username', $username)
                    ->where('status', 1)
                    ->first();
    }

    /**
     * Get business user by email
     */
    public function getUserByEmail($email)
    {
        return $this->where('email', $email)
                    ->where('status', 1)
                    ->first();
    }

    /**
     * Get business user with store information
     */
    public function getUserWithStore($userId)
    {
        return $this->select('business_users.*, store.name as store_name, store.address, store.city, store.state')
                    ->join('store', 'store.store_id = business_users.store_id', 'left')
                    ->where('business_users.id', $userId)
                    ->where('business_users.status', 1)
                    ->first();
    }

    /**
     * Get all users for a specific store
     */
    public function getUsersByStore($storeId)
    {
        return $this->where('store_id', $storeId)
                    ->where('status', 1)
                    ->orderBy('role', 'ASC')
                    ->findAll();
    }

    /**
     * Check if username exists
     */
    public function usernameExists($username, $excludeId = null)
    {
        $builder = $this->where('username', $username);
        
        if ($excludeId) {
            $builder->where('id !=', $excludeId);
        }
        
        return $builder->countAllResults() > 0;
    }

    /**
     * Check if email exists
     */
    public function emailExists($email, $excludeId = null)
    {
        $builder = $this->where('email', $email);
        
        if ($excludeId) {
            $builder->where('id !=', $excludeId);
        }
        
        return $builder->countAllResults() > 0;
    }

    /**
     * Create new business user with hashed password
     */
    public function createUser($data)
    {
        // Ensure username is unique
        if ($this->usernameExists($data['username'])) {
            return false;
        }

        // Ensure email is unique (if provided)
        if (!empty($data['email']) && $this->emailExists($data['email'])) {
            return false;
        }

        // Set defaults
        $data['status'] = $data['status'] ?? 1;
        $data['role'] = $data['role'] ?? 'owner';
        $data['email_verified'] = $data['email_verified'] ?? 0;

        return $this->insert($data);
    }

    /**
     * Update user password
     */
    public function updatePassword($userId, $newPassword)
    {
        return $this->update($userId, [
            'password' => password_hash($newPassword, PASSWORD_DEFAULT)
        ]);
    }

    /**
     * Update last login timestamp
     */
    public function updateLastLogin($userId)
    {
        return $this->update($userId, [
            'last_login' => date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Set verification token for password reset
     */
    public function setVerificationToken($userId, $token)
    {
        return $this->update($userId, [
            'verification_token' => $token
        ]);
    }

    /**
     * Get user by verification token
     */
    public function getUserByToken($token)
    {
        return $this->where('verification_token', $token)
                    ->where('status', 1)
                    ->first();
    }

    /**
     * Clear verification token
     */
    public function clearVerificationToken($userId)
    {
        return $this->update($userId, [
            'verification_token' => null
        ]);
    }

    /**
     * Activate/deactivate user
     */
    public function setUserStatus($userId, $status)
    {
        return $this->update($userId, [
            'status' => $status ? 1 : 0
        ]);
    }
}