<?php

namespace App\Models;

use CodeIgniter\Model;

class BusinessJobModel extends Model
{
    protected $table = 'business_jobs';
    protected $primaryKey = 'id';
    
    protected $allowedFields = [
        'store_id',
        'title',
        'description',
        'employment_type',
        'salary_min',
        'salary_max',
        'salary_type',
        'contact_method',
        'contact_value',
        'is_active',
        'expires_at',
        'created_by',
        'times_renewed',
        'renewal_email_sent'
    ];
    
    protected $useTimestamps = true;
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';
    
    protected $validationRules = [
        'store_id' => 'required|max_length[100]',
        'title' => 'required|min_length[3]|max_length[255]',
        'description' => 'required|min_length[10]',
        'employment_type' => 'required|in_list[full_time,part_time,contract,seasonal]',
        'salary_min' => 'required|numeric|greater_than[0]',
        'salary_max' => 'permit_empty|numeric',
        'salary_type' => 'required|in_list[hourly,monthly,yearly]',
        'contact_method' => 'required|in_list[phone,email,walk_in,online]',
        'contact_value' => 'permit_empty|max_length[255]',
        'is_active' => 'permit_empty|in_list[0,1]'
        
    ];
    
    protected $validationMessages = [
        'store_id' => [
            'required' => 'Store ID is required'
        ],
        'title' => [
            'required' => 'Job title is required',
            'min_length' => 'Job title must be at least 3 characters',
            'max_length' => 'Job title cannot exceed 255 characters'
        ],
        'description' => [
            'required' => 'Job description is required',
            'min_length' => 'Job description must be at least 10 characters'
        ],
        'employment_type' => [
            'required' => 'Employment type is required',
            'in_list' => 'Please select a valid employment type'
        ],
        'salary_min' => [
            'required' => 'Starting wage is required',
            'numeric' => 'Starting wage must be a number',
            'greater_than' => 'Starting wage must be greater than 0'
        ],
        'salary_max' => [
            'numeric' => 'Maximum wage must be a number',
            'greater_than_equal_to' => 'Maximum wage must be greater than or equal to starting wage'
        ],
        'salary_type' => [
            'required' => 'Salary type is required',
            'in_list' => 'Please select a valid salary type'
        ],
        'contact_method' => [
            'required' => 'Contact method is required',
            'in_list' => 'Please select a valid contact method'
        ]
    ];

    /**
     * Get active jobs for a store (up to specified limit)
     */
    public function getActiveJobs($storeId, $limit = 3)
    {
        return $this->where('store_id', $storeId)
                   ->where('is_active', 1)
                   ->where('expires_at >', date('Y-m-d H:i:s'))
                   ->orderBy('created_at', 'DESC')
                   ->limit($limit)
                   ->findAll();
    }

    /**
     * Get count of active jobs for a store
     */
    public function getActiveJobCount($storeId)
    {
        return $this->where('store_id', $storeId)
                   ->where('is_active', 1)
                   ->where('expires_at >', date('Y-m-d H:i:s'))
                   ->countAllResults();
    }

    /**
     * Check if store has any active jobs
     */
    public function hasActiveJobs($storeId)
    {
        return $this->getActiveJobCount($storeId) > 0;
    }

    /**
     * Get jobs for a store within date range
     */
    public function getStoreJobs($storeId, $startDate = null, $endDate = null)
    {
        $builder = $this->where('store_id', $storeId);
        
        if ($startDate) {
            $builder->where('created_at >=', $startDate);
        }
        
        if ($endDate) {
            $builder->where('created_at <=', $endDate . ' 23:59:59');
        }
        
        return $builder->orderBy('created_at', 'DESC')->findAll();
    }

    /**
     * Get jobs expiring within specified days
     */
    public function getExpiringJobs($storeId, $days = 5)
    {
        $endDate = date('Y-m-d H:i:s', strtotime("+{$days} days"));
        
        return $this->where('store_id', $storeId)
                   ->where('is_active', 1)
                   ->where('expires_at <=', $endDate)
                   ->where('expires_at >', date('Y-m-d H:i:s'))
                   ->orderBy('expires_at', 'ASC')
                   ->findAll();
    }

    /**
     * Get jobs that need renewal email (expiring in 2 days, email not sent)
     */
    public function getJobsNeedingRenewalEmail()
    {
        $targetDate = date('Y-m-d H:i:s', strtotime('+2 days'));
        
        return $this->select('business_jobs.*, store.name as store_name, store.email as store_email, business_users.email as user_email, business_users.first_name, business_users.last_name')
                   ->join('store', 'store.store_id = business_jobs.store_id')
                   ->join('business_users', 'business_users.id = business_jobs.created_by', 'left')
                   ->where('business_jobs.is_active', 1)
                   ->where('business_jobs.expires_at <=', $targetDate)
                   ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'))
                   ->where('business_jobs.renewal_email_sent', 0)
                   ->findAll();
    }

    /**
     * Mark renewal email as sent
     */
    public function markRenewalEmailSent($jobId)
    {
        return $this->update($jobId, ['renewal_email_sent' => 1]);
    }

    /**
     * Get expired jobs that should be deactivated
     */
    public function getExpiredJobs()
    {
        return $this->where('is_active', 1)
                   ->where('expires_at <=', date('Y-m-d H:i:s'))
                   ->findAll();
    }

    /**
     * Deactivate expired jobs
     */
    public function deactivateExpiredJobs()
    {
        return $this->where('is_active', 1)
                   ->where('expires_at <=', date('Y-m-d H:i:s'))
                   ->set(['is_active' => 0])
                   ->update();
    }

    /**
     * Get job statistics for a store
     */
    public function getJobStats($storeId)
    {
        $db = \Config\Database::connect();
        
        // Total jobs
        $total = $this->where('store_id', $storeId)->countAllResults(false);
        
        // Active jobs
        $active = $this->where('store_id', $storeId)
                      ->where('is_active', 1)
                      ->where('expires_at >', date('Y-m-d H:i:s'))
                      ->countAllResults(false);
        
        // Expired jobs
        $expired = $this->where('store_id', $storeId)
                       ->where('expires_at <=', date('Y-m-d H:i:s'))
                       ->countAllResults(false);
        
        // Jobs expiring soon (within 7 days)
        $expiringSoon = $this->where('store_id', $storeId)
                            ->where('is_active', 1)
                            ->where('expires_at <=', date('Y-m-d H:i:s', strtotime('+7 days')))
                            ->where('expires_at >', date('Y-m-d H:i:s'))
                            ->countAllResults();
        
        return [
            'total' => $total,
            'active' => $active,
            'expired' => $expired,
            'expiring_soon' => $expiringSoon
        ];
    }

    /**
     * Format job for display with calculated fields
     */
    public function formatJobDisplay($job)
    {
        if (empty($job)) {
            return null;
        }

        // Format salary display
        $salaryDisplay = $this->formatSalaryDisplay($job['salary_min'], $job['salary_max'], $job['salary_type']);
        
        // Format employment type
        $employmentTypeDisplay = ucwords(str_replace('_', ' ', $job['employment_type']));
        
        // Format contact method
        $contactMethodDisplay = $this->formatContactMethod($job, $job['store_id'] ?? null);
        
        // Calculate days until expiration
        $daysUntilExpiration = null;
        if ($job['expires_at']) {
            $expirationTime = strtotime($job['expires_at']);
            $now = time();
            $daysUntilExpiration = max(0, ceil(($expirationTime - $now) / (24 * 60 * 60)));
        }
        
        // Format dates
        $createdAtFormatted = date('M j, Y', strtotime($job['created_at']));
        $expiresAtFormatted = $job['expires_at'] ? date('M j, Y', strtotime($job['expires_at'])) : null;
        
        return array_merge($job, [
            'salary_display' => $salaryDisplay,
            'employment_type_display' => $employmentTypeDisplay,
            'contact_method_display' => $contactMethodDisplay,
            'days_until_expiration' => $daysUntilExpiration,
            'created_at_formatted' => $createdAtFormatted,
            'expires_at_formatted' => $expiresAtFormatted,
            'times_renewed' => $job['times_renewed'] ?? 0
        ]);
    }

    /**
     * Format salary for display
     */
    private function formatSalaryDisplay($salaryMin, $salaryMax, $salaryType)
    {
        $typeMap = [
            'hourly' => '/hour',
            'monthly' => '/month',
            'yearly' => '/year'
        ];
        
        $suffix = $typeMap[$salaryType] ?? '';
        
        if ($salaryMax && $salaryMax > $salaryMin) {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                // Format yearly salaries with K suffix for readability
                $minDisplay = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                $maxDisplay = $salaryMax >= 1000 ? number_format($salaryMax / 1000, 0) . 'K' : number_format($salaryMax, 0);
                return "$" . $minDisplay . "-" . $maxDisplay . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "-" . number_format($salaryMax, 2) . $suffix;
            }
        } else {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                $display = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                return "$" . $display . "+" . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "+" . $suffix;
            }
        }
    }

    /**
     * Format contact method for display
     */
    private function formatContactMethod($job, $storeId = null)
    {
        switch ($job['contact_method']) {
            case 'phone':
                if ($storeId) {
                    $storeModel = new StoreModel();
                    $store = $storeModel->getStoreById($storeId);
                    return $store ? "Call: " . $store['phone'] : "Call us";
                }
                return "Call us";
                
            case 'email':
                if ($storeId) {
                    $storeModel = new StoreModel();
                    $store = $storeModel->getStoreById($storeId);
                    return $store ? "Email: " . $store['email'] : "Email us";
                }
                return "Email us";
                
            case 'walk_in':
                return "Visit us in person";
                
            case 'online':
                return $job['contact_value'] ? "Apply online: " . $job['contact_value'] : "Apply online";
                
            default:
                return "Contact us";
        }
    }

    /**
     * Get job with creator information
     */
    public function getJobWithCreator($jobId)
    {
        return $this->select('business_jobs.*, business_users.first_name, business_users.last_name, business_users.email as user_email')
                   ->join('business_users', 'business_users.id = business_jobs.created_by', 'left')
                   ->find($jobId);
    }

    /**
     * Get all active jobs across all stores (for admin)
     */
    public function getAllActiveJobs($limit = null)
    {
        $builder = $this->select('business_jobs.*, store.name as store_name, store.city, store.state')
                       ->join('store', 'store.store_id = business_jobs.store_id')
                       ->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'))
                       ->orderBy('business_jobs.created_at', 'DESC');
        
        if ($limit) {
            $builder->limit($limit);
        }
        
        return $builder->findAll();
    }

    /**
     * Search jobs by keyword and location
     */
    public function searchJobs($keyword = null, $city = null, $state = null, $employmentType = null, $limit = 20)
    {
        $builder = $this->select('business_jobs.*, store.name as store_name, store.address, store.city, store.state, store.phone, store.email')
                       ->join('store', 'store.store_id = business_jobs.store_id')
                       ->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'))
                       ->where('store.status', 1);
        
        if ($keyword) {
            $builder->groupStart()
                   ->like('business_jobs.title', $keyword)
                   ->orLike('business_jobs.description', $keyword)
                   ->orLike('store.name', $keyword)
                   ->groupEnd();
        }
        
        if ($city) {
            $builder->where('store.city', $city);
        }
        
        if ($state) {
            $builder->where('store.state', $state);
        }
        
        if ($employmentType) {
            $builder->where('business_jobs.employment_type', $employmentType);
        }
        
        return $builder->orderBy('business_jobs.created_at', 'DESC')
                      ->limit($limit)
                      ->findAll();
    }
}