<?php

namespace App\Models;

use CodeIgniter\Model;

class BusinessCouponModel extends Model
{
    protected $table = 'business_coupons';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'store_id',
        'title',
        'description',
        'discount_type',
        'discount_value',
        'minimum_purchase',
        'usage_limit',
        'usage_count',
        'starts_at',
        'expires_at',
        'is_active',
        'created_by'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    // Validation - SIMPLIFIED to remove problematic rules
    protected $validationRules = [
        'store_id' => 'required|max_length[100]',
        'title' => 'required|min_length[3]|max_length[255]',
        'description' => 'required|min_length[10]',
        'discount_type' => 'required|in_list[percentage,fixed_amount,bogo,free_item]',
        'discount_value' => 'permit_empty|numeric|greater_than[0]',  // Changed to permit_empty and numeric
        'minimum_purchase' => 'permit_empty|numeric',  // Simplified
        'usage_limit' => 'permit_empty|is_natural',
        'starts_at' => 'required',  // Simplified - just required
        'expires_at' => 'required',  // Simplified - just required
        'is_active' => 'permit_empty|in_list[0,1]'  // Allow empty for updates
    ];

    protected $validationMessages = [
        'title' => [
            'required' => 'Coupon title is required.',
            'min_length' => 'Title must be at least 3 characters long.'
        ],
        'description' => [
            'required' => 'Coupon description is required.',
            'min_length' => 'Description must be at least 10 characters long.'
        ],
        'discount_value' => [
            'numeric' => 'Discount value must be a number.',
            'greater_than' => 'Discount value must be greater than 0.'
        ]
    ];

    /**
     * Get active coupon for a store
     */
    public function getActiveCoupon($storeId)
    {
        $today = date('Y-m-d');
        
        return $this->where('store_id', $storeId)
                    ->where('is_active', 1)
                    ->where('starts_at <=', $today)
                    ->where('expires_at >=', $today)
                    ->orderBy('created_at', 'DESC')
                    ->first();
    }

    /**
     * Get all coupons for a store (for calendar view)
     */
    public function getStoreCoupons($storeId, $startDate = null, $endDate = null)
    {
        $builder = $this->where('store_id', $storeId);
        
        if ($startDate && $endDate) {
            $builder->groupStart()
                    ->where('starts_at >=', $startDate)
                    ->where('starts_at <=', $endDate)
                    ->orGroupStart()
                        ->where('expires_at >=', $startDate)
                        ->where('expires_at <=', $endDate)
                    ->groupEnd()
                    ->groupEnd();
        }
        
        return $builder->orderBy('starts_at', 'ASC')->findAll();
    }

    /**
     * Get upcoming coupons for a store
     */
    public function getUpcomingCoupons($storeId, $limit = 5)
    {
        $today = date('Y-m-d');
        
        return $this->where('store_id', $storeId)
                    ->where('starts_at >', $today)
                    ->where('is_active', 1)
                    ->orderBy('starts_at', 'ASC')
                    ->limit($limit)
                    ->findAll();
    }

    /**
     * Get expired coupons for a store
     */
    public function getExpiredCoupons($storeId, $limit = 10)
    {
        $today = date('Y-m-d');
        
        return $this->where('store_id', $storeId)
                    ->where('expires_at <', $today)
                    ->orderBy('expires_at', 'DESC')
                    ->limit($limit)
                    ->findAll();
    }

    /**
     * Check if store has active coupon in date range
     */
    public function hasActiveCouponInRange($storeId, $startDate, $endDate, $excludeId = null)
    {
        $builder = $this->where('store_id', $storeId)
                        ->where('is_active', 1)
                        ->groupStart()
                            ->groupStart()
                                ->where('starts_at >=', $startDate)
                                ->where('starts_at <=', $endDate)
                            ->groupEnd()
                            ->orGroupStart()
                                ->where('expires_at >=', $startDate)
                                ->where('expires_at <=', $endDate)
                            ->groupEnd()
                            ->orGroupStart()
                                ->where('starts_at <=', $startDate)
                                ->where('expires_at >=', $endDate)
                            ->groupEnd()
                        ->groupEnd();
        
        if ($excludeId) {
            $builder->where('id !=', $excludeId);
        }
        
        return $builder->countAllResults() > 0;
    }

    /**
     * Increment usage count
     */
    public function incrementUsage($couponId)
    {
        $coupon = $this->find($couponId);
        
        if (!$coupon) {
            return false;
        }
        
        // Check if usage limit reached
        if ($coupon['usage_limit'] !== null && $coupon['usage_count'] >= $coupon['usage_limit']) {
            return false;
        }
        
        return $this->update($couponId, [
            'usage_count' => $coupon['usage_count'] + 1
        ]);
    }

    /**
     * Get coupon statistics for a store
     */
    public function getCouponStats($storeId)
    {
        $today = date('Y-m-d');
        
        $stats = [
            'total' => $this->where('store_id', $storeId)->countAllResults(),
            'active' => $this->where('store_id', $storeId)
                             ->where('is_active', 1)
                             ->where('starts_at <=', $today)
                             ->where('expires_at >=', $today)
                             ->countAllResults(),
            'upcoming' => $this->where('store_id', $storeId)
                               ->where('starts_at >', $today)
                               ->where('is_active', 1)
                               ->countAllResults(),
            'expired' => $this->where('store_id', $storeId)
                              ->where('expires_at <', $today)
                              ->countAllResults()
        ];
        
        // Get total usage across all coupons
        $usageData = $this->selectSum('usage_count', 'total_usage')
                          ->where('store_id', $storeId)
                          ->first();
        
        $stats['total_usage'] = $usageData['total_usage'] ?? 0;
        
        return $stats;
    }

    /**
     * Deactivate overlapping coupons when creating new one
     */
    public function deactivateOverlappingCoupons($storeId, $startDate, $endDate, $excludeId = null)
    {
        $builder = $this->where('store_id', $storeId)
                        ->where('is_active', 1)
                        ->groupStart()
                            ->groupStart()
                                ->where('starts_at >=', $startDate)
                                ->where('starts_at <=', $endDate)
                            ->groupEnd()
                            ->orGroupStart()
                                ->where('expires_at >=', $startDate)
                                ->where('expires_at <=', $endDate)
                            ->groupEnd()
                            ->orGroupStart()
                                ->where('starts_at <=', $startDate)
                                ->where('expires_at >=', $endDate)
                            ->groupEnd()
                        ->groupEnd();
        
        if ($excludeId) {
            $builder->where('id !=', $excludeId);
        }
        
        $overlappingCoupons = $builder->findAll();
        
        foreach ($overlappingCoupons as $coupon) {
            $this->update($coupon['id'], ['is_active' => 0]);
        }
        
        return count($overlappingCoupons);
    }

    /**
     * Get coupon with creator information
     */
    public function getCouponWithCreator($couponId)
    {
        return $this->select('business_coupons.*, business_users.first_name, business_users.last_name, business_users.email as creator_email')
                    ->join('business_users', 'business_users.id = business_coupons.created_by', 'left')
                    ->where('business_coupons.id', $couponId)
                    ->first();
    }

    /**
     * Format coupon for display
     */
    public function formatCouponDisplay($coupon)
    {
        if (!$coupon) {
            return null;
        }
        
        // Format discount display
        switch ($coupon['discount_type']) {
            case 'percentage':
                $coupon['discount_display'] = $coupon['discount_value'] . '% OFF';
                break;
            case 'fixed_amount':
                $coupon['discount_display'] = '$' . number_format($coupon['discount_value'], 2) . ' OFF';
                break;
            case 'bogo':
                $coupon['discount_display'] = 'Buy One Get One';
                break;
            case 'free_item':
                $coupon['discount_display'] = 'FREE ITEM';
                break;
        }
        
        // Format dates
        $coupon['starts_at_formatted'] = date('M d, Y', strtotime($coupon['starts_at']));
        $coupon['expires_at_formatted'] = date('M d, Y', strtotime($coupon['expires_at']));
        
        // Calculate days remaining
        $today = time();
        $expiryTime = strtotime($coupon['expires_at']);
        $daysRemaining = ceil(($expiryTime - $today) / 86400);
        
        if ($daysRemaining > 0) {
            $coupon['days_remaining'] = $daysRemaining;
            $coupon['urgency'] = $daysRemaining <= 3 ? 'high' : ($daysRemaining <= 7 ? 'medium' : 'low');
        } else {
            $coupon['days_remaining'] = 0;
            $coupon['urgency'] = 'expired';
        }
        
        // Check if usage limit reached
        if ($coupon['usage_limit'] !== null) {
            $coupon['usage_remaining'] = max(0, $coupon['usage_limit'] - $coupon['usage_count']);
            $coupon['is_exhausted'] = $coupon['usage_remaining'] === 0;
        } else {
            $coupon['usage_remaining'] = null;
            $coupon['is_exhausted'] = false;
        }
        
        return $coupon;
    }
}