<?php
// =============================================================================
// EMAIL SERVICE HELPER CLASS
// File: app/Libraries/FeedbackEmailService.php
// =============================================================================

namespace App\Libraries;

use CodeIgniter\Email\Email;

class FeedbackEmailService
{
    protected $email;
    protected $db;
    
    public function __construct()
    {
        $this->email = \Config\Services::email();
        $this->db = \Config\Database::connect();
    }
    
    /**
     * Send new case notification to business
     */
    public function sendNewCaseNotification($caseId, $storeId)
    {
        try {
            // Get case and store data
            $data = $this->getCaseEmailData($caseId, $storeId);
            
            if (!$data['store']['email'] || !$data['store']['email_notifications']) {
                return false; // Email notifications disabled or no email
            }
            
            $subject = "🚨 New Customer Concern - Case {$data['case']['case_number']}";
            if ($data['case']['priority'] === 'urgent') {
                $subject = "🚨 URGENT: " . $subject;
            }
            
            $this->email->setTo($data['store']['email']);
            $this->email->setSubject($subject);
            $this->email->setMessage(view('emails/feedback/new_case', $data));
            
            $result = $this->email->send();
            
            // Log email attempt
            $this->logEmailAttempt('new_case', $caseId, $result);
            
            return $result;
            
        } catch (\Exception $e) {
            log_message('error', 'Failed to send new case email: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send daily summary email
     */
    public function sendDailySummary($storeId)
    {
        try {
            $store = $this->getStoreData($storeId);
            
            if (!$store['email'] || !$store['email_notifications']) {
                return false;
            }
            
            // Get yesterday's feedback data
            $summary = $this->getDailySummaryData($storeId);
            $recentFeedback = $this->getRecentFeedback($storeId, 1); // Last 24 hours
            
            $data = [
                'store' => $store,
                'summary' => $summary,
                'recent_feedback' => $recentFeedback
            ];
            
            $this->email->setTo($store['email']);
            $this->email->setSubject("Daily Feedback Summary - {$store['name']}");
            $this->email->setMessage(view('emails/feedback/daily_summary', $data));
            
            return $this->email->send();
            
        } catch (\Exception $e) {
            log_message('error', 'Failed to send daily summary: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send unresponded case reminder
     */
    public function sendCaseReminder($storeId)
    {
        try {
            $store = $this->getStoreData($storeId);
            
            if (!$store['email'] || !$store['email_notifications']) {
                return false;
            }
            
            // Get cases waiting more than 24 hours
            $overdueCases = $this->getOverdueCases($storeId);
            
            if (empty($overdueCases)) {
                return true; // No overdue cases, no email needed
            }
            
            $data = [
                'store' => $store,
                'overdue_cases' => $overdueCases
            ];
            
            $this->email->setTo($store['email']);
            $this->email->setSubject("⚠️ Customer Cases Need Response - {$store['name']}");
            $this->email->setMessage(view('emails/feedback/case_reminder', $data));
            
            return $this->email->send();
            
        } catch (\Exception $e) {
            log_message('error', 'Failed to send case reminder: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Send new message notification
     */
    public function sendNewMessageNotification($messageId, $caseId)
    {
        try {
            $message = $this->getMessageData($messageId);
            $data = $this->getCaseEmailData($caseId, $message['store_id']);
            $data['message'] = $message;
            
            if (!$data['store']['email'] || !$data['store']['email_notifications']) {
                return false;
            }
            
            $this->email->setTo($data['store']['email']);
            $this->email->setSubject("New Customer Reply - Case {$data['case']['case_number']}");
            $this->email->setMessage(view('emails/feedback/new_message', $data));
            
            return $this->email->send();
            
        } catch (\Exception $e) {
            log_message('error', 'Failed to send new message email: ' . $e->getMessage());
            return false;
        }
    }
    
    // Helper methods for data retrieval
    private function getCaseEmailData($caseId, $storeId)
    {
        $case = $this->db->query("
            SELECT fc.*, cf.overall_satisfaction as satisfaction_score, 
                   cf.net_promoter_score as nps_score, cf.comment as feedback_comment,
                   cf.customer_name, cf.customer_contact,
                   DATE_FORMAT(fc.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.case_id = ?
        ", [$caseId])->getRowArray();
        
        $store = $this->getStoreData($storeId);
        
        return ['case' => $case, 'store' => $store];
    }
    
    private function getStoreData($storeId)
    {
        return $this->db->query("
            SELECT store_id, name, email, email_notifications
            FROM store 
            WHERE store_id = ?
        ", [$storeId])->getRowArray();
    }
    
    private function getDailySummaryData($storeId)
    {
        return $this->db->query("
            SELECT 
                COUNT(*) as total_feedback,
                AVG(overall_satisfaction) as avg_satisfaction,
                AVG(net_promoter_score) as avg_nps,
                (SELECT COUNT(*) FROM feedback_cases WHERE store_id = ? AND DATE(created_at) = CURDATE()) as new_cases,
                (SELECT COUNT(*) FROM feedback_cases WHERE store_id = ? AND status IN ('open', 'in_progress')) as open_cases
            FROM customer_feedback 
            WHERE store_id = ? AND DATE(created_at) = CURDATE()
        ", [$storeId, $storeId, $storeId])->getRowArray();
    }
    
    private function getRecentFeedback($storeId, $days = 1)
    {
        return $this->db->query("
            SELECT overall_satisfaction, net_promoter_score, comment,
                   DATE_FORMAT(created_at, '%M %d at %l:%i %p') as created_at_formatted
            FROM customer_feedback 
            WHERE store_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
            ORDER BY created_at DESC
            LIMIT 10
        ", [$storeId, $days])->getResultArray();
    }
    
    private function getOverdueCases($storeId)
    {
        return $this->db->query("
            SELECT fc.case_id, fc.case_number, cf.customer_name, cf.overall_satisfaction as satisfaction_score,
                   cf.net_promoter_score as nps_score, cf.comment as feedback_comment,
                   TIMESTAMPDIFF(HOUR, fc.created_at, NOW()) as hours_waiting
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.store_id = ? AND fc.status = 'open' 
            AND fc.created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
            AND fc.case_id NOT IN (
                SELECT DISTINCT case_id FROM case_messages WHERE sender_type = 'business'
            )
            ORDER BY fc.created_at ASC
        ", [$storeId])->getResultArray();
    }
    
    private function getMessageData($messageId)
    {
        return $this->db->query("
            SELECT cm.*, fc.store_id,
                   DATE_FORMAT(cm.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted
            FROM case_messages cm
            JOIN feedback_cases fc ON cm.case_id = fc.case_id
            WHERE cm.message_id = ?
        ", [$messageId])->getRowArray();
    }
    
    private function logEmailAttempt($type, $relatedId, $success)
    {
        $this->db->query("
            INSERT INTO email_logs (email_type, related_id, success, created_at)
            VALUES (?, ?, ?, NOW())
        ", [$type, $relatedId, $success ? 1 : 0]);
    }
}