<?php

/**
 * Time Debug and Calibration Script
 * 
 * This script will help identify and fix timezone/time calculation issues
 * in the business directory system.
 */

namespace App\Controllers;

use App\Controllers\BaseController;

class TimeDebugController extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    public function index()
    {
        $this->debugTimeSettings();
    }

    /**
     * Comprehensive time debugging
     */
    private function debugTimeSettings()
    {
        echo "<h1>Business Directory Time Debug Report</h1>";
        echo "<style>
            body { font-family: Arial, sans-serif; margin: 20px; }
            .section { margin: 20px 0; padding: 15px; border: 1px solid #ccc; background: #f9f9f9; }
            .error { color: red; font-weight: bold; }
            .success { color: green; font-weight: bold; }
            .warning { color: orange; font-weight: bold; }
            table { border-collapse: collapse; width: 100%; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .code { background: #eee; padding: 5px; font-family: monospace; }
        </style>";

        // 1. Server Time Information
        echo "<div class='section'>";
        echo "<h2>1. Server Time Information</h2>";
        echo "<table>";
        echo "<tr><th>Setting</th><th>Value</th><th>Status</th></tr>";
        
        $serverTime = new \DateTime();
        $phpTimezone = date_default_timezone_get();
        $serverTimestamp = time();
        
        echo "<tr><td>Current Server Time</td><td>" . $serverTime->format('Y-m-d H:i:s T') . "</td><td class='success'>✓</td></tr>";
        echo "<tr><td>PHP Default Timezone</td><td>$phpTimezone</td><td>" . ($phpTimezone === 'America/New_York' ? "<span class='success'>✓ Correct</span>" : "<span class='warning'>⚠ Should be America/New_York</span>") . "</td></tr>";
        echo "<tr><td>Server Timestamp</td><td>$serverTimestamp</td><td class='success'>✓</td></tr>";
        echo "<tr><td>date('N') - Day of Week</td><td>" . date('N') . " (" . date('l') . ")</td><td class='success'>✓</td></tr>";
        echo "<tr><td>Current Time (H:i:s)</td><td>" . date('H:i:s') . "</td><td class='success'>✓</td></tr>";
        echo "</table>";
        echo "</div>";

        // 2. Database Time Settings
        echo "<div class='section'>";
        echo "<h2>2. Database Time Settings</h2>";
        echo "<table>";
        echo "<tr><th>Setting</th><th>Value</th><th>Status</th></tr>";
        
        try {
            $dbTime = $this->db->query("SELECT NOW() as db_time, @@time_zone as db_timezone")->getRowArray();
            $dbTimestamp = $this->db->query("SELECT UNIX_TIMESTAMP() as db_timestamp")->getRowArray();
            
            echo "<tr><td>Database Time</td><td>{$dbTime['db_time']}</td><td class='success'>✓</td></tr>";
            echo "<tr><td>Database Timezone</td><td>{$dbTime['db_timezone']}</td><td>" . (in_array($dbTime['db_timezone'], ['SYSTEM', '-05:00', '-04:00']) ? "<span class='success'>✓</span>" : "<span class='warning'>⚠ Check timezone</span>") . "</td></tr>";
            echo "<tr><td>Database Timestamp</td><td>{$dbTimestamp['db_timestamp']}</td><td class='success'>✓</td></tr>";
            
            $timeDiff = abs($serverTimestamp - $dbTimestamp['db_timestamp']);
            echo "<tr><td>Server/DB Time Difference</td><td>{$timeDiff} seconds</td><td>" . ($timeDiff < 60 ? "<span class='success'>✓ Synced</span>" : "<span class='error'>✗ Out of sync</span>") . "</td></tr>";
            
        } catch (\Exception $e) {
            echo "<tr><td colspan='3' class='error'>Database Error: " . $e->getMessage() . "</td></tr>";
        }
        echo "</table>";
        echo "</div>";

        // 3. Business Timezone Testing
        echo "<div class='section'>";
        echo "<h2>3. Business Timezone Testing</h2>";
        $this->testBusinessTimezones();
        echo "</div>";

        // 4. Next Opening Time Logic Testing
        echo "<div class='section'>";
        echo "<h2>4. Next Opening Time Logic Testing</h2>";
        $this->testNextOpeningLogic();
        echo "</div>";

        // 5. Sample Business Status Check
        echo "<div class='section'>";
        echo "<h2>5. Sample Business Status Check</h2>";
        $this->testSampleBusinesses();
        echo "</div>";

        // 6. Recommendations
        echo "<div class='section'>";
        echo "<h2>6. Recommendations & Fixes</h2>";
        $this->provideRecommendations();
        echo "</div>";
    }

    /**
     * Test business timezone handling
     */
    private function testBusinessTimezones()
    {
        echo "<table>";
        echo "<tr><th>Test</th><th>Expected</th><th>Actual</th><th>Status</th></tr>";

        // Test Eastern Time creation
        try {
            $easternTz = new \DateTimeZone('America/New_York');
            $easternTime = new \DateTime('now', $easternTz);
            echo "<tr><td>Eastern Time Object</td><td>Current Eastern Time</td><td>" . $easternTime->format('Y-m-d H:i:s T') . "</td><td class='success'>✓</td></tr>";
            
            // Test day of week calculation
            $dayOfWeek = $easternTime->format('N');
            echo "<tr><td>Eastern Day of Week</td><td>1-7 (Monday-Sunday)</td><td>$dayOfWeek (" . $easternTime->format('l') . ")</td><td class='success'>✓</td></tr>";
            
        } catch (\Exception $e) {
            echo "<tr><td>Eastern Timezone Test</td><td>Success</td><td class='error'>Error: " . $e->getMessage() . "</td><td class='error'>✗</td></tr>";
        }

        // Test business timezone lookup from database
        try {
            $sampleStore = $this->db->query("
                SELECT s.store_id, s.name, s.store_timezone, tz.name as timezone_name 
                FROM store s 
                LEFT JOIN timezone tz ON s.store_timezone = tz.id 
                LIMIT 1
            ")->getRowArray();
            
            if ($sampleStore) {
                echo "<tr><td>Sample Store Timezone</td><td>Valid timezone</td><td>{$sampleStore['timezone_name']} (ID: {$sampleStore['store_timezone']})</td><td class='success'>✓</td></tr>";
                
                if ($sampleStore['timezone_name']) {
                    $storeTz = new \DateTimeZone($sampleStore['timezone_name']);
                    $storeTime = new \DateTime('now', $storeTz);
                    echo "<tr><td>Store Local Time</td><td>Current time in store timezone</td><td>" . $storeTime->format('Y-m-d H:i:s T') . "</td><td class='success'>✓</td></tr>";
                }
            }
        } catch (\Exception $e) {
            echo "<tr><td>Business Timezone Lookup</td><td>Success</td><td class='error'>Error: " . $e->getMessage() . "</td><td class='error'>✗</td></tr>";
        }

        echo "</table>";
    }

    /**
     * Test the next opening time logic
     */
    private function testNextOpeningLogic()
    {
        echo "<h3>Testing Next Opening Time Calculation Logic</h3>";
        
        // Simulate a business that's closed now but opens tomorrow
        $currentTime = new \DateTime('now', new \DateTimeZone('America/New_York'));
        echo "<p><strong>Current Eastern Time:</strong> " . $currentTime->format('Y-m-d H:i:s T l') . "</p>";
        
        echo "<h4>Test Scenarios:</h4>";
        echo "<table>";
        echo "<tr><th>Scenario</th><th>Current Time</th><th>Next Open</th><th>Expected Result</th><th>Actual Result</th><th>Status</th></tr>";
        
        // Scenario 1: Business closed at 7:16 PM, opens tomorrow at 9:00 AM
        $testTime1 = new \DateTime('2024-01-15 19:16:00', new \DateTimeZone('America/New_York')); // Monday 7:16 PM
        $result1 = $this->calculateNextOpenForTest($testTime1, '09:00:00', 2); // Opens Tuesday at 9 AM
        echo "<tr><td>Evening, opens next day</td><td>Mon 7:16 PM</td><td>Tue 9:00 AM</td><td>Opens tomorrow at 9:00 AM</td><td>$result1</td><td>" . 
             (strpos($result1, 'tomorrow') !== false ? "<span class='success'>✓</span>" : "<span class='error'>✗</span>") . "</td></tr>";
        
        // Scenario 2: Business closed Sunday, opens Monday
        $testTime2 = new \DateTime('2024-01-14 19:16:00', new \DateTimeZone('America/New_York')); // Sunday 7:16 PM
        $result2 = $this->calculateNextOpenForTest($testTime2, '09:00:00', 1); // Opens Monday at 9 AM
        echo "<tr><td>Sunday evening, opens Monday</td><td>Sun 7:16 PM</td><td>Mon 9:00 AM</td><td>Opens tomorrow at 9:00 AM</td><td>$result2</td><td>" . 
             (strpos($result2, 'tomorrow') !== false ? "<span class='success'>✓</span>" : "<span class='error'>✗</span>") . "</td></tr>";
        
        // Scenario 3: Early morning, same day opening
        $testTime3 = new \DateTime('2024-01-15 06:00:00', new \DateTimeZone('America/New_York')); // Monday 6:00 AM
        $result3 = $this->calculateNextOpenForTest($testTime3, '09:00:00', 1); // Opens same day at 9 AM
        echo "<tr><td>Early morning, opens today</td><td>Mon 6:00 AM</td><td>Mon 9:00 AM</td><td>Opens today at 9:00 AM</td><td>$result3</td><td>" . 
             (strpos($result3, 'today') !== false ? "<span class='success'>✓</span>" : "<span class='error'>✗</span>") . "</td></tr>";
        
        echo "</table>";
    }

    /**
     * Calculate next opening time for testing
     */
    private function calculateNextOpenForTest($currentTime, $openTime, $targetDayOfWeek)
    {
        $currentDayOfWeek = $currentTime->format('N');
        $currentTimeStr = $currentTime->format('H:i:s');
        
        // Check today first
        if ($currentDayOfWeek == $targetDayOfWeek && $currentTimeStr < $openTime) {
            return "Opens today at " . date('g:i A', strtotime($openTime));
        }
        
        // Calculate days until next occurrence
        $daysUntil = ($targetDayOfWeek - $currentDayOfWeek + 7) % 7;
        if ($daysUntil == 0) {
            $daysUntil = 7; // Next week
        }
        
        $nextOpenDate = clone $currentTime;
        $nextOpenDate->add(new \DateInterval("P{$daysUntil}D"));
        
        if ($daysUntil == 1) {
            return "Opens tomorrow at " . date('g:i A', strtotime($openTime));
        } else {
            return "Opens " . $nextOpenDate->format('l') . " at " . date('g:i A', strtotime($openTime));
        }
    }

    /**
     * Test actual businesses from database
     */
    private function testSampleBusinesses()
    {
        echo "<h3>Live Business Status Testing</h3>";
        
        try {
            // Get a few sample businesses
            $businesses = $this->db->query("
                SELECT DISTINCT s.store_id, s.name, s.store_timezone, tz.name as timezone_name
                FROM store s 
                LEFT JOIN timezone tz ON s.store_timezone = tz.id 
                WHERE s.status = 1 
                LIMIT 5
            ")->getResultArray();
            
            echo "<table>";
            echo "<tr><th>Business</th><th>Store ID</th><th>Timezone</th><th>Current Status</th><th>Next Open Message</th><th>Analysis</th></tr>";
            
            foreach ($businesses as $business) {
                $storeId = $business['store_id'];
                $businessTimezone = $business['timezone_name'] ?? 'America/New_York';
                
                try {
                    $timezone = new \DateTimeZone($businessTimezone);
                    $now = new \DateTime('now', $timezone);
                    
                    // Get current status using your existing logic
                    $statusData = $this->calculateCurrentStatusForDebug($storeId);
                    $nextOpen = $this->getNextOpenTimeForDebug($storeId, $statusData);
                    
                    $analysis = "";
                    if ($statusData['status'] === 'closed') {
                        if (strpos($nextOpen, 'today') !== false && $now->format('H') >= 19) {
                            $analysis = "<span class='error'>⚠ BUG: Shows 'today' but it's " . $now->format('g:i A') . "</span>";
                        } elseif (strpos($nextOpen, 'tomorrow') !== false) {
                            $analysis = "<span class='success'>✓ Correct</span>";
                        } else {
                            $analysis = "<span class='warning'>? Check logic</span>";
                        }
                    } else {
                        $analysis = "<span class='success'>✓ Open</span>";
                    }
                    
                    echo "<tr>";
                    echo "<td>" . esc($business['name']) . "</td>";
                    echo "<td>" . esc($storeId) . "</td>";
                    echo "<td>$businessTimezone</td>";
                    echo "<td>" . ucfirst($statusData['status']) . "</td>";
                    echo "<td>$nextOpen</td>";
                    echo "<td>$analysis</td>";
                    echo "</tr>";
                    
                } catch (\Exception $e) {
                    echo "<tr><td colspan='6' class='error'>Error testing {$business['name']}: " . $e->getMessage() . "</td></tr>";
                }
            }
            
            echo "</table>";
            
        } catch (\Exception $e) {
            echo "<p class='error'>Error getting sample businesses: " . $e->getMessage() . "</p>";
        }
    }

    /**
     * Debug version of calculateCurrentStatus
     */
    private function calculateCurrentStatusForDebug($storeId)
    {
        $storeData = $this->db->query("
            SELECT s.name, s.store_timezone, tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }
        
        $dayOfWeek = $now->format('N');
        $currentTime = $now->format('H:i:s');
        $currentDate = $now->format('Y-m-d');

        // Check for exceptions first
        $exception = $this->db->query("
            SELECT * FROM store_schedule_exceptions 
            WHERE store_id = ? AND exception_date = ? 
            ORDER BY hour_type_id IS NULL DESC
            LIMIT 1
        ", [$storeId, $currentDate])->getRowArray();

        if ($exception && $exception['exception_type'] === 'closed') {
            return ['status' => 'closed', 'message' => $exception['message'] ?? 'Closed today', 'services' => []];
        }

        // Get all hour types that affect main status
        $statusTypes = $this->db->query("
            SELECT sht.id, sht.custom_name,
                   ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
            AND ss.day_of_week = ?
        ", [$storeId, $dayOfWeek])->getResultArray();

        $openServices = [];
        foreach ($statusTypes as $type) {
            if ($type['is_closed']) continue;

            $isOpen = false;
            if ($type['slot_1_open'] && $type['slot_1_close']) {
                if ($currentTime >= $type['slot_1_open'] && $currentTime <= $type['slot_1_close']) {
                    $isOpen = true;
                }
            }
            if (!$isOpen && $type['slot_2_open'] && $type['slot_2_close']) {
                if ($currentTime >= $type['slot_2_open'] && $currentTime <= $type['slot_2_close']) {
                    $isOpen = true;
                }
            }

            if ($isOpen) {
                $openServices[] = $type['custom_name'];
            }
        }

        if (empty($openServices)) {
            return ['status' => 'closed', 'services' => []];
        }

        return ['status' => 'open', 'services' => $openServices];
    }

    /**
     * Debug version of getNextOpenTime
     */
    private function getNextOpenTimeForDebug($storeId, $statusData)
    {
        if ($statusData['status'] !== 'closed') {
            return null;
        }

        $storeData = $this->db->query("
            SELECT tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }

        // Check today and next 7 days for next opening
        for ($i = 0; $i < 8; $i++) {
            $checkDate = clone $now;
            if ($i > 0) {
                $checkDate->add(new \DateInterval("P{$i}D"));
            }
            $dayOfWeek = $checkDate->format('N');
            
            $nextSchedule = $this->db->query("
                SELECT sht.custom_name, ss.slot_1_open, ss.slot_2_open
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
                AND ss.day_of_week = ? AND ss.is_closed = 0
                AND (ss.slot_1_open IS NOT NULL OR ss.slot_2_open IS NOT NULL)
                ORDER BY ss.slot_1_open, ss.slot_2_open
                LIMIT 1
            ", [$storeId, $dayOfWeek])->getRowArray();

            if ($nextSchedule) {
                $openTime = $nextSchedule['slot_1_open'] ?: $nextSchedule['slot_2_open'];
                
                if ($i === 0) {
                    // Today - check if opening time hasn't passed
                    $currentTime = $now->format('H:i:s');
                    if ($openTime > $currentTime) {
                        $timeDisplay = date('g:i A', strtotime($openTime));
                        return "Opens today at {$timeDisplay}";
                    }
                } else {
                    // Future day
                    $timeDisplay = date('g:i A', strtotime($openTime));
                    
                    if ($i === 1) {
                        return "Opens tomorrow at {$timeDisplay}";
                    } else {
                        $dayName = $checkDate->format('l');
                        return "Opens {$dayName} at {$timeDisplay}";
                    }
                }
            }
        }

        return 'Hours not available';
    }

    /**
     * Provide recommendations for fixing the issues
     */
    private function provideRecommendations()
    {
        echo "<h3>Identified Issues & Fixes</h3>";
        
        echo "<div class='warning'>";
        echo "<h4>Primary Issue: 'Opens today' Logic Bug</h4>";
        echo "<p>The next opening time calculation is not properly checking if the current time has passed the opening time for 'today'.</p>";
        echo "</div>";
        
        echo "<h4>Fixes Needed:</h4>";
        echo "<ol>";
        echo "<li><strong>StoreDetails.php - getNextOpenTime method:</strong><br>";
        echo "<div class='code'>Line ~380: if (\$openTime > \$currentTime) should be more strict about time comparison</div>";
        echo "</li>";
        
        echo "<li><strong>Locator.php - getNextOpenFromCRON method:</strong><br>";
        echo "<div class='code'>Similar logic needs fixing for CRON-based status updates</div>";
        echo "</li>";
        
        echo "<li><strong>Hours.php - calculateCurrentStatus method:</strong><br>";
        echo "<div class='code'>Ensure consistent timezone handling across all methods</div>";
        echo "</li>";
        echo "</ol>";
        
        echo "<h4>Recommended Code Changes:</h4>";
        echo "<div class='code'>";
        echo "// In getNextOpenTime methods, replace:<br>";
        echo "if (\$openTime > \$currentTime) {<br>";
        echo "&nbsp;&nbsp;return \"Opens today at {\$timeDisplay}\";<br>";
        echo "}<br><br>";
        echo "// With:<br>";
        echo "if (\$i === 0) {<br>";
        echo "&nbsp;&nbsp;// Today - check if opening time hasn't passed<br>";
        echo "&nbsp;&nbsp;\$currentTime = \$now->format('H:i:s');<br>";
        echo "&nbsp;&nbsp;\$openTimeSeconds = strtotime(\$openTime);<br>";
        echo "&nbsp;&nbsp;\$currentTimeSeconds = strtotime(\$currentTime);<br>";
        echo "&nbsp;&nbsp;if (\$openTimeSeconds > \$currentTimeSeconds) {<br>";
        echo "&nbsp;&nbsp;&nbsp;&nbsp;\$timeDisplay = date('g:i A', \$openTimeSeconds);<br>";
        echo "&nbsp;&nbsp;&nbsp;&nbsp;return \"Opens today at {\$timeDisplay}\";<br>";
        echo "&nbsp;&nbsp;}<br>";
        echo "}";
        echo "</div>";
        
        echo "<h4>Testing Commands:</h4>";
        echo "<ul>";
        echo "<li>Test this debug script at different times of day</li>";
        echo "<li>Check business cards at 6 AM, 12 PM, 6 PM, and 11 PM</li>";
        echo "<li>Verify CRON job is updating store_current_status table correctly</li>";
        echo "</ul>";
    }
}