<?php

namespace App\Controllers;

use App\Controllers\BaseController;

class Feedback extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    /**
     * QR Code landing page - handles URL like /feedback/qr_store123
     */
    public function qr($qrCode)
    {
        // DEBUG: Log the incoming request
        log_message('info', 'Feedback QR called with code: ' . $qrCode);
        log_message('info', 'Request method: ' . $this->request->getMethod());
        log_message('info', 'POST data: ' . print_r($this->request->getPost(), true));
        
        try {
            // Find store by QR code first
            $store = $this->db->query("
                SELECT store_id, name, feedback_enabled, google_review_url
                FROM store 
                WHERE feedback_qr_code = ? AND status = 1
            ", [$qrCode])->getRowArray();

            log_message('info', 'Store lookup result: ' . print_r($store, true));

            if (!$store) {
                log_message('error', 'Store not found for QR code: ' . $qrCode);
                return redirect()->to('/')->with('error', 'Invalid feedback link.');
            }

            if (!$store['feedback_enabled']) {
                log_message('error', 'Feedback disabled for store: ' . $store['store_id']);
                return redirect()->to('/')->with('error', 'Feedback is not currently available for this business.');
            }

            // Check if this is a POST submission
            if (strtolower($this->request->getMethod()) === 'post') {
                log_message('info', 'Processing POST submission for store: ' . $store['store_id']);
                return $this->submitFeedback($store);
            }

            // Show the form
            log_message('info', 'Showing feedback form for store: ' . $store['store_id']);
            $data = [
                'title' => 'Share Your Experience - ' . $store['name'],
                'store' => $store,
                'qr_code' => $qrCode
            ];

            return view('feedback/qr_form', $data);

        } catch (\Exception $e) {
            log_message('error', 'Feedback QR error: ' . $e->getMessage());
            log_message('error', 'Exception trace: ' . $e->getTraceAsString());
            return redirect()->to('/')->with('error', 'Unable to load feedback form.');
        }
    }

    /**
     * Process feedback submission
     */
    private function submitFeedback($store)
    {
        log_message('info', 'submitFeedback called for store: ' . $store['store_id']);
        log_message('info', 'POST data in submitFeedback: ' . print_r($this->request->getPost(), true));
        
        // Validation rules
        $rules = [
            'overall_satisfaction' => 'required|integer|greater_than[0]|less_than[11]',
            'net_promoter_score' => 'required|integer|greater_than[0]|less_than[11]',
            'comment' => 'permit_empty|max_length[1000]'
        ];
        
        if (!$this->validate($rules)) {
            log_message('error', 'Validation failed: ' . print_r($this->validator->getErrors(), true));
            return redirect()->back()->withInput()->with('validation', $this->validator->getErrors());
        }
        
        $satisfaction = (int)$this->request->getPost('overall_satisfaction');
        $nps = (int)$this->request->getPost('net_promoter_score');
        $comment = trim($this->request->getPost('comment')) ?: null;
        $customerName = trim($this->request->getPost('customer_name')) ?: null;
        $customerContact = trim($this->request->getPost('customer_contact')) ?: null;
        
        log_message('info', "Feedback data - Satisfaction: $satisfaction, NPS: $nps, Comment: $comment");
        
        // Generate feedback ID
        $feedbackId = $this->generateUniqueId('feedback_');
        log_message('info', 'Generated feedback ID: ' . $feedbackId);
        
        // Determine if this needs resolution (any score <= 6)
        $needsResolution = ($satisfaction <= 6 || $nps <= 6) && !empty($customerContact);
        
        // Check for spam prevention
        $ipAddress = $this->request->getIPAddress();
        $userAgent = $this->request->getUserAgent()->getAgentString();
        
        // Basic spam check - limit submissions per IP per hour
        $recentSubmissions = $this->db->query("
            SELECT COUNT(*) as count 
            FROM customer_feedback 
            WHERE ip_address = ? AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ", [$ipAddress])->getRowArray();
        
        if ($recentSubmissions['count'] >= 50) {
            log_message('warning', 'Too many submissions from IP: ' . $ipAddress);
            return redirect()->back()->with('error', 'Too many submissions. Please try again later.');
        }
        
        try {
            log_message('info', 'About to insert feedback into database');
            
            // Insert feedback
            $result = $this->db->query("
                INSERT INTO customer_feedback 
                (feedback_id, store_id, overall_satisfaction, net_promoter_score, comment, customer_name, customer_contact, ip_address, user_agent)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ", [
                $feedbackId,
                $store['store_id'],
                $satisfaction,
                $nps,
                $comment,
                $customerName,
                $customerContact,
                $ipAddress,
                $userAgent
            ]);

            if ($result) {
                log_message('info', 'Feedback inserted successfully with ID: ' . $feedbackId);
            } else {
                log_message('error', 'Database insert failed for feedback: ' . $feedbackId);
            }

            // Handle different score scenarios
            if ($needsResolution) {
                log_message('info', 'Creating resolution flow for feedback: ' . $feedbackId);
                return $this->createResolutionFlow($feedbackId, $store, $satisfaction, $nps);
            } elseif ($satisfaction >= 7 && $nps >= 7) {
                log_message('info', 'Offering Google review for feedback: ' . $feedbackId);
                return $this->offerGoogleReview($feedbackId, $store);
            } else {
                log_message('info', 'Showing thank you for feedback: ' . $feedbackId);
                return $this->showThankYou($store);
            }

        } catch (\Exception $e) {
            log_message('error', 'Feedback submission error: ' . $e->getMessage());
            log_message('error', 'Exception trace: ' . $e->getTraceAsString());
            return redirect()->back()->withInput()->with('error', 'Unable to submit feedback. Please try again.');
        }
    }

    /**
     * Create resolution flow for low scores with contact info
     */
    private function createResolutionFlow($feedbackId, $store, $satisfaction, $nps)
    {
        try {
            // Generate case ID and number
            $caseId = $this->generateUniqueId('case_');
            $caseNumber = $this->generateCaseNumber();

            // Determine which scores triggered the case
            $triggerScores = [];
            if ($satisfaction <= 6) $triggerScores['satisfaction'] = $satisfaction;
            if ($nps <= 6) $triggerScores['nps'] = $nps;

            // Create case
            $this->db->query("
                INSERT INTO feedback_cases 
                (case_id, feedback_id, store_id, case_number, trigger_scores)
                VALUES (?, ?, ?, ?, ?)
            ", [
                $caseId,
                $feedbackId,
                $store['store_id'],
                $caseNumber,
                json_encode($triggerScores)
            ]);

            // Notify business (implement email service later)
            $this->notifyBusinessOfNewCase($store, $caseNumber);

            $data = [
                'title' => 'We\'d Like to Help - ' . $store['name'],
                'store' => $store,
                'case_number' => $caseNumber,
                'message_type' => 'resolution_offer'
            ];

            return view('feedback/resolution_offer', $data);

        } catch (\Exception $e) {
            log_message('error', 'Case creation error: ' . $e->getMessage());
            return $this->showThankYou($store);
        }
    }

    /**
     * Offer Google review for high scores
     */
    private function offerGoogleReview($feedbackId, $store)
    {
        // Mark that Google redirect was offered
        $this->db->query("
            UPDATE customer_feedback 
            SET google_redirect_offered = 1 
            WHERE feedback_id = ?
        ", [$feedbackId]);

        $data = [
            'title' => 'Thank You! - ' . $store['name'],
            'store' => $store,
            'feedback_id' => $feedbackId,
            'message_type' => 'google_offer'
        ];

        return view('feedback/google_offer', $data);
    }

    /**
     * Show simple thank you for mixed scores
     */
    private function showThankYou($store)
    {
        $data = [
            'title' => 'Thank You - ' . $store['name'],
            'store' => $store,
            'message_type' => 'thank_you'
        ];

        return view('feedback/thank_you', $data);
    }

    /**
     * Handle Google review redirect acceptance
     */
    public function acceptGoogleReview()
    {
        $feedbackId = $this->request->getPost('feedback_id');
        
        if ($feedbackId) {
            // Mark that customer accepted Google redirect
            $this->db->query("
                UPDATE customer_feedback 
                SET google_redirect_accepted = 1 
                WHERE feedback_id = ?
            ", [$feedbackId]);

            // Get store's Google review URL
            $store = $this->db->query("
                SELECT s.google_review_url, s.name
                FROM customer_feedback cf
                JOIN store s ON cf.store_id = s.store_id
                WHERE cf.feedback_id = ?
            ", [$feedbackId])->getRowArray();

            if ($store && $store['google_review_url']) {
                return redirect()->to($store['google_review_url']);
            }
        }

        return redirect()->to('/')->with('error', 'Unable to redirect to Google reviews.');
    }

    /**
     * Generate unique IDs for records
     */
    private function generateUniqueId($prefix = '')
    {
        return $prefix . uniqid() . '_' . mt_rand(1000, 9999);
    }

    /**
     * Generate formatted case number
     */
    private function generateCaseNumber()
    {
        $year = date('Y');
        
        // Get next number for this year
        $result = $this->db->query("
            SELECT COALESCE(MAX(CAST(SUBSTRING(case_number, 9) AS UNSIGNED)), 0) + 1 as next_number
            FROM feedback_cases 
            WHERE case_number LIKE ?
        ", ["FB-{$year}-%"])->getRowArray();

        $nextNumber = $result['next_number'];
        
        return sprintf('FB-%s-%03d', $year, $nextNumber);
    }

    /**
     * Notify business of new case (placeholder for email service)
     */
    private function notifyBusinessOfNewCase($store, $caseNumber)
    {
        // TODO: Implement email notification to business
        log_message('info', "New feedback case {$caseNumber} created for store {$store['store_id']}");
    }
}