<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Models\BusinessChangeRequestModel;
use CodeIgniter\HTTP\ResponseInterface;

class Profile extends BaseController
{
    protected $businessUserModel;
    protected $storeModel;
    protected $changeRequestModel;
    protected $session;

    // Fields that businesses can edit
    protected $editableFields = [
        'name', 'description', 'email', 'phone', 'website',
        'address', 'city', 'state', 'postal_code',
        'facebook', 'instagram', 'x_twitter', 'linkedin', 
        'youtube', 'tiktok', 'snapchat', 'google_business', 
        'yelp', 'nextdoor', 'additional_info'
    ];

    public function __construct()
    {
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->changeRequestModel = new BusinessChangeRequestModel();
        $this->session = \Config\Services::session();
    }

    /**
     * Display business profile edit form
     */
    public function edit()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');

        // Get current store data
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Business not found.');
        }

        // Check for pending requests
        $hasPending = $this->changeRequestModel->hasPendingRequests($storeId);

        // Get recent change requests
        $recentRequests = $this->changeRequestModel->getRequestsByStore($storeId, 3);

        $data = [
            'title' => 'Edit Business Profile',
            'store' => $store,
            'editable_fields' => $this->editableFields,
            'has_pending' => $hasPending,
            'recent_requests' => $recentRequests,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error'),
            'validation' => session()->getFlashdata('validation')
        ];

        return view('business/profile/edit', $data);
    }

    /**
     * Process profile update request
     */
    public function update()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');

        // Validation rules for business profile fields
        $rules = [
            'name' => 'required|min_length[2]|max_length[255]',
            'description' => 'permit_empty|max_length[1000]',
            'email' => 'permit_empty|valid_email|max_length[255]',
            'phone' => 'permit_empty|min_length[10]|max_length[20]',
            'website' => 'permit_empty|valid_url|max_length[255]',
            'address' => 'permit_empty|max_length[255]',
            'city' => 'permit_empty|max_length[100]',
            'state' => 'permit_empty|max_length[2]',
            'postal_code' => 'permit_empty|max_length[10]',
            'facebook' => 'permit_empty|max_length[255]',
            'instagram' => 'permit_empty|max_length[255]',
            'x_twitter' => 'permit_empty|max_length[255]',
            'linkedin' => 'permit_empty|max_length[255]',
            'youtube' => 'permit_empty|max_length[255]',
            'tiktok' => 'permit_empty|max_length[255]',
            'snapchat' => 'permit_empty|max_length[255]',
            'google_business' => 'permit_empty|max_length[255]',
            'yelp' => 'permit_empty|max_length[255]',
            'nextdoor' => 'permit_empty|max_length[255]',
            'additional_info' => 'permit_empty|max_length[500]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator->getErrors());
        }

        // Get current store data
        $currentStore = $this->storeModel->find($storeId);
        
        if (!$currentStore) {
            return redirect()->to('/business/dashboard')->with('error', 'Business not found.');
        }

        // Collect changes from form
        $changes = [];
        $hasChanges = false;

        foreach ($this->editableFields as $field) {
            $newValue = $this->request->getPost($field) ?? '';
            $currentValue = $currentStore[$field] ?? '';

            // Normalize values for comparison
            $newValue = trim($newValue);
            $currentValue = trim($currentValue);

            if ($newValue !== $currentValue) {
                $changes[$field] = [
                    'old' => $currentValue,
                    'new' => $newValue
                ];
                $hasChanges = true;
            }
        }

        if (!$hasChanges) {
            return redirect()->back()->with('error', 'No changes detected. Please modify at least one field to submit a request.');
        }

        // Determine request type
        $requestType = 'profile_update';
        if (count($changes) > 3) {
            $requestType = 'multiple_fields';
        }

        // Create change request
        $requestData = [
            'store_id' => $storeId,
            'business_user_id' => $userId,
            'request_type' => $requestType,
            'field_changes' => $changes,
            'status' => 'pending'
        ];

        if ($this->changeRequestModel->createRequest($storeId, $userId, $requestType, $changes)) {
            // TODO: Send notification email to admin
            $this->sendAdminNotification($storeId, $changes);
            
            return redirect()->to('/business/dashboard/change-requests')->with('success', 'Change request submitted successfully! An administrator will review your changes shortly.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to submit change request. Please try again.');
        }
    }

    /**
     * Preview changes before submitting
     */
    public function preview()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');

        // Get current store data
        $currentStore = $this->storeModel->find($storeId);
        
        if (!$currentStore) {
            return redirect()->to('/business/dashboard')->with('error', 'Business not found.');
        }

        // Get form data
        $formData = $this->request->getPost();
        
        // Compare changes
        $changes = [];
        foreach ($this->editableFields as $field) {
            $newValue = $formData[$field] ?? '';
            $currentValue = $currentStore[$field] ?? '';

            // Normalize values for comparison
            $newValue = trim($newValue);
            $currentValue = trim($currentValue);

            if ($newValue !== $currentValue) {
                $changes[$field] = [
                    'old' => $currentValue,
                    'new' => $newValue,
                    'field_label' => $this->getFieldLabel($field)
                ];
            }
        }

        if (empty($changes)) {
            return redirect()->back()->withInput()->with('error', 'No changes detected.');
        }

        $data = [
            'title' => 'Preview Changes',
            'changes' => $changes,
            'store' => $currentStore,
            'form_data' => $formData
        ];

        return view('business/profile/preview', $data);
    }

    /**
     * Business hours management
     */
    public function hours()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');

        // Get store information
        $store = $this->storeModel->getStoreById($storeId);
        
        // Get current hours
        $hours = $this->storeModel->getStoreHours($storeId);

        $data = [
            'title' => 'Business Hours',
            'store' => $store,
            'hours' => $hours,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/profile/hours', $data);
    }

    /**
     * Update business hours
     */
    public function updateHours()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');

        // Validate hours data
        $hoursData = $this->request->getPost('hours');
        
        if (!$hoursData || !is_array($hoursData)) {
            return redirect()->back()->with('error', 'Invalid hours data provided.');
        }

        // Get current hours for comparison
        $currentHours = $this->storeModel->getStoreHours($storeId);
        
        // Process hours changes
        $changes = $this->processHoursChanges($currentHours, $hoursData);

        if (empty($changes)) {
            return redirect()->back()->with('error', 'No changes detected in business hours.');
        }

        // Create change request for hours
        if ($this->changeRequestModel->createRequest($storeId, $userId, 'hours_update', $changes)) {
            $this->sendAdminNotification($storeId, $changes, 'hours');
            
            return redirect()->to('/business/dashboard/change-requests')->with('success', 'Hours change request submitted successfully!');
        } else {
            return redirect()->back()->with('error', 'Failed to submit hours change request.');
        }
    }

    /**
     * Process hours changes for comparison
     */
    private function processHoursChanges($currentHours, $newHours)
    {
        $changes = [];
        $daysOfWeek = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

        foreach ($daysOfWeek as $day) {
            $currentDay = $currentHours[$day] ?? null;
            $newDay = $newHours[$day] ?? null;

            // Compare open/close times and closed status
            if ($this->hoursChanged($currentDay, $newDay)) {
                $changes[$day] = [
                    'old' => $currentDay,
                    'new' => $newDay
                ];
            }
        }

        return $changes;
    }

    /**
     * Check if hours changed for a specific day
     */
    private function hoursChanged($current, $new)
    {
        if (!$current && !$new) return false;
        if (!$current || !$new) return true;

        return ($current['open_time'] !== $new['open_time'] ||
                $current['close_time'] !== $new['close_time'] ||
                $current['is_closed'] !== $new['is_closed']);
    }

    /**
     * Get field label for display
     */
    private function getFieldLabel($field)
    {
        $labels = [
            'name' => 'Business Name',
            'description' => 'Description',
            'email' => 'Email Address',
            'phone' => 'Phone Number',
            'website' => 'Website',
            'address' => 'Street Address',
            'city' => 'City',
            'state' => 'State',
            'postal_code' => 'Zip Code',
            'facebook' => 'Facebook',
            'instagram' => 'Instagram',
            'x_twitter' => 'X (Twitter)',
            'linkedin' => 'LinkedIn',
            'youtube' => 'YouTube',
            'tiktok' => 'TikTok',
            'snapchat' => 'Snapchat',
            'google_business' => 'Google Business',
            'yelp' => 'Yelp',
            'nextdoor' => 'Nextdoor',
            'additional_info' => 'Additional Info (Vimeo ID)'
        ];

        return $labels[$field] ?? ucfirst(str_replace('_', ' ', $field));
    }

    /**
     * Send notification to admin about new change request
     */
    private function sendAdminNotification($storeId, $changes, $type = 'profile')
    {
        // TODO: Implement email notification to admin
        // This would typically use CodeIgniter's Email class
        // For now, we'll just log it or set a flag
        
        log_message('info', "New {$type} change request for store {$storeId}: " . json_encode($changes));
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && !empty($this->session->get('business_user_id'));
    }
}