<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Models\BusinessCouponModel;
use App\Models\BusinessJobModel;

class Onboarding extends BaseController
{
    protected $businessUserModel;
    protected $storeModel;
    protected $couponModel;
    protected $jobModel;
    protected $session;

    public function __construct()
    {
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->couponModel = new BusinessCouponModel();
        $this->jobModel = new BusinessJobModel();
        $this->session = \Config\Services::session();
    }

    /**
     * Main onboarding welcome page
     */
    public function index()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);

        $data = [
            'title' => 'Fast, Guided Setup - Welcome',
            'store' => $store,
            'current_step' => 0,
            'total_steps' => 4
        ];

        return view('business/onboarding/welcome', $data);
    }

    /**
     * Step 1: Basic Profile Information
     */
    public function step1()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            return $this->saveStep1();
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);

        // Get saved session data from previous attempts
        $sessionData = $this->session->get('onboarding_step1') ?? [];

        $data = [
            'title' => 'Step 1: Tell Us About Your Business',
            'store' => $store,
            'saved_data' => $sessionData,
            'current_step' => 1,
            'total_steps' => 4,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error'),
            'validation' => session()->getFlashdata('validation')
        ];

        return view('business/onboarding/step1', $data);
    }

    /**
     * Save Step 1 data
     */
    private function saveStep1()
    {
        // Validation rules for basic profile
        $rules = [
            'name' => 'required|min_length[2]|max_length[255]',
            'description' => 'permit_empty|max_length[1000]',
            'email' => 'permit_empty|valid_email|max_length[255]',
            'phone' => 'permit_empty|min_length[10]|max_length[20]',
            'website' => 'permit_empty|valid_url|max_length[255]',
            'address' => 'permit_empty|max_length[255]',
            'city' => 'permit_empty|max_length[100]',
            'state' => 'permit_empty|max_length[2]',
            'postal_code' => 'permit_empty|max_length[10]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator->getErrors());
        }

        $storeId = $this->session->get('business_store_id');
        
        // Prepare data for direct database update
        $updateData = [
            'name' => trim($this->request->getPost('name')),
            'description' => trim($this->request->getPost('description')),
            'email' => trim($this->request->getPost('email')),
            'phone' => trim($this->request->getPost('phone')),
            'website' => trim($this->request->getPost('website')),
            'address' => trim($this->request->getPost('address')),
            'city' => trim($this->request->getPost('city')),
            'state' => strtoupper(trim($this->request->getPost('state'))),
            'postal_code' => trim($this->request->getPost('postal_code'))
        ];

        // Remove empty values
        $updateData = array_filter($updateData, function($value) {
            return $value !== '';
        });

        // Direct database update (no approval process)
        if ($this->storeModel->update($storeId, $updateData)) {
            // Save to session for progress tracking
            $this->session->set('onboarding_step1_completed', true);
            
            // Send admin notification
            $this->notifyAdminProgress($storeId, 'step1', $updateData);
            
            return redirect()->to('/business/onboarding/step2')->with('success', 'Basic information saved! Let\'s set up your hours next.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to save information. Please try again.');
        }
    }

    /**
     * Step 2: Business Hours
     */
    public function step2()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            return $this->saveStep2();
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);

        // Get current hours if they exist
        $hours = $this->storeModel->getStoreHours($storeId);

        $data = [
            'title' => 'Step 2: When Are You Open?',
            'store' => $store,
            'hours' => $hours,
            'current_step' => 2,
            'total_steps' => 4,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/onboarding/step2', $data);
    }

    /**
     * Save Step 2 data (business hours) - integrates with real hours system
     */
    private function saveStep2()
    {
        $storeId = $this->session->get('business_store_id');
        $hoursType = $this->request->getPost('hours_type');

        // Handle different hour setup options
        switch ($hoursType) {
            case 'skip':
                // User chose to skip hours setup
                $this->session->set('onboarding_step2_completed', true);
                $this->notifyAdminProgress($storeId, 'step2_skipped', null);
                return redirect()->to('/business/onboarding/step3')->with('success', 'Hours setup skipped. You can set detailed hours later from your dashboard.');
                
            case 'complex':
                // User has complex scheduling needs - redirect to full hours manager after onboarding
                $this->session->set('onboarding_step2_completed', true);
                $this->session->set('onboarding_needs_complex_hours', true);
                $this->notifyAdminProgress($storeId, 'step2_complex', null);
                return redirect()->to('/business/onboarding/step3')->with('success', 'We\'ll set up your detailed hours system after the basic setup is complete.');
                
            case 'simple':
                // Process simple hours data
                return $this->saveSimpleHours($storeId);
                
            default:
                return redirect()->back()->with('error', 'Please select a hours setup option.');
        }
    }

    /**
     * Save simple hours to the database using proper hours system
     */
    private function saveSimpleHours($storeId)
    {
        $hoursData = $this->request->getPost('hours');
        $userId = $this->session->get('business_user_id');

        if (!$hoursData || !is_array($hoursData)) {
            return redirect()->to('/business/onboarding/step3')->with('success', 'Hours setup skipped. You can set hours later from your dashboard.');
        }

        $db = \Config\Database::connect();
        
        try {
            $db->transStart();

            // Check if store already has a "Regular Hours" hour type
            $existingHourType = $db->query("
                SELECT id FROM store_hour_types 
                WHERE store_id = ? AND custom_name = 'Regular Hours' AND is_active = 1
            ", [$storeId])->getRowArray();

            if (!$existingHourType) {
                // Create "Regular Hours" hour type
                $db->query("
                    INSERT INTO store_hour_types (store_id, global_type_id, custom_name, display_order, affects_main_status, is_active) 
                    VALUES (?, 1, 'Regular Hours', 1, 1, 1)
                ", [$storeId]);
                
                $hourTypeId = $db->insertID();
            } else {
                $hourTypeId = $existingHourType['id'];
                
                // Clear existing schedules for this hour type
                $db->query("
                    DELETE FROM store_schedules 
                    WHERE store_id = ? AND hour_type_id = ?
                ", [$storeId, $hourTypeId]);
            }

            // Process each day's hours
            $hasValidHours = false;
            foreach ($hoursData as $dayNum => $dayData) {
                if (!is_numeric($dayNum) || $dayNum < 1 || $dayNum > 7) continue;

                $isClosed = isset($dayData['closed']) ? 1 : 0;
                $openTime = null;
                $closeTime = null;

                if (!$isClosed && !empty($dayData['open']) && !empty($dayData['close'])) {
                    $openTime = $dayData['open'] . ':00'; // Convert to HH:MM:SS format
                    $closeTime = $dayData['close'] . ':00';
                    $hasValidHours = true;
                }

                // Insert schedule for this day
                $db->query("
                    INSERT INTO store_schedules 
                    (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, slot_2_open, slot_2_close, is_closed) 
                    VALUES (?, ?, ?, ?, ?, NULL, NULL, ?)
                ", [$storeId, $hourTypeId, $dayNum, $openTime, $closeTime, $isClosed]);
            }

            $db->transComplete();

            if ($db->transStatus() === false) {
                throw new \Exception('Database transaction failed');
            }

            // Mark step as completed
            $this->session->set('onboarding_step2_completed', true);
            
            // Notify admin with summary
            $hoursData['hour_type_created'] = !$existingHourType;
            $hoursData['has_valid_hours'] = $hasValidHours;
            $this->notifyAdminProgress($storeId, 'step2_simple', $hoursData);

            if ($hasValidHours) {
                return redirect()->to('/business/onboarding/step3')->with('success', 'Business hours saved! Now let\'s create a special offer to attract customers.');
            } else {
                return redirect()->to('/business/onboarding/step3')->with('success', 'Hours setup completed. You can add specific times later from your dashboard.');
            }

        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Onboarding Step 2 save failed: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to save hours. Please try again or skip this step for now.');
        }
    }

    /**
     * Step 3: Optional Coupon Creation
     */
    public function step3()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            return $this->saveStep3();
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);

        $data = [
            'title' => 'Step 3: Create a Special Offer (Optional)',
            'store' => $store,
            'current_step' => 3,
            'total_steps' => 4,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/onboarding/step3', $data);
    }

    /**
     * Save Step 3 data (optional coupon)
     */
    private function saveStep3()
    {
        $action = $this->request->getPost('action');
        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');

        if ($action === 'skip') {
            $this->session->set('onboarding_step3_completed', true);
            return redirect()->to('/business/onboarding/step4')->with('success', 'No worries! You can create offers anytime later.');
        }

        // Validate coupon data
        $rules = [
            'title' => 'required|min_length[3]|max_length[255]',
            'description' => 'required|min_length[10]',
            'discount_type' => 'required|in_list[percentage,fixed_amount]',
            'discount_value' => 'required|numeric|greater_than[0]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator->getErrors());
        }

        // Create coupon directly
        $couponData = [
            'store_id' => $storeId,
            'title' => trim($this->request->getPost('title')),
            'description' => trim($this->request->getPost('description')),
            'discount_type' => $this->request->getPost('discount_type'),
            'discount_value' => (float)$this->request->getPost('discount_value'),
            'starts_at' => date('Y-m-d'),
            'expires_at' => date('Y-m-d', strtotime('+30 days')),
            'is_active' => 1,
            'created_by' => $userId,
            'usage_count' => 0
        ];

        if ($this->couponModel->insert($couponData)) {
            $this->session->set('onboarding_step3_completed', true);
            $this->notifyAdminProgress($storeId, 'step3', $couponData);
            return redirect()->to('/business/onboarding/step4')->with('success', 'Special offer created! One more optional step.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to create offer. Please try again.');
        }
    }

    /**
     * Step 4: Optional Job Posting
     */
    public function step4()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'POST') {
            return $this->saveStep4();
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);

        $data = [
            'title' => 'Step 4: Are You Hiring? (Optional)',
            'store' => $store,
            'current_step' => 4,
            'total_steps' => 4,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/onboarding/step4', $data);
    }

    /**
     * Save Step 4 data (optional job posting)
     */
    private function saveStep4()
    {
        $action = $this->request->getPost('action');
        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');

        if ($action === 'skip') {
            return $this->completeOnboarding();
        }

        // Validate job data
        $rules = [
            'title' => 'required|min_length[3]|max_length[255]',
            'description' => 'required|min_length[10]',
            'employment_type' => 'required|in_list[full_time,part_time]',
            'salary_min' => 'required|numeric|greater_than[0]',
            'contact_method' => 'required|in_list[phone,email,walk_in]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('validation', $this->validator->getErrors());
        }

        // Create job posting directly
        $jobData = [
            'store_id' => $storeId,
            'title' => trim($this->request->getPost('title')),
            'description' => trim($this->request->getPost('description')),
            'employment_type' => $this->request->getPost('employment_type'),
            'salary_min' => (float)$this->request->getPost('salary_min'),
            'salary_type' => 'hourly',
            'contact_method' => $this->request->getPost('contact_method'),
            'is_active' => 1,
            'expires_at' => date('Y-m-d H:i:s', strtotime('+30 days')),
            'created_by' => $userId,
            'times_renewed' => 0,
            'renewal_email_sent' => 0
        ];

        if ($this->jobModel->insert($jobData)) {
            $this->notifyAdminProgress($storeId, 'step4', $jobData);
            return $this->completeOnboarding();
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to create job posting. Please try again.');
        }
    }

    /**
     * Complete onboarding and redirect to dashboard
     */
    private function completeOnboarding()
    {
        $storeId = $this->session->get('business_store_id');
        
        // Mark onboarding as completed
        $this->session->set('onboarding_completed', true);
        
        // Check if user needs complex hours setup
        if ($this->session->get('onboarding_needs_complex_hours')) {
            $this->session->remove('onboarding_needs_complex_hours');
            // Send final admin notification
            $this->notifyAdminProgress($storeId, 'completed_needs_complex_hours', null);
            
            // Clear onboarding session data
            $this->session->remove(['onboarding_step1', 'onboarding_step2', 'onboarding_step3', 'onboarding_step4']);
            
            return redirect()->to('/admin/hours/store/' . $storeId)->with('success', 'Basic setup complete! Now let\'s configure your detailed hours system.');
        }
        
        // Send final admin notification
        $this->notifyAdminProgress($storeId, 'completed', null);
        
        // Clear onboarding session data
        $this->session->remove(['onboarding_step1', 'onboarding_step2', 'onboarding_step3', 'onboarding_step4']);
        
        return redirect()->to('/business/dashboard')->with('success', 'Setup complete! Your business profile is now live and ready for customers.');
    }

    /**
     * Send admin notification about onboarding progress
     */
    private function notifyAdminProgress($storeId, $step, $data)
    {
        // Get store and user info
        $store = $this->storeModel->getStoreById($storeId);
        $userId = $this->session->get('business_user_id');
        $user = $this->businessUserModel->find($userId);
        
        // Log the progress
        log_message('info', "Onboarding progress - Store: {$storeId}, Step: {$step}, User: {$userId}");
        
        // TODO: Send email notification to admin
        // This would use your existing email service
        // For now, we're just logging the activity
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && 
               !empty($this->session->get('business_user_id'));
    }
}