<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;

class Tools extends BaseController
{
    protected $db;
    protected $businessUserModel;
    protected $storeModel;
    
    public function __construct()
    {
        $this->db = \Config\Database::connect();
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        
        // Check admin authentication (matching your pattern)
        if (!session()->get('admin_logged_in')) {
            header('Location: ' . base_url('admin/login'));
            exit();
        }
    }
    
    /**
     * Display tools menu
     */
    public function index()
    {
        $data = [
            'title' => 'Admin Tools',
            'heading' => 'System Administration Tools'
        ];
        
        // Check if business users already exist
        $builder = $this->db->table('business_users');
        $userCount = $builder->countAllResults();
        
        $data['businessUserCount'] = $userCount;
        $data['totalStores'] = $this->db->table('store')->countAllResults();
        
        // Get pending change requests count
        $data['pendingRequests'] = $this->db->table('business_change_requests')
                                           ->where('status', 'pending')
                                           ->countAllResults();
        
        return view('admin/tools/index', $data);
    }
    
    /**
     * Generate business users - Step 1: Confirm
     */
    public function generateBusinessUsers()
    {
        // Check if temp_password column exists
        $fields = $this->db->getFieldData('business_users');
        $hasTempPassword = false;
        foreach ($fields as $field) {
            if ($field->name === 'temp_password') {
                $hasTempPassword = true;
                break;
            }
        }
        
        // Get stores without users
        $sql = "SELECT s.* FROM store s 
                LEFT JOIN business_users bu ON s.store_id = bu.store_id 
                WHERE bu.id IS NULL 
                ORDER BY s.name";
        $storesWithoutUsers = $this->db->query($sql)->getResultArray();
        
        $data = [
            'title' => 'Generate Business Users',
            'heading' => 'Bulk Create Business User Accounts',
            'hasTempPassword' => $hasTempPassword,
            'storesWithoutUsers' => $storesWithoutUsers,
            'totalToCreate' => count($storesWithoutUsers)
        ];
        
        return view('admin/tools/generate_users', $data);
    }
    
    /**
     * Process business user generation
     */
    public function processGenerateUsers()
    {
        set_time_limit(300); // 5 minutes max
        
        // First, ensure temp_password column exists
        if (!$this->ensureTempPasswordColumn()) {
            return redirect()->back()->with('error', 'Failed to add temp_password column to database.');
        }
        
        // Get all stores without users
        $sql = "SELECT s.* FROM store s 
                LEFT JOIN business_users bu ON s.store_id = bu.store_id 
                WHERE bu.id IS NULL 
                ORDER BY s.name";
        $stores = $this->db->query($sql)->getResultArray();
        
        if (empty($stores)) {
            return redirect()->back()->with('info', 'All stores already have user accounts.');
        }
        
        $results = [];
        $created = 0;
        $errors = 0;
        
        foreach ($stores as $store) {
            // Generate unique username
            $username = $this->generateUniqueUsername();
            
            // Generate password (1 capital letter + 5 digits)
            $tempPassword = $this->generatePassword();
            
            // Prepare user data - ensure unique email for UNIQUE constraint
            $userData = [
                'store_id' => $store['store_id'],
                'username' => $username,
                'email' => 'email_' . $username . '@not.published', // Always unique
                'password' => password_hash($tempPassword, PASSWORD_DEFAULT),
                'temp_password' => $this->encryptPassword($tempPassword),
                'role' => 'owner',
                'status' => 1,
                'email_verified' => 0
            ];
            
            // Use real email if available and valid
            if (!empty($store['email']) && filter_var($store['email'], FILTER_VALIDATE_EMAIL)) {
                // Check if this email is already used
                $emailExists = $this->db->table('business_users')
                                       ->where('email', $store['email'])
                                       ->countAllResults() > 0;
                
                if (!$emailExists) {
                    $userData['email'] = $store['email'];
                }
                // If email exists, keep the unique placeholder
            }
            
            try {
                // Insert user directly to bypass model validation
                $this->db->table('business_users')->insert($userData);
                
                $results[] = [
                    'success' => true,
                    'store_name' => $store['name'],
                    'store_id' => $store['store_id'],
                    'username' => $username,
                    'password' => $tempPassword,
                    'email' => $userData['email'],
                    'address' => $store['address'],
                    'city' => $store['city'],
                    'phone' => $store['phone']
                ];
                $created++;
                
            } catch (\Exception $e) {
                $results[] = [
                    'success' => false,
                    'store_name' => $store['name'],
                    'store_id' => $store['store_id'],
                    'error' => $e->getMessage()
                ];
                $errors++;
            }
        }
        
        // Store results in session for download
        session()->set('bulk_user_results', $results);
        
        $data = [
            'title' => 'Business Users Created',
            'heading' => 'Bulk User Creation Results',
            'results' => $results,
            'created' => $created,
            'errors' => $errors,
            'total' => count($stores)
        ];
        
        return view('admin/tools/generate_users_results', $data);
    }
    
    /**
     * Download credentials as CSV
     */
    public function downloadCredentials()
    {
        $results = session()->get('bulk_user_results');
        
        if (empty($results)) {
            return redirect()->to('/admin/tools')->with('error', 'No credential data available for download.');
        }
        
        // Filter only successful results
        $credentials = array_filter($results, function($r) { return $r['success']; });
        
        // Set headers for CSV download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="business_credentials_' . date('Y-m-d_His') . '.csv"');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        // Open output stream
        $output = fopen('php://output', 'w');
        
        // Write CSV headers
        fputcsv($output, [
            'Business Name',
            'Address',
            'City',
            'Phone',
            'Username',
            'Temporary Password',
            'Email in System',
            'Store ID'
        ]);
        
        // Write data rows
        foreach ($credentials as $cred) {
            fputcsv($output, [
                $cred['store_name'],
                $cred['address'],
                $cred['city'],
                $cred['phone'],
                $cred['username'],
                $cred['password'],
                $cred['email'],
                $cred['store_id']
            ]);
        }
        
        fclose($output);
        exit();
    }
    
    /**
     * View/manage temporary passwords
     */
    public function managePasswords()
    {
        $search = $this->request->getGet('search');
        
        $builder = $this->db->table('business_users bu');
        $builder->select('bu.*, s.name as store_name, s.address, s.phone');
        $builder->join('store s', 's.store_id = bu.store_id', 'left');
        
        if ($search) {
            $builder->groupStart()
                    ->like('bu.username', $search)
                    ->orLike('s.name', $search)
                    ->orLike('bu.email', $search)
                    ->groupEnd();
        }
        
        $builder->orderBy('s.name', 'ASC');
        $users = $builder->get()->getResultArray();
        
        // Decrypt temporary passwords for display
        foreach ($users as &$user) {
            if (!empty($user['temp_password'])) {
                $user['temp_password_decrypted'] = $this->decryptPassword($user['temp_password']);
            } else {
                $user['temp_password_decrypted'] = null;
            }
        }
        
        $data = [
            'title' => 'Manage Business Passwords',
            'heading' => 'Business User Password Management',
            'users' => $users,
            'search' => $search
        ];
        
        return view('admin/tools/manage_passwords', $data);
    }
    
    /**
     * Reset a user's password
     */
    public function resetPassword($userId)
    {
        $user = $this->businessUserModel->find($userId);
        
        if (!$user) {
            return redirect()->back()->with('error', 'User not found.');
        }
        
        // Generate new password
        $newPassword = $this->generatePassword();
        
        // Update user
        $updateData = [
            'password' => password_hash($newPassword, PASSWORD_DEFAULT),
            'temp_password' => $this->encryptPassword($newPassword)
        ];
        
        if ($this->businessUserModel->update($userId, $updateData)) {
            // Get store info for display
            $store = $this->storeModel->where('store_id', $user['store_id'])->first();
            
            session()->setFlashdata('success', 
                "Password reset successfully for {$store['name']}. New password: <strong>{$newPassword}</strong>");
        } else {
            session()->setFlashdata('error', 'Failed to reset password.');
        }
        
        return redirect()->to('/admin/tools/manage-passwords');
    }
    
    /**
     * Helper: Ensure temp_password column exists
     */
    private function ensureTempPasswordColumn()
    {
        try {
            $sql = "ALTER TABLE business_users ADD COLUMN temp_password VARCHAR(100) DEFAULT NULL AFTER password";
            $this->db->query($sql);
            return true;
        } catch (\Exception $e) {
            // Column might already exist
            if (strpos($e->getMessage(), 'Duplicate column name') !== false) {
                return true;
            }
            return false;
        }
    }
    
    /**
     * Helper: Generate unique 6-digit username
     */
    private function generateUniqueUsername()
    {
        $attempts = 0;
        
        do {
            $username = str_pad(random_int(100000, 999999), 6, '0', STR_PAD_LEFT);
            
            // Check if exists
            $exists = $this->db->table('business_users')
                              ->where('username', $username)
                              ->countAllResults() > 0;
            
            $attempts++;
            
            // Fallback after too many attempts
            if ($attempts > 1000) {
                $username = 'BU' . str_pad($this->db->table('business_users')->countAllResults() + 1, 4, '0', STR_PAD_LEFT);
                break;
            }
            
        } while ($exists);
        
        return $username;
    }
    
    /**
     * Helper: Generate password (1 capital letter + 5 digits)
     */
    private function generatePassword()
    {
        $letter = chr(random_int(65, 90)); // A-Z
        $numbers = str_pad(random_int(0, 99999), 5, '0', STR_PAD_LEFT);
        return $letter . $numbers;
    }
    
    /**
     * Helper: Encrypt password for storage
     */
    private function encryptPassword($password)
    {
        $key = getenv('encryption.key') ?: 'athens_mcminn_2024_secret_key';
        return base64_encode(openssl_encrypt($password, 'AES-256-CBC', $key, 0, substr($key, 0, 16)));
    }
    
    /**
     * Helper: Decrypt password for viewing
     */
    private function decryptPassword($encrypted)
    {
        if (empty($encrypted)) return null;
        
        $key = getenv('encryption.key') ?: 'athens_mcminn_2024_secret_key';
        return openssl_decrypt(base64_decode($encrypted), 'AES-256-CBC', $key, 0, substr($key, 0, 16));
    }
}