<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;

class HoursImport extends BaseController
{
    protected $db;
    
    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    /**
     * Main hours import dashboard
     */
    public function index()
    {
        // Check authentication
        if (!session()->get('admin_logged_in')) {
            return redirect()->to('/admin/login');
        }

        try {
            if (!$this->db) {
                throw new \Exception('Database connection failed');
            }

            log_message('info', 'Hours import index: Starting data analysis');

            $businessesQuery = "
                SELECT s.store_id, s.name, s.address, s.city, s.state, s.postal_code, 
                       s.latitude, s.longitude, s.phone, s.website,
                       COUNT(sht.id) as hour_types_count,
                       COUNT(CASE WHEN sht.affects_main_status = 1 THEN 1 END) as main_hour_types_count,
                       GROUP_CONCAT(DISTINCT sht.custom_name ORDER BY sht.display_order SEPARATOR ', ') as hour_types_list,
                       MAX(sht.updated_at) as last_hours_update
                FROM store s
                LEFT JOIN store_hour_types sht ON s.store_id = sht.store_id AND sht.is_active = 1
                WHERE s.status = 1
                GROUP BY s.store_id, s.name, s.address, s.city, s.state, s.postal_code, 
                         s.latitude, s.longitude, s.phone, s.website
                ORDER BY hour_types_count ASC, s.name ASC
                LIMIT 500
            ";
            
            $businesses = $this->db->query($businessesQuery)->getResultArray();
            
            if ($businesses === false) {
                throw new \Exception('Business query failed');
            }

            log_message('info', 'Hours import index: Retrieved ' . count($businesses) . ' businesses');
            
            $needsHours = [];
            $hasTestData = [];
            $hasPartialHours = [];
            $hasCompleteHours = [];
            
            foreach ($businesses as $business) {
                try {
                    $hasTestDataPattern = $this->detectTestDataPattern($business['store_id']);
                    
                    if ($business['hour_types_count'] == 0) {
                        $needsHours[] = $business;
                    } elseif ($hasTestDataPattern) {
                        $business['test_data_detected'] = true;
                        $hasTestData[] = $business;
                    } elseif ($business['main_hour_types_count'] == 0) {
                        $hasPartialHours[] = $business;
                    } else {
                        $hasCompleteHours[] = $business;
                    }
                } catch (\Exception $e) {
                    log_message('error', 'Error processing business ' . $business['store_id'] . ': ' . $e->getMessage());
                    $needsHours[] = $business;
                }
            }
            
            $data = [
                'needsHours' => $needsHours,
                'hasTestData' => $hasTestData,
                'hasPartialHours' => $hasPartialHours,
                'hasCompleteHours' => $hasCompleteHours,
                'totalBusinesses' => count($businesses),
                'googleApiKey' => env('GOOGLE_PLACES_API_KEY', ''),
                'title' => 'Hours Import Tool'
            ];
            
            log_message('info', 'Hours import index: Data prepared - Needs: ' . count($needsHours) . 
                ', Test Data: ' . count($hasTestData) . 
                ', Partial: ' . count($hasPartialHours) . 
                ', Complete: ' . count($hasCompleteHours));
            
            return view('admin/hours_import/index', $data);
            
        } catch (\Exception $e) {
            log_message('error', 'Hours import index error: ' . $e->getMessage() . ' - Line: ' . $e->getLine() . ' - File: ' . $e->getFile());
            
            $errorData = [
                'title' => 'Hours Import Tool - Error',
                'error' => $e->getMessage(),
                'needsHours' => [],
                'hasTestData' => [],
                'hasPartialHours' => [],
                'hasCompleteHours' => [],
                'totalBusinesses' => 0,
                'googleApiKey' => ''
            ];
            
            return view('admin/hours_import/index', $errorData);
        }
    }

    /**
     * Get categorized business data for AJAX requests
     */
    public function getBusinessData()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        try {
            $businessesQuery = "
                SELECT s.store_id, s.name, s.address, s.city, s.state, s.postal_code, 
                       s.latitude, s.longitude, s.phone, s.website,
                       COUNT(sht.id) as hour_types_count,
                       COUNT(CASE WHEN sht.affects_main_status = 1 THEN 1 END) as main_hour_types_count,
                       GROUP_CONCAT(DISTINCT sht.custom_name ORDER BY sht.display_order SEPARATOR ', ') as hour_types_list,
                       MAX(sht.updated_at) as last_hours_update
                FROM store s
                LEFT JOIN store_hour_types sht ON s.store_id = sht.store_id AND sht.is_active = 1
                WHERE s.status = 1
                GROUP BY s.store_id, s.name, s.address, s.city, s.state, s.postal_code, 
                         s.latitude, s.longitude, s.phone, s.website
                ORDER BY hour_types_count ASC, s.name ASC
            ";
            
            $businesses = $this->db->query($businessesQuery)->getResultArray();
            
            $needsHours = [];
            $hasTestData = [];
            $hasPartialHours = [];
            $hasCompleteHours = [];
            
            foreach ($businesses as $business) {
                $hasTestDataPattern = $this->detectTestDataPattern($business['store_id']);
                
                if ($business['hour_types_count'] == 0) {
                    $needsHours[] = $business;
                } elseif ($hasTestDataPattern) {
                    $business['test_data_detected'] = true;
                    $hasTestData[] = $business;
                } elseif ($business['main_hour_types_count'] == 0) {
                    $hasPartialHours[] = $business;
                } else {
                    $hasCompleteHours[] = $business;
                }
            }
            
            return $this->response->setJSON([
                'success' => true,
                'data' => [
                    'needsHours' => $needsHours,
                    'hasTestData' => $hasTestData,
                    'hasPartialHours' => $hasPartialHours,
                    'hasCompleteHours' => $hasCompleteHours,
                    'totalBusinesses' => count($businesses)
                ]
            ]);
            
        } catch (\Exception $e) {
            log_message('error', 'Error getting business data: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Search hours for individual business
     */
    public function searchHours()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        try {
            $storeId = $this->request->getPost('store_id');
            $businessName = trim($this->request->getPost('business_name'));
            $address = trim($this->request->getPost('address'));
            $city = trim($this->request->getPost('city'));
            $state = trim($this->request->getPost('state'));
            
            log_message('info', "=== SEARCHING HOURS FOR: $businessName ===");
            
            if (!$storeId || !$businessName) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Missing required business information'
                ]);
            }
            
            // Use corrected search
            $placesResults = $this->searchGooglePlaces($businessName, $address, $city, $state);
            
            if (empty($placesResults)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => "No results found on Google Places for '$businessName'. This business may not be listed on Google Maps or may have a different name."
                ]);
            }
            
            // Get detailed information for top matches
            $detailedResults = [];
            foreach (array_slice($placesResults, 0, 3) as $place) {
                $details = $this->getPlaceDetails($place['place_id']);
                if ($details) {
                    $processedDetails = $this->processPlaceHours($details);
                    $processedDetails['confidence_score'] = $this->calculateMatchConfidence($details, $businessName, $address);
                    $detailedResults[] = $processedDetails;
                }
            }
            
            // Sort by confidence score
            usort($detailedResults, function($a, $b) {
                return $b['confidence_score'] - $a['confidence_score'];
            });
            
            return $this->response->setJSON([
                'success' => true,
                'results' => $detailedResults,
                'store_id' => $storeId
            ]);
            
        } catch (\Exception $e) {
            log_message('error', 'Hours search error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Search failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Import hours for individual business
     */
    public function importHours()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        try {
            $storeId = $this->request->getPost('store_id');
            $placeId = $this->request->getPost('place_id');
            $strategy = $this->request->getPost('strategy') ?: 'replace_all';
            
            if (!$storeId || !$placeId) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Missing store ID or place ID'
                ]);
            }
            
            // Get place details
            $placeDetails = $this->getPlaceDetails($placeId);
            if (!$placeDetails) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Could not retrieve place details'
                ]);
            }
            
            $processedPlace = $this->processPlaceHours($placeDetails);
            
            if (!$processedPlace['hours_available']) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'No hours data available for this location'
                ]);
            }
            
            // Start database transaction
            $this->db->transStart();
            
            // Import the hours with data replacement
            $result = $this->importBusinessHours($storeId, $processedPlace, $strategy);
            
            $this->db->transComplete();
            
            if ($this->db->transStatus() === false) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Database transaction failed'
                ]);
            }
            
            log_message('info', "Hours imported successfully for store $storeId using strategy: $strategy");
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Hours imported successfully',
                'action' => $result['action'],
                'hours_type' => $processedPlace['is_24x7'] ? '24/7' : 'regular'
            ]);
            
        } catch (\Exception $e) {
            $this->db->transRollback();
            log_message('error', 'Hours import error: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Import failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Bulk import hours
     */
    public function bulkImport()
    {
        if (!session()->get('admin_logged_in')) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authorized']);
        }

        try {
            log_message('info', 'Bulk import started - POST data: ' . print_r($_POST, true));
            
            $storeIds = [];
            if (isset($_POST['store_ids']) && is_array($_POST['store_ids'])) {
                $storeIds = $_POST['store_ids'];
            }
            
            $autoMatch = $this->request->getPost('auto_match') === '1';
            $replaceExisting = $this->request->getPost('replace_existing') === '1';
            $confidenceThreshold = intval($this->request->getPost('confidence_threshold') ?: 70);
            $mergeStrategy = $this->request->getPost('merge_strategy') ?: 'replace_all';
            
            if (empty($storeIds)) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'No businesses selected'
                ]);
            }

            $results = [];
            $successCount = 0;
            $failCount = 0;
            
            foreach ($storeIds as $index => $storeId) {
                try {
                    log_message('info', "Processing business " . ($index + 1) . " of " . count($storeIds) . " - Store ID: {$storeId}");
                    
                    $business = $this->db->query("
                        SELECT name, address, city, state 
                        FROM store 
                        WHERE store_id = ? AND status = 1
                    ", [$storeId])->getRowArray();
                    
                    if (!$business) {
                        $results[] = "Store ID $storeId: Business not found";
                        $failCount++;
                        continue;
                    }
                    
                    if ($autoMatch) {
                        try {
                            $placesResults = $this->searchGooglePlaces(
                                $business['name'], 
                                $business['address'], 
                                $business['city'], 
                                $business['state']
                            );
                            
                            if (empty($placesResults)) {
                                $results[] = "{$business['name']}: No Google Places results found";
                                $failCount++;
                                continue;
                            }
                            
                            $placeDetails = $this->getPlaceDetails($placesResults[0]['place_id']);
                            if (!$placeDetails) {
                                $results[] = "{$business['name']}: Could not get place details";
                                $failCount++;
                                continue;
                            }
                            
                            $processedPlace = $this->processPlaceHours($placeDetails);
                            
                            if ($processedPlace && $processedPlace['hours_available']) {
                                $confidence = $this->calculateMatchConfidence($placeDetails, $business['name'], $business['address']);
                                
                                if ($confidence < $confidenceThreshold) {
                                    $results[] = "{$business['name']}: Low confidence match ({$confidence}%), skipped";
                                    $failCount++;
                                    continue;
                                }
                                
                                $this->db->transStart();
                                
                                // Import with proper data replacement
                                $importResult = $this->importBusinessHours($storeId, $processedPlace, $mergeStrategy);
                                
                                $this->db->transComplete();
                                
                                if ($this->db->transStatus() !== false) {
                                    $hoursType = $processedPlace['is_24x7'] ? '24/7' : 'regular';
                                    $results[] = "{$business['name']}: Hours imported successfully ({$confidence}% confidence, $hoursType)";
                                    $successCount++;
                                } else {
                                    $results[] = "{$business['name']}: Database error during import";
                                    $failCount++;
                                }
                            } else {
                                $results[] = "{$business['name']}: No hours data available from Google Places";
                                $failCount++;
                            }
                        } catch (\Exception $e) {
                            $results[] = "{$business['name']}: Error - " . $e->getMessage();
                            $failCount++;
                        }
                    }
                    
                } catch (\Exception $e) {
                    $results[] = "Store ID {$storeId}: Error - " . $e->getMessage();
                    $failCount++;
                }
            }
            
            return $this->response->setJSON([
                'success' => true,
                'results' => $results,
                'summary' => [
                    'total_processed' => count($storeIds),
                    'successful' => $successCount,
                    'failed' => $failCount
                ]
            ]);
            
        } catch (\Exception $e) {
            log_message('error', 'Bulk import failed: ' . $e->getMessage());
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Bulk import failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * CORRECTED: Google Places search with valid fields only
     */
    private function searchGooglePlaces($name, $address, $city, $state)
    {
        $apiKey = env('GOOGLE_PLACES_API_KEY');
        if (!$apiKey) {
            throw new \Exception('Google Places API key not configured');
        }
        
        // Try multiple search strategies
        $searchQueries = [
            "$name $city $state",
            "$name $address $city $state",
            "$name $city",
            $name
        ];
        
        foreach ($searchQueries as $index => $query) {
            log_message('info', "Search strategy " . ($index + 1) . ": '$query'");
            
            $url = 'https://maps.googleapis.com/maps/api/place/textsearch/json?' . http_build_query([
                'query' => $query,
                'key' => $apiKey,
                'type' => 'establishment',
                'region' => 'us'
            ]);
            
            $response = $this->makeGoogleApiCall($url);
            $data = json_decode($response, true);
            
            if ($data && $data['status'] === 'OK' && !empty($data['results'])) {
                log_message('info', "Strategy " . ($index + 1) . " found " . count($data['results']) . " results");
                return $data['results'];
            } elseif ($data && $data['status'] === 'ZERO_RESULTS') {
                log_message('info', "Strategy " . ($index + 1) . " returned zero results");
                continue;
            } else {
                log_message('warning', "Strategy " . ($index + 1) . " failed: " . ($data['status'] ?? 'Unknown error'));
            }
        }
        
        log_message('warning', "All search strategies failed for: $name");
        return [];
    }

    /**
     * CORRECTED: Get place details with ONLY valid fields
     */
    private function getPlaceDetails($placeId)
    {
        $apiKey = env('GOOGLE_PLACES_API_KEY');
        
        // ✅ MINIMAL WORKING FIELDS - Tested and verified
        $fields = [
            'name',
            'formatted_address',
            'opening_hours'
        ];
        
        $url = 'https://maps.googleapis.com/maps/api/place/details/json?' . http_build_query([
            'place_id' => $placeId,
            'fields' => implode(',', $fields),
            'key' => $apiKey
        ]);
        
        log_message('info', "Place Details API URL: $url");
        
        $response = $this->makeGoogleApiCall($url);
        $data = json_decode($response, true);
        
        if (!$data || $data['status'] !== 'OK') {
            log_message('error', 'Place Details API error: ' . ($data['status'] ?? 'Invalid response') . ' - Response: ' . $response);
            return null;
        }
        
        $result = $data['result'] ?? null;
        
        if ($result) {
            $hasHours = isset($result['opening_hours']);
            log_message('info', "Place Details SUCCESS for $placeId: Has opening_hours = " . ($hasHours ? 'YES' : 'NO'));
            
            if ($hasHours) {
                log_message('info', "Hours data found: " . json_encode($result['opening_hours']));
            }
        }
        
        return $result;
    }

    /**
     * Process place hours with 24/7 detection
     */
    private function processPlaceHours($placeDetails)
    {
        $result = $placeDetails;
        
        // Check if hours data exists
        if (!isset($placeDetails['opening_hours'])) {
            $result['opening_hours'] = null;
            $result['hours_available'] = false;
            $result['is_24x7'] = false;
            return $result;
        }
        
        $hoursData = $placeDetails['opening_hours'];
        
        // Check for 24/7 operation
        if (isset($hoursData['weekday_text'])) {
            $is24x7 = $this->detect24x7Hours($hoursData['weekday_text']);
            if ($is24x7) {
                $result['is_24x7'] = true;
                $result['opening_hours'] = $this->create24x7Schedule();
                $result['hours_available'] = true;
                log_message('info', "Detected 24/7 operation for: " . $placeDetails['name']);
                return $result;
            }
        }
        
        // Process regular hours
        $result['opening_hours'] = $hoursData;
        $result['hours_available'] = isset($hoursData['periods']) && !empty($hoursData['periods']);
        $result['is_24x7'] = false;
        
        return $result;
    }

    /**
     * Detect 24/7 operation
     */
    private function detect24x7Hours($weekdayText)
    {
        $text = implode(' ', $weekdayText);
        $text = strtolower($text);
        
        $patterns = [
            'open 24 hours',
            '24 hours',
            'open daily',
            'always open',
            '24/7',
            'open all day'
        ];
        
        foreach ($patterns as $pattern) {
            if (strpos($text, $pattern) !== false) {
                return true;
            }
        }
        
        // Check if all days have 24-hour patterns
        $openAllDays = true;
        foreach ($weekdayText as $dayText) {
            if (strpos(strtolower($dayText), 'closed') !== false) {
                $openAllDays = false;
                break;
            }
        }
        
        return $openAllDays && count($weekdayText) >= 7;
    }

    /**
     * Create 24/7 schedule structure
     */
    private function create24x7Schedule()
    {
        $periods = [];
        
        for ($day = 0; $day <= 6; $day++) {
            $periods[] = [
                'close' => ['day' => $day, 'time' => '0000'],
                'open' => ['day' => $day, 'time' => '0000']
            ];
        }
        
        return [
            'open_now' => true,
            'periods' => $periods,
            'weekday_text' => [
                'Monday: Open 24 hours',
                'Tuesday: Open 24 hours', 
                'Wednesday: Open 24 hours',
                'Thursday: Open 24 hours',
                'Friday: Open 24 hours',
                'Saturday: Open 24 hours',
                'Sunday: Open 24 hours'
            ],
            'special_24x7' => true
        ];
    }

    /**
     * CORE: Import business hours with guaranteed data replacement
     */
    private function importBusinessHours($storeId, $processedPlace, $strategy = 'replace_all')
    {
        log_message('info', "Importing hours for store $storeId with strategy: $strategy");
        
        // ALWAYS clear existing data first - this ensures fresh import
        log_message('info', "Clearing existing hours data for store $storeId");
        
        // Step 1: Delete all existing schedules and hour types
        $this->db->query("DELETE FROM store_schedules WHERE store_id = ?", [$storeId]);
        $this->db->query("DELETE FROM store_hour_types WHERE store_id = ?", [$storeId]);
        
        log_message('info', "Existing data cleared for store $storeId");
        
        // Step 2: Import fresh data
        if ($processedPlace['is_24x7']) {
            return $this->import24x7Hours($storeId, $processedPlace);
        } else {
            return $this->importRegularHours($storeId, $processedPlace);
        }
    }

    /**
     * Import 24/7 hours
     */
    private function import24x7Hours($storeId, $placeDetails)
    {
        log_message('info', "Importing 24/7 hours for store $storeId");
        
        // Create Regular Hours type
        $hourTypeId = $this->createHourType($storeId, 'Regular Hours', 1, 1, true);
        
        // Create 24/7 schedule (00:00 to 23:59 every day)
        for ($day = 1; $day <= 7; $day++) {
            $this->db->query("
                INSERT INTO store_schedules 
                (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, is_closed, created_at, updated_at)
                VALUES (?, ?, ?, '00:00:00', '23:59:00', 0, NOW(), NOW())
            ", [$storeId, $hourTypeId, $day]);
        }
        
        log_message('info', "24/7 hours imported successfully for store $storeId");
        
        return [
            'action' => '24x7_import',
            'message' => '24/7 hours imported successfully'
        ];
    }

    /**
     * Import regular hours
     */
    private function importRegularHours($storeId, $placeDetails)
    {
        log_message('info', "Importing regular hours for store $storeId");
        
        // Create Regular Hours type
        $hourTypeId = $this->createHourType($storeId, 'Regular Hours', 1, 1, true);
        
        // Import the Google hours schedule
        $this->importGoogleHoursToSchedule($storeId, $hourTypeId, $placeDetails['opening_hours']);
        
        log_message('info', "Regular hours imported successfully for store $storeId");
        
        return [
            'action' => 'regular_import',
            'message' => 'Regular hours imported successfully'
        ];
    }

    /**
     * Create new hour type
     */
    private function createHourType($storeId, $customName, $globalTypeId, $displayOrder, $affectsMainStatus)
    {
        $this->db->query("
            INSERT INTO store_hour_types 
            (store_id, global_type_id, custom_name, display_order, affects_main_status, is_active, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, 1, NOW(), NOW())
        ", [$storeId, $globalTypeId, $customName, $displayOrder, $affectsMainStatus ? 1 : 0]);
        
        return $this->db->insertID();
    }

    /**
     * Import Google hours to schedule table
     */
    private function importGoogleHoursToSchedule($storeId, $hourTypeId, $googleHours)
    {
        if (!isset($googleHours['periods']) || empty($googleHours['periods'])) {
            // No periods data - mark all days as closed
            for ($day = 1; $day <= 7; $day++) {
                $this->db->query("
                    INSERT INTO store_schedules 
                    (store_id, hour_type_id, day_of_week, is_closed, created_at, updated_at)
                    VALUES (?, ?, ?, 1, NOW(), NOW())
                ", [$storeId, $hourTypeId, $day]);
            }
            return;
        }
        
        // Initialize week schedule
        $weekSchedule = [];
        for ($day = 1; $day <= 7; $day++) {
            $weekSchedule[$day] = [
                'is_closed' => 1,
                'slot_1_open' => null,
                'slot_1_close' => null,
                'slot_2_open' => null,
                'slot_2_close' => null
            ];
        }
        
        // Process periods
        foreach ($googleHours['periods'] as $period) {
            if (!isset($period['open'])) continue;
            
            $openDay = $this->convertGoogleDayToOurs($period['open']['day']);
            $openTime = $this->convertGoogleTimeToOurs($period['open']['time']);
            
            if (!$openTime) continue;
            
            // Handle close time
            $closeTime = '23:59:00';
            if (isset($period['close'])) {
                $closeTime = $this->convertGoogleTimeToOurs($period['close']['time']);
                if (!$closeTime) $closeTime = '23:59:00';
            }
            
            $weekSchedule[$openDay]['is_closed'] = 0;
            
            // Assign to first available slot
            if (empty($weekSchedule[$openDay]['slot_1_open'])) {
                $weekSchedule[$openDay]['slot_1_open'] = $openTime;
                $weekSchedule[$openDay]['slot_1_close'] = $closeTime;
            } else {
                $weekSchedule[$openDay]['slot_2_open'] = $openTime;
                $weekSchedule[$openDay]['slot_2_close'] = $closeTime;
            }
        }
        
        // Insert the schedule
        foreach ($weekSchedule as $day => $schedule) {
            $this->db->query("
                INSERT INTO store_schedules 
                (store_id, hour_type_id, day_of_week, slot_1_open, slot_1_close, slot_2_open, slot_2_close, is_closed, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
            ", [
                $storeId, 
                $hourTypeId, 
                $day, 
                $schedule['slot_1_open'],
                $schedule['slot_1_close'],
                $schedule['slot_2_open'],
                $schedule['slot_2_close'],
                $schedule['is_closed']
            ]);
        }
        
        log_message('info', "Schedule imported for store $storeId, hour type $hourTypeId");
    }

    /**
     * Make Google API call
     */
    private function makeGoogleApiCall($url)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Business Directory Hours Import Tool');
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($response === false || $httpCode !== 200) {
            throw new \Exception("Google API Error: $curlError (HTTP $httpCode)");
        }
        
        return $response;
    }

    // Helper methods
    
    private function detectTestDataPattern($storeId)
    {
        try {
            // First check if this store has recently imported Google hours (within last 7 days)
            $recentImport = $this->db->query("
                SELECT COUNT(*) as recent_count
                FROM store_hour_types sht
                WHERE sht.store_id = ? AND sht.is_active = 1 
                AND sht.custom_name = 'Regular Hours'
                AND sht.updated_at > DATE_SUB(NOW(), INTERVAL 7 DAY)
            ", [$storeId])->getRowArray();
            
            // If recently imported from Google, it's NOT test data
            if ($recentImport['recent_count'] > 0) {
                return false;
            }
            
            // Check for common test patterns
            $testPatterns = $this->db->query("
                SELECT COUNT(*) as pattern_count
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.is_active = 1
                AND (
                    (ss.slot_1_open = '09:00:00' AND ss.slot_1_close = '12:00:00' 
                     AND ss.slot_2_open = '13:00:00' AND ss.slot_2_close = '17:00:00')
                    OR (ss.slot_1_open = '09:00:00' AND ss.slot_1_close = '17:00:00' AND ss.day_of_week BETWEEN 1 AND 5)
                    OR (sht.custom_name = 'Phone Support' AND ss.slot_1_open = '08:00:00' AND ss.slot_1_close = '18:00:00')
                    OR (ss.slot_1_open = '08:00:00' AND ss.slot_1_close = '17:00:00' AND ss.day_of_week BETWEEN 1 AND 5)
                )
            ", [$storeId])->getRowArray();

            return $testPatterns['pattern_count'] > 0;
        } catch (\Exception $e) {
            log_message('error', 'Error detecting test data pattern for store ' . $storeId . ': ' . $e->getMessage());
            return false;
        }
    }

    private function calculateMatchConfidence($placeDetails, $businessName, $businessAddress)
    {
        $confidence = 50;
        
        similar_text(strtolower($placeDetails['name']), strtolower($businessName), $nameSimilarity);
        $confidence += ($nameSimilarity / 100) * 30;
        
        if (!empty($businessAddress) && isset($placeDetails['formatted_address'])) {
            similar_text(strtolower($placeDetails['formatted_address']), strtolower($businessAddress), $addressSimilarity);
            $confidence += ($addressSimilarity / 100) * 15;
        }
        
        if (isset($placeDetails['rating']) && $placeDetails['rating'] > 4.0) $confidence += 3;
        if (isset($placeDetails['opening_hours'])) $confidence += 5;
        if (isset($placeDetails['phone'])) $confidence += 2;
        
        return min(round($confidence), 95);
    }

    private function convertGoogleDayToOurs($googleDay)
    {
        // Google: 0=Sunday, 1=Monday, etc.
        // Ours: 1=Monday, 7=Sunday
        $conversion = [0 => 7, 1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6];
        return $conversion[$googleDay] ?? 1;
    }

    private function convertGoogleTimeToOurs($googleTime)
    {
        if (strlen($googleTime) !== 4) {
            return null;
        }
        
        $hours = substr($googleTime, 0, 2);
        $minutes = substr($googleTime, 2, 2);
        
        return $hours . ':' . $minutes . ':00';
    }
}