<?php

/**
 * Save this file as: app/Config/RuntimeTracker.php
 */

namespace App\Config;

class RuntimeFileTracker 
{
    private static $instance = null;
    private $logFile;
    private $startTime;
    private $executedFiles = [];
    
    private function __construct() 
    {
        $this->logFile = WRITEPATH . 'logs' . DIRECTORY_SEPARATOR . 'file_usage.json';
        $this->startTime = microtime(true);
        
        // Start tracking included files
        $this->startTracking();
    }
    
    public static function getInstance() 
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function startTracking() 
    {
        // Register shutdown function to capture all included files
        register_shutdown_function([$this, 'logExecutedFiles']);
    }
    
    public function logExecutedFiles() 
    {
        $includedFiles = get_included_files();
        $appPath = realpath(APPPATH);
        $executionTime = microtime(true) - $this->startTime;
        
        $relevantFiles = [];
        foreach ($includedFiles as $file) {
            // Only track files in the app directory
            if (strpos(realpath($file), $appPath) === 0) {
                $relativePath = str_replace($appPath . DIRECTORY_SEPARATOR, '', realpath($file));
                $relevantFiles[] = $relativePath;
            }
        }
        
        $logEntry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'url' => $_SERVER['REQUEST_URI'] ?? 'CLI',
            'method' => $_SERVER['REQUEST_METHOD'] ?? 'CLI',
            'execution_time' => round($executionTime * 1000, 2) . 'ms',
            'files' => $relevantFiles,
            'file_count' => count($relevantFiles)
        ];
        
        // Load existing log data
        $existingData = [];
        if (file_exists($this->logFile)) {
            $content = file_get_contents($this->logFile);
            $existingData = json_decode($content, true) ?: [];
        }
        
        // Add new entry
        $existingData[] = $logEntry;
        
        // Keep only last 100 requests to prevent huge log files
        if (count($existingData) > 100) {
            $existingData = array_slice($existingData, -100);
        }
        
        // Save updated data
        file_put_contents($this->logFile, json_encode($existingData, JSON_PRETTY_PRINT));
    }
    
    public static function getUsageReport() 
    {
        $instance = self::getInstance();
        $logFile = $instance->logFile;
        
        if (!file_exists($logFile)) {
            return [
                'total_requests' => 0,
                'unique_files' => [],
                'file_frequency' => [],
                'recent_requests' => []
            ];
        }
        
        $data = json_decode(file_get_contents($logFile), true) ?: [];
        
        // Analyze the data
        $allFiles = [];
        $fileFrequency = [];
        
        foreach ($data as $request) {
            foreach ($request['files'] as $file) {
                $allFiles[] = $file;
                $fileFrequency[$file] = ($fileFrequency[$file] ?? 0) + 1;
            }
        }
        
        // Sort by frequency
        arsort($fileFrequency);
        
        return [
            'total_requests' => count($data),
            'unique_files' => array_unique($allFiles),
            'file_frequency' => $fileFrequency,
            'recent_requests' => array_slice($data, -10) // Last 10 requests
        ];
    }
    
    public static function clearLog() 
    {
        $instance = self::getInstance();
        if (file_exists($instance->logFile)) {
            unlink($instance->logFile);
        }
    }
}

// Auto-start tracking if this file is included
if (class_exists('CodeIgniter\Config\Services')) {
    \App\Config\RuntimeFileTracker::getInstance();
}